/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "didadit.h"
#include "didadit.moc"

#include <kwmmapp.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include "toolbox.h"
#include "channel.h"
#include "global.h"
#include "main.h"
#include <errno.h>

#include "crc.h"


extern TopLevel *toplevel;



#define LASTBLOCK_INFO 1
#define LASTBLOCK_START 2
#define LASTBLOCK_REQ 3
#define LASTBLOCK_FIN 4
#define LASTBLOCK_ECHOREQUEST 5



//   DIDADIT::DIDADIT(char *_filename, bool senden)
//
// Konstruktor der Klasse DIDADIT.
//
// Parameter: _filename: Den Dateinamen mitsamt Pfad
//            senden:    true, wenn die Datei ausgesendet werden soll, sonst
//                       false
DIDADIT::DIDADIT(QWidget *_chan, char *_filename, bool senden) : QObject()
{
   struct stat stat_data;
   int i,len;
   char *tmp;

   chan = _chan;
   blocksize = 2048;
   fd_data = -1;
   fd_r = -1;
   rxlistroot = NULL;
   reqlistroot = NULL;
   win = NULL;
   rxbytes = 0;
   reqCount = 0;
   reqLength = 0;
   lastBlockSent = -1;

   if (senden)
   {
      longname = (char *) strdup(_filename);
      // Shortname (den eigentlichen Dateinamen) 'rausfinden
      tmp = (char *) strdup(_filename);
      if ((i = lPOS('/',_filename)) != -1)
      {
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }
      shortname = (char *) strdup(tmp);

      free(tmp);

      stat(longname, &stat_data);

      size = stat_data.st_size;
      modtime = stat_data.st_mtime;
      if (!calc_MD5_wholefile(longname, md5))
      {
      }
   }
   else
   {
      shortname = NULL;
      longname = NULL;
      size = -1;
      modtime = -1;
      offset = 0;
   }

   rxData = NULL;
   rxDataLen = 0;
   starttime = time(NULL);

   if (senden)
   {
      status = 1;
      firstblock = false;
      sendInfoBlock();
      waitForOkAbort = true;
      toplevel->chanListTable->setMode( chan, MODE_DIDADITTX );
      ((Channel *)chan)->updateStatusBar();
   }
   else
   {
      status = 100;
      firstblock = true;
      waitForOkAbort = false;
      toplevel->chanListTable->setMode( chan, MODE_DIDADITRX );
      ((Channel *)chan)->updateStatusBar();
   }

   timer = new QTimer( this );
   connect(timer, SIGNAL(timeout()), this, SLOT(sendBlock()));
}


DIDADIT::~DIDADIT()
{
   if (!allReceived())
      writeRFile();
   else
      deleteRFile();

   if (rxData != NULL) free(rxData);
   if (longname != NULL) free(longname);
   if (shortname != NULL) free(shortname);

   if (fd_data != -1)
   {
     ::close(fd_data);
     fd_data = -1;
   }

   if (win != NULL)
      delete win;

   delete timer;

   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar();
}


void EscapeIt (char *dest, int & dlen, char ch)
{
  if (ch == FEND)
  {
     dest[dlen] = FESC;
     dest[dlen+1] = TFEND;
     dlen += 2;
  }
  else if (ch == FESC)
       {
          dest[dlen] = FESC;
          dest[dlen+1] = TFESC;
          dlen += 2;
       }
       else
       {
          dest[dlen] = ch;
          dlen++;
       }
}



void DIDADIT::SendBlock (char *data, int datalen, unsigned short blockid)
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;



   if ((blockid == KBLOCK_ERR) ||
       (blockid == KBLOCK_DATA) ||
       (blockid == KBLOCK_FIN) ||
       (blockid == KBLOCK_REQ) ||
       (blockid == KBLOCK_FINACK))
   {
      // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
      memmove(data+16, data, datalen);
      datalen += 16;
      memcpy(data, md5, 16);
   }

   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = calc_crcthp_str(data, datalen, CRC);

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
   {
      EscapeIt(sSend, slen, data[i]);
/*      if (slen > 750)
      {
         SendBytes( sSend, slen);
         slen = 0;
      }*/
   }

   sSend[slen] = FEND;
   slen++;

   SendBytes( sSend, slen);
}


//   void DIDADIT::SendBytes(char *data, int len)
// Eine bestimmte Anzahl Bytes soll ausgesendet werden (nicht anzeigen).
void DIDADIT::SendBytes(char *data, int len)
{
   ((Channel *)chan)->sendString( len, data, false );
}


/*
 * void didadit::DecodeBlock(char *data, int len, t_KBlock *block)
 *
 * Den Block in data dekodieren und in die t_KBlock-Struktur
 * packen.
 */
void DIDADIT::DecodeBlock(char *data, int len, t_KBlock *block)
{
   int i;
   unsigned short crc=0;

   block->error = false;


   block->data = (char *) malloc(len);

   block->len = 0;
   i = 1;
   while (i < len)
   {
      if (data[i] == FESC)
      {
         i++;
         switch (data[i])
         {
            case TFEND: block->data[block->len] = FEND; break;
            case TFESC: block->data[block->len] = FESC; break;
            default: block->error = true;
         }
      }
      else block->data[block->len] = data[i];

      i++;
      block->len++;
   }

   // Block-ID
   memcpy(&block->id,block->data,sizeof(block->id));

   // CRC
   memcpy(&block->crc, block->data+block->len-2, sizeof(block->crc));

   // CRC der Daten und der Block-ID errechnen (ohne CRC)
   crc = calc_crcthp_str(block->data, block->len-2, crc);

   // Die Laenge der CRC und die der ID wieder abziehen
   block->len -= 4;
   memmove(block->data, block->data+2, block->len);


   if (crc != block->crc) block->error = true;

   if (block->error) return;

   // Wenn in diesem Block ein MD5-Hash drin ist, wird er aus dem
   // daten-Feld in das Hash-Feld gepackt
   if ((block->id == KBLOCK_ERR) ||
       (block->id == KBLOCK_DATA) ||
       (block->id == KBLOCK_FIN) ||
       (block->id == KBLOCK_REQ) ||
       (block->id == KBLOCK_FINACK))
   {
      // Ja, MD5-Hash
      memcpy(block->md5, block->data, 16);
      memmove(block->data, block->data+16, block->len-16);
      block->len -= 16;
   }
}


//   void DIDIADIT::sendMessage(char *str)
// Eine Nachricht (Statusmeldung) auf dem Bildschirm anzeigen und, wenn
// moeglich (Gegenstation keine automatische Station) auch aussenden.
void DIDADIT::sendMessage(char *str)
{
   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( str );
   else
      ((Channel *)chan)->outText( str, strlen(str), config->colors->txText );
}


// void DIDADIT::checkBlock(char *data, int len)
//
// Diese Funktion wird immer dann aufgerufen, wenn Daten angekommen
// sind.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist,
//            1, wenn der Block fehlerhaft ist
//            2, wenn der Block OK ist und verarbeitet werden soll
//            3, wenn der Block noch nicht vollstaendig empfangen wurde
int DIDADIT::checkBlock(char *data, int len)
{
   int i;

   // Erstmal warten, bis dieser Block komplett empfangen wurde
   if (rxData == NULL)
   {
      // Speicher reservieren
      rxData = (char *) malloc(MAXROHBLOCKSIZE*2);
      rxDataLen = 0;
   }

   // Die jetzt gerade empfangenen Daten an rxData anhaengen
   memcpy(rxData+rxDataLen,data,len);
   rxDataLen += len;

   // Alles vor dem ersten Auftreten von FEND loeschen
   if ((i = nPOS(FEND, rxData, rxDataLen)) == -1)
   {
      // FEND ist garnicht vorhanden. rxData loeschen
      free(rxData);
      rxData = NULL;
      rxDataLen = 0;
      return 3;
   }
   else
   {
      memmove(rxData,rxData+i,rxDataLen-i);
      rxDataLen -= i;
      rxData[rxDataLen] = '\0';
   }

   // Gucken, ob FEND nochmal vorhanden ist. Wenn ja, ist der ganze
   // Block empfangen worden. In diesem Fall wird der Block dekodiert
   // und in rxData geloescht.

   if ((i = nPOS(FEND, rxData+1, rxDataLen-1)) == -1)
   {
      return 3;
   }

   // Das uebersprungenen ersten FEND dazu
   i++;

   // Den Block dekodieren
   DecodeBlock(rxData,i,&block);

   memcpy(rxData,rxData+i+1,rxDataLen-i-1);
   rxDataLen -= (i+1);

   // Wenn der Block ungueltig ist, wird abgebrochen.
   if (block.error)
   {
printf("block.error\n");
      return 1;
   }

   return 2;
}


//   void DIDADIT::sendInfoBlock()
//
// Sendet ein Info-Frame mit Informationen zu dieser Datei aus
void DIDADIT::sendInfoBlock()
{
   char tmp[1000];
   char hexmd5[33];

   getHexMD5(md5, hexmd5);

   sprintf(tmp,"FILENAME=%s\rSIZE=%li\rMD5=%s\rTIME=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
               shortname, size, hexmd5, modtime, blocksize, DIDADIT_VERSION);

   SendBlock (tmp, strlen(tmp), KBLOCK_INFO );


/*   lastBlockSent = LASTBLOCK_INFO;
   timer->start(*/
}


//   void DIDADIT::readInfoBlock()
//
// "Packt" den Info-Block "aus" und speichert alle Infos ab.
void DIDADIT::readInfoBlock()
{
   char tmp[500], tmp2[500], *data;
   int i,len;
   char id[200];


   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }

      if (!strcmp(id, "FILENAME"))
      {
         if (longname != NULL) free(longname);
         longname = (char *) strdup(tmp);
         if ((i = POS('/', longname)) == -1)
         {
            if (shortname == NULL)
               shortname = (char *) strdup(longname);
            else
               strcpy(shortname, longname);
         }
         else
         {
            len = strlen(longname)-i-1;
            memcpy(tmp2, longname+i+1, len);
            tmp2[len] = '\0';
            shortname = (char *) strdup(tmp2);
         }
      }
      if (!strcmp(id, "SIZE"))
      {
         i = atoi(tmp);
         if (i != 0) size = i;
      }
      if (!strcmp(id, "MD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, md5);
      }
      if (!strcmp(id, "TIME"))
      {
         i = atoi(tmp);
         if (i != 0) modtime = i;
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
   }

   ((Channel *)chan)->sendString( 5, "#OK#\r", true );

   // Gucken, ob ein .r - File existiert. Wenn ja wird da der zu benutzende
   // Offset und die Liste der schon empfangenen Bytes eingelesen, wenn der
   // MD5-Hash stimmt.
   offset = lookForRFile();

   free(data);
}


//   void DIDADIT::sendStartBlock()
//
// Sendet den zu dieser Datei gehoerenden Startblock aus.
void DIDADIT::sendStartBlock()
{
   char tmp[1000];


   sprintf(tmp,"OFFSET=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
               offset, blocksize, DIDADIT_VERSION);

   SendBlock (tmp, strlen(tmp), KBLOCK_START );

   // DIDADIT-Empfangs-Fenster erstellen
   if (win == NULL)
   {
      // Entsprechendes Fenster oeffnen
      win = new TransferWin( chan, TRANSART_DIDADITRX, shortname, size );
      win->show();
      win->setReceivedBytes( 0 );
      if (toplevel->currentChannel != NULL)
         if (toplevel->currentChannel->isActiveWindow())
         {
            KWM::activate(toplevel->currentChannel->winId());
            toplevel->currentChannel->setFocus();
         }
   }
}


//   void DIDADIT::readStartBlock()
//
// "Packt" den Start-Blocks "aus" und speichert alle Infos ab.
void DIDADIT::readStartBlock()
{
   char tmp[200],*data;
   int i,len,partoffset=-1;
   char id[200];
   unsigned char partmd5[16], calcpartmd5[16];

   partmd5[0] = '\0';

   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }

      if (!strcmp(id, "OFFSET"))
      {
         offset = atoi(tmp);
      }
      if (!strcmp(id, "PARTMD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, partmd5);
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
   }

   // Gucken, ob partmd5 korrekt ist, wenn er uebergeben wurde
   if (partmd5[0] != '\0' && offset != 0)
   {
      if (!calc_MD5_partfile( longname, calcpartmd5, offset ))
      {
      }

      if (!memcmp(partmd5, calcpartmd5, 16)) partoffset = 0;
   }



   if (partoffset != -1) offset = partoffset;

   free(data);
}


//   int DIDADIT::proceed(char *data, int len)
//
// Diese Funktion wird vom Terminalprogramm mit allen Didadit-Daten
// aufgerufen und wickelt eigentlich alles ab.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist;
//            1, wenn dies der erste Block ist, der fehlerhaft ist.
//            2, wenn alles OK ist und nix gemacht werden muss
//
// Ist der allererste Block fehlerhaft (im Empfangsfall), scheint die
// Uebertragung nicht mit Didadit zu erfolgen -> fallback auf AutoBIN.
int DIDADIT::proceed(char *data, int len)
{
   char tmp[100];

   if (waitForOkAbort)
   {
      if (!strncmp(data, "#ABORT#", 7))
      {
         strcpy(tmp, "<LinKT>: DIDADIT-Transmission aborted by receiver.\r");
         ((Channel *)chan)->outText( tmp, strlen(tmp), config->colors->txText );
         return 0;
      }
      if (strncmp(data, "#OK#", 4)) return 2;
      waitForOkAbort = false;
      ((Channel *)chan)->flags &= ~CH_LINEMODE;
   }

   switch (checkBlock(data,len))
   {
      case 0: return 0;
      case 1: if (firstblock)
                 return 1;
              else
                 return 2;
      case 3: return 2;
   }
   firstblock = false;

   ///////////////////////////////////////////////////////////////////
   // Debugging-Output
   switch (block.id)
   {
      case KBLOCK_INFO: printf("KBLOCK_INFO\n"); break;
      case KBLOCK_START: printf("KBLOCK_START\n"); break;
      case KBLOCK_ERR: printf("KBLOCK_ERR\n"); break;
      case KBLOCK_DATA: printf("KBLOCK_DATA\n"); break;
      case KBLOCK_FIN: printf("KBLOCK_FIN\n"); break;
      case KBLOCK_REQ: printf("KBLOCK_REQ\n"); break;
      case KBLOCK_FINACK: printf("KBLOCK_FINACK\n"); break;
      case KBLOCK_TYPEREJ: printf("KBLOCK_TYPEREJ\n"); break;
      case KBLOCK_ECHOREQ: printf("KBLOCK_ECHOREQ\n"); break;
      case KBLOCK_ECHOREP: printf("KBLOCK_ECHOREP\n"); break;
      case KBLOCK_ABORT: printf("KBLOCK_ABORT\n"); break;
   }
   ///////////////////////////////////////////////////////////////////


   switch (status)
   {
      case 1: // Erwarte START-Block
              readStartBlock();
              if (offset != (unsigned long)size)
                 ((Channel *)chan)->sendDidadit();
              SendBlock( tmp, 0, KBLOCK_FIN );
              status = 2;
              break;
      case 2: switch (block.id)
              {
                 case KBLOCK_FINACK:
                      readFinAck();
                      return 0;
                 case KBLOCK_REQ:
                      readReqBlock();
                      break;
              }
      // RX //
      case 100: // Erwarte INFO-Block - alles andere wird ignoriert
                if (block.id == KBLOCK_INFO)
                {
                   readInfoBlock();
                   sendStartBlock();
                   createDataFiles();
                   status = 101;
                }
                break;
      case 101: // 102 = 101, KBLOCK_FIN wurde schon empfangen
      case 102: switch (block.id)
                {
                   case KBLOCK_DATA:
                        if (readDataBlock())
                           return 0;
                        break;
                   case KBLOCK_FIN:
                        if (readFinBlock())
                           return 0;
                        status = 102;
                        break;
                }
                break;
   }

   return 2;
}


//   void DIDADIT::readDataBlock()
//
// Ein Datenblock wird empfangen. Abspeichern und merken.
//
// Rueckgabe: true, wenn die komplette Uebertragung fertig ist,
//            false wenn nicht.
bool DIDADIT::readDataBlock()
{
   unsigned long offset;
   unsigned short blocklen;
   char tmp[50];

   // Die ersten 4 Bytes ist der Offset, danach kommt die Blocklaenge
   // (2 Bytes)
   memcpy(&offset, block.data, 4);
   memcpy(&blocklen, block.data+4, 2);
   memmove(block.data, block.data+6, block.len-6);
   block.len -= 6;

   write(fd_data, block.data, block.len);

   addRXListe( offset, offset+blocklen-1 );

   rxbytes += blocklen;
   win->setReceivedBytes( rxbytes );

   if (status == 102)
      if (allReceived())
      {
         SendBlock( tmp, 0, KBLOCK_FINACK );

         // Ergebnis der Uebertragung ausgeben
         showEndText( true );

         // .r - File loeschen
         deleteRFile();

         if (win != NULL)
         {
            delete win;
            win = NULL;
         }
         return true;
      }

   return false;
}


//   void DIDADIT::createDataFiles()
//
// Erzeugt ein File mit dem Namen <filename>.r im abin-Verzeichnis, in dem
// alle moeglichen Informationen abgespeichert werden sowie das eigentliche
// Datenfile (<filename>).
void DIDADIT::createDataFiles()
{
   char tmp[500];

   sprintf(tmp, "%s/%s", config->dirABin, shortname);

   if ((fd_data = open(tmp, O_WRONLY|O_CREAT)) == -1)
   {
      return;
   }
   // Zugriffsrechte einstellen
   fchmod(fd_data, S_IRUSR|S_IWUSR);
}


bool DIDADIT::getNextPacket( char *txdata, int & len )
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;
   unsigned short blockid=KBLOCK_DATA;
   long datalen;
   unsigned short dlen;
   bool ready=false;
   char data[MAXROHBLOCKSIZE];
   unsigned long loffset;
   s_reqliste *tmp;



   // Weitere Daten aus dem Sourcefile holen
   if (fd_data == -1)
   {
      if ((fd_data = open(longname, O_RDONLY)) == -1)
      {
printf("cannot open sourcefile: %s\n", strerror(errno));
         len = 0;
         return true;
      }
      lseek(fd_data, offset, SEEK_CUR);
   }

   if (reqlistroot != NULL)
   {
      loffset = lseek( fd_data, reqlistroot->offset, SEEK_SET );
      datalen = read( fd_data, data, reqlistroot->len );

      tmp = reqlistroot;
      reqlistroot = reqlistroot->next;
      free(tmp);

      if (reqlistroot == NULL) ready = true;
   }
   else
   {
      loffset = lseek(fd_data, 0, SEEK_CUR);
      if ((datalen = read(fd_data, data, MAXBLOCKSIZE)) != MAXBLOCKSIZE)
      {
         if (datalen == -1)
         {
            len = 0;
            printf("read()-error: %s\n", strerror(errno));
            return true;
         }
         // Das File ist komplett ausgesendet worden.
/*         ::close(fd_data);
         fd_data = -1;*/
         ready = true;
      }
   }

   // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
   dlen = (unsigned short) datalen;
   memmove(data+22, data, datalen);
   datalen += 22;
   // Offset und Blocklaenge dazu
   memcpy(data, md5, 16);
   memcpy(data+16, &loffset, 4);
   memcpy(data+20, &dlen, 2);


   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = calc_crcthp_str(data, datalen, CRC);

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
   {
      EscapeIt(sSend, slen, data[i]);
/*      if (slen > 750)
      {
         SendBytes( sSend, slen);
         slen = 0;
      }*/
   }

   sSend[slen] = FEND;
   slen++;


/////////////////////////// DEBUGGING //////////////////////////////////
/*if (loffset == 0)
   sSend[100] = 'A';*/
/////////////////////////// DEBUGGING //////////////////////////////////


   memcpy( txdata, sSend, slen );
   len = slen;

   return ready;
}


//   void DIDADIT::addRXListe( long start, long ende )
//
// Ein neuer Block wurde empfangen - Eintragen in die Liste im Speicher
void DIDADIT::addRXListe( unsigned long start, unsigned long ende )
{
   s_rxliste *tmp, *neu, *next, *last;
   bool ready;


   // Allererster Eintrag: Eintragen und 'raus
   if (rxlistroot == NULL)
   {
      rxlistroot = (s_rxliste *) malloc(sizeof(s_rxliste));
      rxlistroot->next = NULL;
      rxlistroot->start = start;
      rxlistroot->ende = ende;
      return;
   }

   // Gucken, ob das im Augenblick empfangene Frame das naechste in
   // der Folge ist
   tmp = rxlistroot;
   ready = false;
   while (tmp != NULL && !ready)
   {
      if (tmp->ende == start-1)
      {
         tmp->ende = ende;
         ready = true;
      }
      tmp = tmp->next;
   }

   if (!ready)
   {
      // Irgendwo war ein Loch. Gucken, an welcher Stelle dieser Eintrag
      // 'rein muss und einen neuen Eintrag erzeugen
      tmp = rxlistroot;
      ready = false;
      last = NULL;
      while (tmp != NULL && !ready)
      {
         if (tmp->ende < start)
         {
            neu = (s_rxliste *) malloc(sizeof(s_rxliste));
            neu->next = tmp->next;
            neu->start = start;
            neu->ende = ende;
            tmp->next = neu;
            ready = true;
         }
         else
            if (tmp->start > start)
            {
               if (tmp->ende-1 > ende)
               {
                  // Neuer Eintrag
                  neu = (s_rxliste *) malloc(sizeof(s_rxliste));
                  if (last == NULL)
                  {
                     rxlistroot = neu;
                     rxlistroot->next = tmp;
                  }
                  else
                  {
                     last->next = neu;
                     neu->next = tmp;
                  }
                  neu->start = start;
                  neu->ende = ende;
               }
               else
                  tmp->start = start;
               ready = true;
            }
            else tmp = tmp->next;
         last = tmp;
      }
   }


   // Gucken, ob irgendwelche Bereiche doppelt abgedeckt sind, Eintraege
   // also ueberlappen
   tmp = rxlistroot;
   while (tmp != NULL)
   {
      next = tmp->next;
      if (next != NULL)
         while (tmp->ende >= next->start-1)
         {
            // next reicht in den Bereich von tmp 'rein
            tmp->ende = next->ende;
            // next loeschen
            tmp->next = next->next;
            free(next);
            next = tmp->next;
            if (next == NULL) break;
         }
      tmp = tmp->next;
   }


   // Debugging: Alle Eintraege ausgeben
/*   tmp = rxlistroot;
   while (tmp != NULL)
   {
      printf("  start: %li, ende: %li\n", tmp->start, tmp->ende);
      tmp = tmp->next;
   }*/
}


//   void DIDADIT::showEndText( bool rx )
//
// Gibt Statusinformationen am Ende der Uebertragungen zurueck.
// Parameter rx ist true, wenn das File empfangen wurde, sonst false.
void DIDADIT::showEndText( bool rx )
{
   time_t timediff;
   char *timeptr;
   char tmp2[100];
   char text[100];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   if (rx)
      strcpy(text,"<LinKT>: DIDADIT-RX OK. (time: %s, %li baud)\xD" \
                  "         Requested Blocks: %li (%li bytes)\xD");
   else
      strcpy(text,"<LinKT>: DIDADIT-TX OK. (time: %s, %li baud)\xD" \
                  "         Retransmitted Blocks: %li (%li bytes)\xD");

   sprintf(tmp2, text,
                 timeptr,(size*8)/timediff, reqCount, reqLength);


   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp2 );
   else
      ((Channel *)chan)->outText( tmp2, strlen(tmp2), config->colors->txText );

   free(timeptr);
}


//   void DIDADIT::readFinBlock()
//
// Ein FIN-Block gucken, ob alles da ist. Wenn nicht werden die fehlenden
// Teile neu angefordert.
bool DIDADIT::readFinBlock()
{
   char tmp[100];
   s_rxliste *ltmp;


   if (rxlistroot == NULL)
   {
      // Alles neu, es wurde garnichts empfangen
      sendReqBlock( 0, size );
      return false;
   }


   if ((rxlistroot->start == 0) && (rxlistroot->ende == (unsigned long)size-1))
   {
      // Alles wurde empfangen. FINACK-Block senden; Feierabend
      SendBlock( tmp, 0, KBLOCK_FINACK );

      // Ergebnis der Uebertragung ausgeben
      showEndText( true );

      if (win != NULL)
      {
         delete win;
         win = NULL;
      }
      return true;
   }


   // Teile wurden nicht empfangen
   ltmp = rxlistroot;

   if (rxlistroot != NULL)
      if (rxlistroot->start != 0)
         sendReqBlock( 0, ltmp->start-1 );

   while (ltmp != NULL)
   {
      if (ltmp->next != NULL)
         sendReqBlock( ltmp->ende+1, ltmp->next->start-1 );
      else
         if (ltmp->ende+1 < (unsigned long)size)
            sendReqBlock( ltmp->ende+1, size-ltmp->ende );

      ltmp = ltmp->next;
   }

   return false;
}


void DIDADIT::readFinAck()
{
   // Ergebnis der Uebertragung ausgeben
   showEndText( false );

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
}


//   void DIDADIT::sendReqBlock( unsigned long offset, unsigned char blocklen )
//
// Fordert einen Bereich des Files neu an
void DIDADIT::sendReqBlock( unsigned long start, unsigned long stop )
{
   char data[50];
   unsigned long len=stop-start+1;

   memcpy(data, &start, 4);
   memcpy(data+4, &len, 4);

   reqCount++;
   reqLength += len;

   SendBlock( data, 8, KBLOCK_REQ );
}


//   void DIDADOT::readReqBlock()
//
// Ein REQ-Block wird empfangen. Daten in die entsprechende verkettete
// Liste eintragen und das Aussenden anfordern, wenn gerade keine REQ-
// Bloecke empfangen werden.
void DIDADIT::readReqBlock()
{
   unsigned long offset, len;
   s_reqliste *tmp;


   memcpy( &offset, block.data, 4 );
   memcpy( &len, block.data+4, 4 );

   reqCount++;
   reqLength += len;

   if (reqlistroot == NULL)
   {
      // Erster Eintrag
      reqlistroot = (s_reqliste *)malloc(sizeof(s_reqliste));
      tmp = reqlistroot;
   }
   else
   {
      tmp = reqlistroot;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_reqliste *)malloc(sizeof(s_reqliste));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->offset = offset;
   tmp->len = len;

   ((Channel *)chan)->sendDidadit();
}


bool DIDADIT::allReceived()
{
   if (rxlistroot == NULL)
      return false;
   return (rxlistroot->start == 0 && rxlistroot->ende == (unsigned long)size-1);
}


//   void DIDADIT::writeRFile()
//
// Die Infos zu diesem Transfer werden unter <filename>.r abgespeichert
// (Fuer den Resume-Mode)
void DIDADIT::writeRFile()
{
   char tmp[500];
   s_rxliste *rxlist;
   unsigned char hash[16];


   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if ((fd_r = open(tmp, O_RDWR|O_CREAT)) == -1)
      return;

   // Zugriffsrechte einstellen
   fchmod(fd_r,S_IRUSR|S_IWUSR);

   sprintf(tmp, "%s/%s", config->dirABin, shortname);
   if (!calc_MD5_wholefile(tmp, hash))
   {
   }

   // Die ersten 16 Bytes ist der MD5-Hash des fertigen Files
   write(fd_r, md5, 16);
   // Danach kommt der Hash des schon empfangenen Files
   write(fd_r, hash, 16);

   // Dann kommt die Liste der empfangenen Bytes
   rxlist = rxlistroot;
   while (rxlist != NULL)
   {
      sprintf(tmp,"%li %li\xA", rxlist->start, rxlist->ende);
      write(fd_r, tmp, strlen(tmp));

      rxlist = rxlist->next;
   }

   ::close(fd_r);
}


//   void DIDADIT::deleteRFile()
//
// Wenn das R-File (<filename>.r) existiert, wird es geloescht
void DIDADIT::deleteRFile()
{
   char tmp[500];

   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if (file_exist(tmp))
      unlink(tmp);
}


//   unsigned long DIDADIT::lookForRFile()
//
// Guckt, ob ein .r-File fuer diesen Dateinamen vorhanden ist. Wenn
// ja wird es eingelesen und der zu verwendende Offset zurueckgegeben.
unsigned long DIDADIT::lookForRFile()
{
   int fd;
   char tmp[500];
   int i;
   unsigned char hash[16];
   FILE *f;
   unsigned long start, ende;
   s_rxliste *rxliste, *last=NULL;


   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if ((fd = open(tmp, O_RDONLY)) == -1)
      return 0;

   sprintf(tmp, "%s/%s", config->dirABin, shortname);
   if (!calc_MD5_wholefile(tmp, hash))
   {
   }

   // Der Hash des schon gespeicherten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, md5, 16) != 0)
      return 0;

   // Der Hash des kompletten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, hash, 16) != 0)
      return 0;

   if ((f = fdopen(fd, "r")) == NULL)
      return 0;

   while (fgets(tmp, 499, f) != NULL)
   {
      if ((i = POS('\xA', tmp)) != -1) tmp[i] = '\0';

      if ((i = POS(' ',tmp)) != -1)
      {
         if (sscanf(tmp, "%li %li", &start, &ende) == 2)
         {
            // Neuer Eintrag
            if (rxlistroot == NULL)
            {
               // Erster Eintrag
               rxlistroot = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxlistroot;
            }
            else
            {
               rxliste = rxlistroot;
               while (rxliste->next != NULL) rxliste = rxliste->next;

               rxliste->next = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxliste->next;
            }

            rxliste->next = NULL;
            rxliste->start = start;
            rxliste->ende = ende;
            last = rxliste;
         }
      }
   }

   if (last == NULL) return 0;

   return (last->ende+1);

   ::close(fd);
}


void DIDADIT::getHexMD5( unsigned char *md5, char *hexmd5 )
{
   sprintf(hexmd5, "%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X",
                   md5[0],md5[1],md5[2],md5[3],md5[4],md5[5],md5[6],md5[7],
                   md5[8],md5[9],md5[10],md5[11],md5[12],md5[13],md5[14],md5[15]);
}


void DIDADIT::md5HexToBin( char *hex, unsigned char *bin )
{
   int i;
   char tmp[3];

   tmp[3] = '\0';

   for (i=0; i<16; i++)
   {
      tmp[0] = hex[i*2];
      tmp[1] = hex[(i*2)+1];
      bin[i] = get_hex (tmp);
   }
}


//   void DIDADIT::abortIt()
//
// Sendet einen ABORT-Block an die Gegenseite
void DIDADIT::abortIt()
{
   char tmp[2];

   SendBlock( tmp, 0, KBLOCK_ABORT );
}


//   bool DIDADIT::didaditTx()
//
// Gibt true zurueck, wenn wir etwas DIDADIT-maessiges aussenden, ansonsten
// false (wenn etwas empfangen wird).
bool DIDADIT::didaditTx()
{
   if (status >= 100)
      return false;
   return true;
}


void DIDADIT::sendBlock()
{
}



