/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <qfiledialog.h>
#include <qbuttongroup.h>

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "dialog.h"
#include "version.h"
#include "main.h"
#include "ax25k.h"
#include "bin.h"
#include "toolbox.h"
#include "global.h"

#include <qtimer.h>
#include <qpixmap.h>
#include <qtooltip.h>

#include <kmsgbox.h>
#include <kiconloader.h>


#define BTNLENGTH 80
#define BTNHEIGHT 25


#include <kapp.h>

extern KApplication *mykapp;
extern TopLevel *toplevel;


////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


Connect::Connect(QWidget *parent) : QDialog(parent, klocale->translate("Connect"), TRUE)
{
  s_ports *tmp;
  connCall *savecall;

  resize(300, 130);
  setFixedSize(size());

  setCaption(klocale->translate("Connect"));


  QLabel *l;

  l = new QLabel(klocale->translate("Call"), this);

  l->setGeometry(10, 5, 100, 20);
  l->setAlignment(AlignLeft);

  call = new QComboBox( TRUE, this );
  call->setGeometry( 10, 20, 280, 20 );
  call->setFocus();

  savecall = config->connCallsList->last();
  while (savecall != NULL)
  {
     call->insertItem(savecall->call());
     savecall = config->connCallsList->prev();
  }
  call->insertItem("", 0);


  l = new QLabel("Port", this);
  l->setGeometry(10,45,100,20);
  l->setAlignment(AlignLeft);


  l = new QLabel("MyCall", this);
  l->setGeometry(200,45,100,20);
  l->setAlignment(AlignLeft);


  combo = new QComboBox( this );
  // Hier werden alle vorhandenen Portnamen in die Combobox eingetragen
  tmp = portlist;
  while (tmp != NULL)
  {
     if (tmp->ax25 != NULL)
        combo->insertItem(tmp->name);
     tmp = tmp->next;
  }
  combo->setGeometry(10,60,combo->width(),20);


  mycall = new QLineEdit( this );
  mycall->setText( config->send_mycall );
  mycall->setGeometry(200, 60, 90, 20);



  QPushButton *b_yes = new QPushButton(klocale->translate("&Ok"), this);
  b_yes->setGeometry((width()-(2*80))/3, height() - 30, 80, 25);
  connect(b_yes, SIGNAL(clicked()), this, SLOT(accept()));
  b_yes->setDefault(TRUE);


  QPushButton *b_no = new QPushButton(klocale->translate("&Abort"), this);
  b_no->setGeometry(((width()-(2*80))/3)*2+80, height() - 30, 80, 25);
  connect(b_no, SIGNAL(clicked()), this, SLOT(reject()));


  a = new QAccel( this );
  a->connectItem( a->insertItem(Key_Return), this, SLOT(accept()));

}


const char * Connect::getText()
{
   return call->currentText();
}


const char * Connect::getPort()
{
   return combo->currentText();
}


const char * Connect::getMycall()
{
   return mycall->text();
}


Connect::~Connect()
{
   delete combo;
   delete call;
   delete mycall;
   delete a;
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

AboutDlg::AboutDlg(QWidget *parent) : QDialog(parent, klocale->translate("About"), TRUE)
{
  QPixmap pm;
  resize(350, 220);
  setFixedSize(size());

  setCaption(klocale->translate("About LinKT"));


  KIconLoader *loader = mykapp->getIconLoader();

  pm = loader->loadIcon("linkt.xpm");

  QLabel *l;

  l = new QLabel("XX", this);
  l->setPixmap(pm);
  l->move(0,0);

  l = new QLabel("LinKT", this);
  l->setFont(QFont("Utopia", 26, QFont::Normal));
  l->setGeometry(30, 20,l->sizeHint().width(),l->sizeHint().height());

  l = new QLabel(klocale->translate("the Linux KDE PR-Terminal"), this);
  l->setFont(QFont("Helvetica", 14/*, QFont::Normal*/));
  l->setGeometry(30, 50,l->sizeHint().width(),l->sizeHint().height());

  QString s;
  s = "Version " LINKT_VERSION " (" DATE ")" \
  "\n\nCopyright (c) 1997-1999 Jochen Sarrazin, DG6VJ\n"\
  "DG6VJ@DB0EA.#NRW.DEU.EU\n" \
  "jochen@1409.org";


  l = new QLabel(s.data(), this);
  l->setGeometry(30, 80, 300, 200);
  l->setAlignment(AlignLeft|WordBreak|ExpandTabs);



  QPushButton *b_ok = new QPushButton(klocale->translate("&Ok"), this);

  b_ok->setGeometry(width()/2-40, height() - 40, 80, 32);
  b_ok->setDefault(true);
  connect(b_ok, SIGNAL(clicked()), this, SLOT(accept()));
  b_ok->setFocus();
}



//////////////////////////////////////////////////////////////////////////
///   Fenster zum Anzeigen eines Datentransfers (BIN, DIDADIT, 7+)     ///
//////////////////////////////////////////////////////////////////////////

TransferWin::TransferWin( void *rxparent, int art, char *filename, int size )
               : QDialog()
{
   QLabel *tmplabel;
   char tmp[100];


   setFixedSize(310, 200);

   parent = rxparent;

   transart = art;
   switch (art)
   {
      case TRANSART_SIEBEN:
           setCaption( klocale->translate("7plus-Autosave") );
           break;
      case TRANSART_ABINTX:
           setCaption( klocale->translate("AutoBIN-Transmission") );
           break;
      case TRANSART_ABINRX:
           setCaption( klocale->translate("AutoBIN-Autosave") );
           break;
      case TRANSART_DIDADITTX:
           setCaption( klocale->translate("DIDADIT-Transmission") );
           break;
      case TRANSART_DIDADITRX:
           setCaption( klocale->translate("DIDADIT-Autosave") );
           break;
      case TRANSART_YAPPRX:
           setCaption( klocale->translate("YAPP-Save") );
           break;
      case TRANSART_YAPPTX:
           setCaption( klocale->translate("YAPP-Transmission") );
           break;
   }


   // Variablen abspeichern
   filesize = size;
   bytes_got = 0;


   // Linke Seite
   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 10, 90, 20);
   tmplabel->setText( klocale->translate("Filename") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 30, 90, 20);
   tmplabel->setText( klocale->translate("Size") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 70, 90, 20);
   tmplabel->setText( klocale->translate("Time elapsed") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 90, 90, 20);
   tmplabel->setText( klocale->translate("Time left") );


   // Rechte Seite
   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 110, 10, 190, 20);
   tmplabel->setText( filename );

   lBytes = new QLabel( this );
   lBytes->setGeometry( 110, 30, 190, 20);
   sprintf(tmp, "0 of %i Bytes", size);
   lBytes->setText( tmp );

   lBaudEff = new QLabel( this );
   lBaudEff->setGeometry( 110, 50, 190, 20);
   lBaudEff->setText( "0 Baud eff." );

   lTimeCons = new QLabel( this );
   lTimeCons->setGeometry( 110, 70, 190, 20);
   lTimeCons->setText( "00:00:00" );

   lTimeLeft = new QLabel( this );
   lTimeLeft->setGeometry( 110, 90, 190, 20);
   lTimeLeft->setText( "??:??:??" );

   // Progress-Bar
   // Immer fest 1000 Steps, die Umrechnung erfolgt in dieser Klasse.
   // (mehr bringt nix, 1000 ist eh nicht darzustellen)
   progressbar = new QProgressBar( 500, this );
   progressbar->setGeometry( 10, 130, 290, 20 );

   QPushButton *btn = new QPushButton( klocale->translate("&Abort"), this );
   btn->setGeometry( 110, 160, 100, 30 );
   connect( btn, SIGNAL(clicked()), SLOT(closeTransferWin()) );

   // Alle 500ms nach der Uhr und der Baud-Anzeige gucken
   QTimer *timer = new QTimer(this);
   connect(timer, SIGNAL(timeout()), SLOT(updateClockAndBaud()));
   timer->start(1000, FALSE);

   starttime = time(NULL);
}


void TransferWin::updateClockAndBaud()
{
   time_t act_time=time(NULL);
   time_t timediff=act_time - starttime;
   time_t timeleft;
   char hours[5], mins[5], secs[5], tmp[100];

   // Baud-Rate
   if (timediff == 0) timediff = 1;
   sprintf(tmp, "%li Baud eff.", (bytes_got*8)/timediff);
   lBaudEff->setText( tmp );

   // Fortlaufende Uhr
   sprintf(secs, "%.2li", timediff%60);
   timediff /= 60;
   sprintf(mins, "%.2li", timediff%60);
   timediff /= 60;
   sprintf(hours, "%.2li", timediff);

   sprintf(tmp, "%s:%s:%s", hours, mins, secs);
   lTimeCons->setText( tmp );


   // Verbleibende Zeit
   timediff=act_time - starttime;
   if ((bytes_got/timediff) < 1)
      timeleft = 1;
   else
      timeleft = filesize/(bytes_got/timediff)-timediff;

   sprintf(secs, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(mins, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(hours, "%.2i", (int)timeleft);

   sprintf(tmp, "%s:%s:%s", hours, mins, secs);
   lTimeLeft->setText( tmp );
}


void TransferWin::closeTransferWin()
{
   // Dem Eltern-Widget bescheid geben, dass der Transfer abgebrochen
   // werden soll.
   switch (transart)
   {
      case TRANSART_SIEBEN:
           ((auto7plus *)parent)->abortRX();
           break;
      case TRANSART_ABINTX:
           ((Channel *)parent)->abortABinTX();
           break;
      case TRANSART_ABINRX:
           ((BinRX *)parent)->abortRX();
           break;
      case TRANSART_DIDADITTX:
      case TRANSART_DIDADITRX:
           ((Channel *)parent)->abortDIDADIT();
           break;
      case TRANSART_YAPPRX:
      case TRANSART_YAPPTX:
           ((Channel *)parent)->abortYAPP();
           break;
   }

//   delete this;
}


//  void setReceivedBytes( int bytes )
// Die Anzahl der empfangenen Bytes hat sich veraendert
void TransferWin::setReceivedBytes( int bytes )
{
   time_t timediff=time(NULL) - starttime;
   time_t timeleft;
   char hours[5], mins[5], secs[5], tmp[100];

   bytes_got = bytes;

   sprintf(tmp, "%i of %i Bytes", bytes, filesize);
   lBytes->setText( tmp );

   progressbar->setProgress( (int) (((float)bytes/(float)filesize) * 500) );

   // Baud-Rate
   if (timediff == 0) timediff = 1;
   sprintf(tmp, "%li Baud eff.", (bytes*8)/timediff);
   lBaudEff->setText( tmp );

   // Verbleibende Zeit
   if ((bytes_got/timediff) < 1)
      timeleft = 1;
   else
      timeleft = filesize/(bytes/timediff)-timediff;

   sprintf(secs, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(mins, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(hours, "%.2i", (int)timeleft);

   sprintf(tmp, "%s:%s:%s", hours, mins, secs);
   lTimeLeft->setText( tmp );
}


void TransferWin::setFilesizeBytes( int bytes )
{
   char tmp[100];


   filesize = bytes;

   sprintf(tmp, "%i of %i Bytes", bytes, filesize);
   lBytes->setText( tmp );
}


TransferWin::~TransferWin()
{
   delete lBytes;
   delete lBaudEff;
   delete lTimeCons;
   delete lTimeLeft;
   delete progressbar;
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


send7plus::send7plus() : QDialog(0, "", true)
{
   QLabel *tmplabel;
   QButtonGroup *tmpbtngrp;

   setFixedSize( 360, 220 );

   filecount = -1;

   setCaption( klocale->translate("7plus-Autosend") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 10, 100, 20);
   tmplabel->setText( klocale->translate("Board") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 30, 100, 20);
   tmplabel->setText( klocale->translate("Lifetime") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 50, 100, 20);
   tmplabel->setText( klocale->translate("Title of the mails") );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 10, 70, 100, 20);
   tmplabel->setText( klocale->translate("Filename") );

   // Boardname
   board = new QLineEdit( this );
   board->setGeometry( 120, 10, 100, 20);
   board->setText( "" );
   board->setMaxLength( 8 );

   tmplabel = new QLabel( this );
   tmplabel->setGeometry( 230, 10, 20, 20);
   tmplabel->setText( "@" );

   verteiler = new QLineEdit( this );
   verteiler->setGeometry( 250, 10, 100, 20);
   verteiler->setText( "" );
   verteiler->setMaxLength( 8 );

   lifetime = new QLineEdit( this );
   lifetime->setGeometry( 120, 30, 40, 20);
   lifetime->setText( "" );
   lifetime->setMaxLength( 3 );

   title = new QLineEdit( this );
   title->setGeometry( 120, 50, 230, 20);
   title->setText( "" );

   filename = new QLineEdit( this );
   filename->setGeometry( 120, 70, 170, 20);
   filename->setText( "" );

   info = new QLabel( this );
   info->setGeometry( 120, 90, 170, 20);
   info->setText( "" );

   search = new QPushButton( this );
   search->setGeometry( 300, 70, 50, 20 );
   search->setText( klocale->translate("S&earch") );
   connect(search, SIGNAL(clicked()), this, SLOT(slotSelectFile()));


   // Radiobuttons fuer den Mailtypen (S/SB/SP)
   tmpbtngrp = new QButtonGroup( this );
   tmpbtngrp->setGeometry( 10, 110, 150, 50 );
   tmpbtngrp->setFrameStyle( 49 );
   tmpbtngrp->setAlignment( 1 );
   tmpbtngrp->lower();
   tmpbtngrp->setTitle(klocale->translate("Send-Command"));

   cmd_s = new QRadioButton( tmpbtngrp );
   cmd_s->setGeometry( 10, 20, 40, 20 );
   cmd_s->setText( "S" );
   tmpbtngrp->insert( cmd_s );
   cmd_s->setChecked( true );

   cmd_sb = new QRadioButton( tmpbtngrp );
   cmd_sb->setGeometry( 50, 20, 40, 20 );
   cmd_sb->setText( "SB" );
   tmpbtngrp->insert( cmd_sb );

   cmd_sp = new QRadioButton( tmpbtngrp );
   cmd_sp->setGeometry( 100, 20, 40, 20 );
   cmd_sp->setText( "SP" );
   tmpbtngrp->insert( cmd_sp );


   // Radiobuttons fuer den Mailtypen (S/SB/SP)
   tmpbtngrp = new QButtonGroup( this );
   tmpbtngrp->setGeometry( 170, 110, 170, 50 );
   tmpbtngrp->setFrameStyle( 49 );
   tmpbtngrp->setAlignment( 1 );
   tmpbtngrp->lower();
   tmpbtngrp->setTitle(klocale->translate("Send files"));

   send_all = new QRadioButton( tmpbtngrp );
   send_all->setGeometry( 10, 20, 40, 20 );
   send_all->setText( "all" );
   tmpbtngrp->insert( send_all );
   send_all->setChecked( true );

   send_startwiththis = new QRadioButton( tmpbtngrp );
   send_startwiththis->setGeometry( 60, 20, 70, 20 );
   send_startwiththis->setText( "start with" );
   tmpbtngrp->insert( send_startwiththis );
   connect(send_startwiththis, SIGNAL(toggled(bool)), SLOT(slotToggleStartWith(bool)));

   startwith = new QLineEdit( tmpbtngrp );
   startwith->setGeometry( 130, 20, 30, 20);
   startwith->setText( "" );
   startwith->setMaxLength( 5 );
   startwith->setEnabled( false );



   int abstand = (width()-(2*BTNLENGTH))/3;
   QPushButton *send = new QPushButton( this );
   send->setGeometry(abstand, height()-BTNHEIGHT-5, BTNLENGTH, BTNHEIGHT);
   send->setText( klocale->translate("&Send") );
   send->setFocus();
   connect(send, SIGNAL(clicked()), this, SLOT(slotCheckInput()));

   QPushButton *abort = new QPushButton( this );
   abort->setGeometry(abstand*2+BTNLENGTH, height()-BTNHEIGHT-5, BTNLENGTH, BTNHEIGHT);
   abort->setText( klocale->translate("&Abort") );
   connect(abort, SIGNAL(clicked()), this, SLOT(reject()));

   board->setFocus();
}


void send7plus::slotToggleStartWith(bool active)
{
   startwith->setEnabled( active );
}


send7plus::~send7plus()
{
}


void send7plus::slotCheckInput()
{
   int lt;
   int startnr;

   // Gucken, ob alle noetigen Felder ausgefuellt sind.
   if (board->text()[0] == '\0')
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("You have to specify a boardname."));
      return;
   }
/*   if (verteiler->text()[0] == '\0')
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("You have to specify a @mbx-field."));
      return;
   }*/
   if (title->text()[0] == '\0')
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("You have to specify a mail-title."));
      return;
   }
   if (filename->text()[0] == '\0')
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("You have to specify a filename."));
      return;
   }
/*   if (lifetime->text()[0] == '\0')
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("You have to specify a Lifetime."));
      return;
   }*/

   // Hat die Lifetime die richtige Groesse?
   lt = atoi(lifetime->text());
   if (lt<1 || lt>999)
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("The lifetime, if given, must be between 1 and 999 days."));
      return;
   }

   // Existiert der Dateiname?
   if (!file_exist(filename->text()))
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("The specified file is not available."));
      return;
   }

   if (filecount == -1)
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("The specified file is not a 7plus-file."));
      return;
   }

   if (send_startwiththis->isChecked())
   {
      startnr = atoi(startwith->text());
      if (startnr < 1 || startnr>filecount)
      {
         KMsgBox::message(NULL, klocale->translate("Failure"),
                                klocale->translate("You cannot start with that filenumber."));
         return;
      }
   }

   if (checkForAllFiles())
      accept();
}


void send7plus::slotSelectFile()
{
   char tmp[500];

   strcpy(tmp, config->dirUp);
   strcpy(tmp, (const char *)QFileDialog::getOpenFileName(tmp));

   if (tmp[0] != '\0')
      filename->setText( tmp );
   checkFileCount();
}


//  bool send7plus::checkForAllFiles()
// Guckt, ob alle 7plus-Files, die zu dieser Datei gehoeren, vorhanden
// sind. Wenn ja, wird true zurueckgegeben, sonst false.
bool send7plus::checkForAllFiles()
{
   char str[1000], tmp[1000];
   int i;

   checkFileCount();

   COPY(str, filename->text(), 0, lPOS('.',filename->text()));

   if (filecount == 1)
   {
      sprintf(tmp, "%s.7pl", str);
      if (file_exist(tmp)) return true;
      i = lPOS('/',tmp);
      if (i != -1)
         COPY(tmp, tmp, i+1, strlen(tmp)-i-1);
      sprintf(str, klocale->translate("'%s' is not available."), tmp);
      KMsgBox::message(NULL, klocale->translate("Failure"), str );
      return false;
   }

   for (i=1;i<=filecount;i++)
   {
      sprintf(tmp, "%s.p%02x", str, i);
      if (!file_exist(tmp))
      {
         i = lPOS('/',tmp);
         if (i != -1)
            COPY(tmp, tmp, i+1, strlen(tmp)-i-1);
         sprintf(str, klocale->translate("'%s' is not available."), tmp);
         KMsgBox::message(NULL, klocale->translate("Failure"), str );
         return false;
      }
   }

   return true;
}


//  void send7plus::checkFileCount()
// Guckt, wieviele Dateien zu dem ausgewaehlten File gehoeren.
void send7plus::checkFileCount()
{
   FILE *f;
   char tmp[100];

   if ((f = fopen(filename->text(), "r")) == NULL)
   {
      filecount = -1;
      info->setText( klocale->translate("File not found.") );
      return;
   }

   if (fgets(tmp, 99, f) == NULL)
   {
      filecount = -1;
      info->setText( klocale->translate("Cannot read the file.") );
      fclose(f);
      return;
   }

   if (strncmp(tmp," go_7+. ",8))
   {
      filecount = -1;
      info->setText( klocale->translate("No 7plus-file.") );
      fclose(f);
      return;
   }

   // Koennte ein 7plus-File sein
    
   // Steht " of " an der richtigen Stelle?
   if (strncmp(tmp+11," of ",4))
   {
      // Noe.
      filecount = -1;
      info->setText( klocale->translate("No 7plus-file.") );
      fclose(f);
      return;
   }

   // Wieviele Teile sinds insgesamt?
   COPY(tmp,tmp,16,3);
   if ((filecount = atoi(tmp)) == 0)
   {
      filecount = -1;
      info->setText( klocale->translate("No 7plus-file.") );
      fclose(f);
      return;
   }

   sprintf(tmp,"%i files.", filecount);
   info->setText(tmp);

   fclose(f);
}


const char * send7plus::getFilename()
{
   return filename->text();
}


int send7plus::getFileCount()
{
   return filecount;
}


int send7plus::startWithFile()
{
   if (!send_startwiththis->isChecked()) return 1;

   return(atoi(startwith->text()));
}


const char * send7plus::getSendLine( int filenr )
{
   char str[500], tmp[500];

   if (cmd_s->isChecked())
      sprintf(str, "S %s", board->text());
   if (cmd_sp->isChecked())
      sprintf(str, "SP %s", board->text());
   if (cmd_sb->isChecked())
      sprintf(str, "SB %s", board->text());

   if (strlen(verteiler->text()) > 0)
   {
      strcat(str, " @ ");
      strcat(str, verteiler->text());
   }

   if (strlen(lifetime->text()) > 0)
   {
      strcat(str, " #");
      strcat(str, lifetime->text());
   }

   macroLine( tmp, title->text(), filenr );

   strcat(str, " ");
   strcat(str, tmp);
   strcat(str, "\r");

   return str;
}


void send7plus::macroLine( char *dest, const char *src, int filenr )
{
  int i,k,count;
  char help[500];


  count = strlen(src);

  i = 0;
  k = 0;
  while (i < count)
  {
    switch(src[i])
    {
      case '%': i++;
                if (i < count)
                   switch(src[i])
                   {
                     case '%': dest[k] = '%';
                               k++;
                               break;
                     case 'a': // Die Nummer des aktuellen Files
                               dest[k] = '\0';
                               sprintf(help,"%i", filenr);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'c': // Anzahl der 7plus-Dateien
                               dest[k] = '\0';
                               sprintf(help,"%i",filecount);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'A': // Die Nummer des aktuellen Files (hex)
                               dest[k] = '\0';
                               sprintf(help,"%.2x", filenr);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'C': // Anzahl der 7plus-Dateien (hex)
                               dest[k] = '\0';
                               sprintf(help,"%.2x",filecount);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     default: dest[k] = '%'; k++;
                              dest[k] = src[i]; k++;
                   }
                break;
      default: dest[k] = src[i];
               k++;
    }
    
    i++;
  }
  
  dest[k] = '\0';
}


////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////

AwayDlg::AwayDlg(QWidget *parent) : QDialog(parent, klocale->translate("Away"), TRUE)
{
   resize(340, 140);
   setFixedSize(size());

   setCaption(klocale->translate("Away"));

   // Fenster aufbauen
   QButtonGroup *tmpbtngrp;

   tmpbtngrp = new QButtonGroup( this );
   tmpbtngrp->setGeometry( 10, 10, 240, 115 );
   tmpbtngrp->setFrameStyle( 49 );
   tmpbtngrp->setAlignment( 1 );
   tmpbtngrp->lower();
   tmpbtngrp->setTitle(klocale->translate ("You are"));

   here = new QRadioButton( this );
   here->setGeometry( 20, 30, 200, 20 );
   here->setText( klocale->translate("here") );
   QToolTip::add( here, config->away.back );
   tmpbtngrp->insert( here );

   phone = new QRadioButton( this );
   phone->setGeometry( 20, 50, 200, 20 );
   phone->setText( klocale->translate("on the phone") );
   QToolTip::add( phone, config->away.phone );
   tmpbtngrp->insert( phone );

   sleep = new QRadioButton( this );
   sleep->setGeometry( 20, 70, 200, 20 );
   sleep->setText( klocale->translate("in bed") );
   QToolTip::add( sleep, config->away.bed );
   tmpbtngrp->insert( sleep );

   other = new QRadioButton( this );
   other->setGeometry( 20, 90, 20, 20 );
   other->setText( klocale->translate("away") );
   tmpbtngrp->insert( other );
   connect(other, SIGNAL(toggled(bool)), SLOT(slotToggleOther(bool)));

   othertext = new QLineEdit( this );
   othertext->setGeometry( 40, 95, 200, 20 );
   othertext->setText( config->away.text );
   othertext->setMaxLength( 60 );


   ok = new QPushButton( this );
   ok->setGeometry( 260, 30, 70, 25 );
   ok->setText( klocale->translate ("&OK") );
   connect(ok, SIGNAL(clicked()), this, SLOT(slotReady()));
   ok->setDefault(true);

   QAccel *a = new QAccel( this );
   a->connectItem( a->insertItem( Key_Return ), this, SLOT(slotReady()) );

   abort = new QPushButton( this );
   abort->setGeometry( 260, 65, 70, 25 );
   abort->setText( klocale->translate ("&Abort") );
   connect(abort, SIGNAL(clicked()), this, SLOT(reject()));


   // Den richtigen Radiobutton anwaehlen
   here->setFocus();
   switch (config->away.wo)
   {
      case 1: phone->setChecked( true );
              othertext->setEnabled( false );
              break;
      case 2: sleep->setChecked( true );
              othertext->setEnabled( false );
              break;
      case 3: other->setChecked( true );
              othertext->setEnabled( true );
              break;
      default: here->setChecked( true );
               phone->setFocus();
               othertext->setEnabled( false );
               break;

   }
}


AwayDlg::~AwayDlg()
{
}


void AwayDlg::slotToggleOther(bool checked)
{
   othertext->setEnabled( checked );
}


void AwayDlg::slotReady()
{
   strcpy(config->away.text, othertext->text());
   accept();
}


void AwayDlg::getInfos(unsigned char & wo, char *text)
{
   if (here->isChecked())
      wo = 0;
   if (phone->isChecked())
      wo = 1;
   if (sleep->isChecked())
      wo = 2;
   if (other->isChecked())
      wo = 3;

   strcpy(text, othertext->text());
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


DlgAddRoute::DlgAddRoute(QWidget *parent) : QDialog(parent, klocale->translate("Add route"), TRUE)
{
   QLabel *l;
   s_ports *tmp;


   setFixedSize(330, 140);
   setCaption(klocale->translate("Add route"));


   l = new QLabel(klocale->translate("call"), this);
   l->setGeometry(10, 5, 100, 20);
   l->resize( l->sizeHint() );

   l = new QLabel(klocale->translate("via"), this);
   l->setGeometry(10, 50, 100, 20);
   l->resize( l->sizeHint() );

   l = new QLabel(klocale->translate("port"), this);
   l->setGeometry(220, 5, 100, 20);
   l->resize( l->sizeHint() );


   call = new QLineEdit( this );
   call->setGeometry(10, 25, 100, 20);

   port = new QComboBox( this );
   port->setGeometry(220, 25, 100, 20);
   // Hier werden alle vorhandenen Portnamen in die Combobox eingetragen
   tmp = portlist;
   while (tmp != NULL)
   {
      if (tmp->ax25 != NULL)
         port->insertItem(tmp->name);
      tmp = tmp->next;
   }

   via = new QLineEdit( this );
   via->setGeometry(10, 70, 310, 20);

   // OK- und Abbruch-Knopf
   int abstand = (width()-(2*BTNLENGTH))/3;
   btnok = new QPushButton( this );
   btnok->setText( klocale->translate ("&Ok") );
   btnok->setGeometry(abstand, 115, BTNLENGTH, BTNHEIGHT);
   btnok->setDefault( true );
   connect(btnok, SIGNAL(clicked()), this, SLOT(accept()));
   btnok->setDefault(true);

   btnabort = new QPushButton( this );
   btnabort->setText( klocale->translate ("&Abort") );
   btnabort->setGeometry(abstand*2+BTNLENGTH, 115, BTNLENGTH, BTNHEIGHT);
   connect(btnabort, SIGNAL(clicked()), this, SLOT(reject()));

   QAccel *a = new QAccel( this );
   a->connectItem( a->insertItem(Key_Return), this, SLOT(accept()));
   call->setFocus();
}


DlgAddRoute::~DlgAddRoute()
{
}


void DlgAddRoute::getRoute( char *_call, char *_via, char *_port )
{
   strcpy(_call, call->text());
   strcpy(_via, via->text());
   strcpy(_port, port->currentText());
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


DlgYAPP::DlgYAPP( QWidget *parent ) : QDialog( parent, "", true )
{
   setFixedSize( 250, 200 );
   setCaption(klocale->translate("YAPP-Transfer"));


   btngrp = new QButtonGroup( this );
   btngrp->setGeometry( 0, 10, 230, 50 );
   btngrp->setFrameStyle( QButtonGroup::NoFrame );

   rx = new QRadioButton( btngrp );
   rx->setGeometry( 10, 10, 210, 20 );
   rx->setText( klocale->translate("Receive YAPP") );

   tx = new QRadioButton( btngrp);
   tx->setGeometry( 10, 30, 210, 20 );
   tx->setText( klocale->translate("Transmit YAPP") );


   lFName = new QLabel( this );
   lFName->setGeometry( 10, 75, 230, 20 );
   lFName->setText( klocale->translate("Filename") );

   filename = new QLineEdit( this );
   filename->setGeometry( 10, 95, 230, 20 );

   browse = new QPushButton( this );
   browse->setGeometry( 180, 120, 60, 25 );
   browse->setText( klocale->translate("browse") );
   connect( browse, SIGNAL(clicked()), this, SLOT(slotBrowseFile()) );



   int abstand = (width()-(2*BTNLENGTH))/3;
   btnok = new QPushButton( this );
   btnok->setGeometry(abstand, height()-BTNHEIGHT-5, BTNLENGTH, BTNHEIGHT);
   btnok->setText( klocale->translate("&Ok") );
   btnok->setFocus();
   connect(btnok, SIGNAL(clicked()), this, SLOT(accept()));

   btnabort = new QPushButton( this );
   btnabort->setGeometry(abstand*2+BTNLENGTH, height()-BTNHEIGHT-5, BTNLENGTH, BTNHEIGHT);
   btnabort->setText( klocale->translate("&Abort") );
   connect(btnabort, SIGNAL(clicked()), this, SLOT(reject()));


   rx->setChecked( true );
}


DlgYAPP::~DlgYAPP()
{
   delete rx;
   delete tx;
   delete btngrp;
   delete lFName;
   delete filename;
   delete browse;
   delete btnok;
   delete btnabort;
}


void DlgYAPP::getData( int & xfer, char *name )
{
   xfer = 0;
   if (rx->isChecked()) xfer = 1;
   if (tx->isChecked()) xfer = 2;

   strcpy(name, filename->text());
}


void DlgYAPP::slotBrowseFile()
{
   char tmp[2000];


   if (tx->isChecked())
   {
      strcpy(tmp, config->dirUp);
      QString f = QFileDialog::getOpenFileName(tmp);

      // Cancel
      if (f.isEmpty()) return;

      strcpy(tmp, (const char *)f);

      if (!file_exist(tmp))
      {
         KMsgBox::message(NULL, klocale->translate("File not found"),
                                klocale->translate("Cannot find specified file."));
         return;
      }

      filename->setText(tmp);
      return;
   }


   if (rx->isChecked())
   {
      strcpy(tmp, config->dirABin);
      QString f = QFileDialog::getSaveFileName(tmp);

      strcpy(tmp, (const char *)f);

      // Cancel
      if (f.isEmpty()) return;

      if (file_exist(tmp))
      {
         if (KMsgBox::yesNo(NULL, klocale->translate("Overwrite?"),
                                  klocale->translate("The specified file allready exists. Overwrite?"),
                                  0,
                                  klocale->translate("&Yes"),
                                  klocale->translate("&No")) != 1)
         return;
      }

      filename->setText(tmp);
      return;
   }

   KMsgBox::message(NULL, klocale->translate("YAPP-Transfer"),
                          klocale->translate("Do you want to upload or save a YAPP-file?\nPlease choose."));
}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////


TransferInfo::TransferInfo( QWidget *parent, int art, char *filename, int size )
                   : QWidget( parent )
{
   setFixedHeight( 23 );
   int x=0,y=2;

   filesize = size;
   starttime = time(NULL);

   progressbar = new QProgressBar( 500, this );
   progressbar->setGeometry( x, y, 150, 20 );
   progressbar->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   x += 150;


   lFileName = new QLabel( this );
   lFileName->setGeometry( x, y, 150, 20 );
   lFileName->setText( filename );
   lFileName->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   x += 150;

   lTimeLeft = new QLabel( this );
   lTimeLeft->setGeometry( x, y, 55, 20 );
   lTimeLeft->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   x += 55;

   lBytes = new QLabel( this );
   lBytes->setGeometry( x, y, 90, 20 );
   lBytes->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   lBytes->setAlignment( AlignRight|AlignVCenter );
   x += 90;

   lBaudEff = new QLabel( this );
   lBaudEff->setGeometry( x, y, 80, 20 );
   lBaudEff->setFrameStyle( QFrame::Panel | QFrame::Sunken );
   lBaudEff->setAlignment( AlignRight|AlignVCenter );
   x += 80;

   show();
}


TransferInfo::~TransferInfo()
{
   delete progressbar;
   delete lBytes;
   delete lBaudEff;
   delete lFileName;
   delete lTimeLeft;
}


void TransferInfo::setReceivedBytes( int rxbytes )
{
   time_t timediff=time(NULL) - starttime;
   time_t timeleft;
   char hours[5], mins[5], secs[5], tmp[100];

   bytes_got = rxbytes;

   sprintf(tmp, "%i bytes", rxbytes);
   lBytes->setText( tmp );

   progressbar->setProgress( (int) (((float)rxbytes/(float)filesize) * 500) );

   // Baud-Rate
   if (timediff == 0) timediff = 1;
   sprintf(tmp, "%li bps", (rxbytes*8)/timediff);
   lBaudEff->setText( tmp );

   // Verbleibende Zeit
   if ((bytes_got/timediff) < 1)
      timeleft = 1;
   else
      timeleft = filesize/(rxbytes/timediff)-timediff;

   sprintf(secs, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(mins, "%.2i", (int)timeleft%60);
   timeleft /= 60;
   sprintf(hours, "%.2i", (int)timeleft);

   sprintf(tmp, "%s:%s:%s", hours, mins, secs);
   lTimeLeft->setText( tmp );
}


void TransferInfo::setFilesizeBytes( int bytes )
{
   filesize = bytes;
}



#include "dialog.moc"

