/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "channel.h"

#include <qfiledialog.h>
#include "settingsdlg.h"
#include "toolbox.h"
#include "global.h"
#include "version.h"
#include "main.h"
#include "comp.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#ifdef HAVE_BROKEN_AX25_H
#include "local_ax25.h"
#include <netrose/rose.h>
#elif HAVE_NETAX25_AX25_H
#include <netax25/ax25.h>
#include <netrose/rose.h>
#else
#include <linux/ax25.h>
#include <linux/rose.h>
#endif


#include <kmsgbox.h>
#include <kapp.h>

#include <qclipboard.h>


extern TopLevel *toplevel;
extern KApplication *mykapp;


char TNC_Status[15][4] = {"DIS","SET","FMR","DRQ","IXF","REJ","WAK","DBS","RBS","BBS","WDB","WRB",
                          "RDB","RRB","RBB"};

QColor colors[17] = {black,white,darkGray,gray,lightGray,red,green,
                     blue,cyan,magenta,yellow,darkRed,darkGreen,darkBlue,
                     darkCyan,darkMagenta,darkYellow};

// System-Meldungen
static char conn_str[] = "CONNECTED to";
static char disc_str[] = "DISCONNECTED fm";


s_swtype swType[SW_ANZ] = {{SW_UNKNOWN, "Unknown"},
                           {SW_AWZNODE, "AWZNode"},
                           {SW_BAYBOX,  "BayBox"},
                           {SW_DPBOX,   "DPBOX"},
                           {SW_FBB,     "FBB"},
                           {SW_FLEXNET, "FlexNet"},
                           {SW_GP85,    "GP85"},
                           {SW_LINKT,   "LinKT"},
                           {SW_MCUT,    "MCuT"},
                           {SW_THEBOX,  "TheBox"},
                           {SW_TNN,     "TNN"},
                           {SW_WINGT,   "WinGT"},
                           {SW_XPACKET, "XPacket"}};


//  Konstruktor fuer die Klasse Channel.
// Hier wird das zu diese Kanal gehoerende Fenster aufgebaut, alle
// Kernel-AX.25-Geschichten erledigt etc.
Channel::Channel( QWidget *parent, char *port, char *mycall, char *call, char *digis, int fd, bool connected )
           : QDialog(parent)
{
   char tmp[100];
   int screenwidth;
   QFont f(config->rxfontstr, config->qsofontsize);
   QFont vorschreib_f(config->txfontstr, config->txfontsize);
   int idle=0;

   // Variablen initialisieren
   transferDlg = NULL;
   save_text = NULL;
   save_text_len = 0;
   save_text_showed = 0;
   len_allready_showed = 0;
   typeCheckLine = 0;
   bytesRX = 0;
   bytesTX = 0;
   closeWinDisc = false;


   // Fenster-Ueberschrift
   sprintf(tmp,"LinKT: %s <-> %s (port: %s)",mycall, call, port);
   setCaption( tmp );


   // Die Statusbar
   statusbar = new KStatusBar( this );
   statusbar->insertItem("UnSent: 00", ID_STATUS_UNSENT);
   statusbar->insertItem("UnAck: 00", ID_STATUS_UNACK);
   statusbar->insertItem("Retry: 00", ID_STATUS_RETRY);
   statusbar->insertItem("IXF ", ID_STATUS_STATE);
   statusbar->insertItem("XXXXXXXXXXXXXXXXXXXX", ID_GENERAL);


   statusbar->changeItem("UnSent: 0", ID_STATUS_UNSENT);
   statusbar->changeItem("UnAck: 0", ID_STATUS_UNACK);
   statusbar->changeItem("Retry: 0", ID_STATUS_RETRY);
   statusbar->changeItem("", ID_GENERAL);

   if (connected)
   {
      statusbar->changeItem("IXF", ID_STATUS_STATE);
      old_status = 4;
   }
   else
   {
      statusbar->changeItem("SET", ID_STATUS_STATE);
      old_status = 1;
   }


   flags = CH_STDFLAGS;

   pannerHome = new QWidget( this );
   pannerHome->setGeometry(0,statusbar->height(),width(),height()-statusbar->height());

   panner = new KNewPanner(pannerHome, "panner", KNewPanner::Horizontal,
                                                 KNewPanner::Percent);
   panner->resize(pannerHome->size());

   vorschreib = new myVorschreib(toplevel->chanListTable, this, pannerHome);
   if (config->colors->txwin_bold)
      vorschreib_f.setBold(true);
   vorschreib->setFont ( vorschreib_f );
   vorschreib->setColors( colors[config->colors->txwin], colors[config->colors->background] );
   vorschreib->setBackgroundColor( colors[config->colors->txbackground] );
   vorschreib->setFocus();

   output = new OutputWidget(pannerHome);
   if (config->colors->qsowin_bold)
      f.setBold(true);
   output->setFont( f );
   output->setBackgroundColor( colors[config->colors->background] );
   connect( output, SIGNAL(kontextMenu()), SLOT(showKontextMenu()) );
   connect( vorschreib, SIGNAL(returnPressed()), SLOT(returnPressedGot()) );
   connect( vorschreib, SIGNAL(signalPageUp()), output, SLOT(slotPageUp()) );
   connect( vorschreib, SIGNAL(signalPageDown()), output, SLOT(slotPageDown()) );
   connect( vorschreib, SIGNAL(signalFunctionKey(int)), toplevel, SLOT(slotCallKey(int)) );
   connect( vorschreib, SIGNAL(kontextMenu()), SLOT(showKontextVorschrMenu()) );

   // Die Spaltenbreite auf 80 * die Breite des Zeichens 'm' einstellen
   QFontMetrics fm(f);
   screenwidth = 84 * fm.width("m");
   resize(screenwidth, (int)(screenwidth * 0.6));

   // Soll das Vorschreibfenster oben oder unten sein?
   if (config->txWinOben)
   {
      panner->activate(vorschreib,output);
      panner->setSeparatorPos( config->txWinSize );
   }
   else
   {
      panner->activate(output,vorschreib);
      panner->setSeparatorPos( 100-config->txWinSize );
   }

   if (config->openChannelWin || !connected)
      show();

   // Variablen besetzen
   data_firstcall = strdup(call);
   data_call = strdup(call);
   data_mycall = strdup(mycall);
   data_logintime = time(NULL);
   data_port = strdup(port);
   data_digis = strdup(digis);

   sendqueue = NULL;
   framesize = 255;
   externalConnection = false;


   // Verbindungs-Kommunikation
   txfd = fd;
   sockrx = new QSocketNotifier( fd, QSocketNotifier::Read, this );
   connect( sockrx, SIGNAL(activated(int)), this, SLOT(rxFrame(int)) );

   // IDLE-Disconnect abschalten
   setsockopt( fd, SOL_AX25, AX25_IDLE, &idle, sizeof(idle) );

   // Shortcuts fuer diverse Befehle erzeugen
   accels = new QList<QAccel>();

   QAccel *a;
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_C+ALT ), this, SLOT(newConnect()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_D+ALT ), this, SLOT(discStation()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_S+ALT ), this, SLOT(settings()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_F+ALT ), this, SLOT(slotSendFile()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_7+ALT ), this, SLOT(slotSend7plus()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_T+ALT ), this, SLOT(slotSaveText()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_B+ALT ), this, SLOT(slotSendAbinFile()) );
   // Kanaele durchwechseln mit CTRL-Up / CTRL-Down
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_Up+CTRL ), this, SLOT(slotCTRLKeyUp()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_Down+CTRL ), this, SLOT(slotCTRLKeyDown()) );
   // Passwort mit ALT-P aussenden
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_P+ALT ), this, SLOT(slotSendPW()) );
   // Away-Dialog mit ALT-A
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_A+ALT ), this, SLOT(slotAway()) );
   // Mit der Menue-Taste wird das Kontextmenu angezeigt
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_Menu ), this, SLOT(showKontextMenu()) );
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_M+ALT ), toplevel->chanListTable, SLOT(activateWindow()) );
   // Mit CTRL-C kann die Check-Liste dieses Kanals in den Vordergrund geholt werden
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_C+CTRL ), this, SLOT(activateCheckWin()) );
   // Mit ALT-H kann die TOP-Online-Komprimierung (#HUF# aktiviert werden)
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_H+ALT ), this, SLOT(slotHufComp()) );
   // Mit CTRL-Y kann ein YAPP-File ausgesendet oder empfangen werden
   a = new QAccel( this );
   accels->append(a);
   a->connectItem( a->insertItem( Key_Y+CTRL ), this, SLOT(slotYAPP()) );


   // Kontextmenue erstellen
   kontextmenu = new QPopupMenu();
   kontextmenu->insertItem( klocale->translate("&Connect"), this, SLOT(newConnect()));
   kontextmenu->insertItem( klocale->translate("&Disconnect"), this, SLOT(discStation()));
   kontextmenu->insertSeparator( -1 );
   kontextmenu->insertItem( klocale->translate("&Settings"), this, SLOT(settings()));
   kontextmenu->insertSeparator( -1 );
   kontextmenu->insertItem( klocale->translate("Send &File"), this, SLOT(slotSendFile()));
   kontextmenu->insertItem( klocale->translate("Send &7plus"), this, SLOT(slotSend7plus()));
   mnu_sendabin = kontextmenu->insertItem( klocale->translate("Send &AutoBIN"), this, SLOT(slotSendAbinFile()));
   mnu_didadit = kontextmenu->insertItem( klocale->translate("Send D&IDADIT"), this, SLOT(slotSendDIDADIT()));
   mnu_yapp = kontextmenu->insertItem( klocale->translate("&YAPP-Transfer"), this, SLOT(slotYAPP()));
   mnu_textfile = kontextmenu->insertItem( klocale->translate("Save &Textfile"), this, SLOT(slotSaveText()));
   kontextmenu->insertSeparator( -1 );
   mnu_boxcheck = kontextmenu->insertItem( klocale->translate("BoxCheck"), this, SLOT(activateCheckWin()));
   kontextmenu->setItemEnabled( mnu_boxcheck, false );

   // Kontextmenue fuer die rechte Maustaste im Vorschreib-Fenster erstellen
   kontextvorschr = new QPopupMenu();
   kontextvorschr->insertItem( klocale->translate("&Send quoted"), this, SLOT(slotSendQuoted()));
   kontextvorschr->insertItem( klocale->translate("&Insert quoted"), this, SLOT(slotInsertQuoted()));

   // Remote-Geschichten
   remote = new RemoteCmds( this );

   // Die Passwort-Verarbeitung
   password = new Passwords( this );

   // Das BoxCheck-Zeugs
   boxcheck = new BoxCheck( this );

   // User-Informationen
   userinfo = new UserInfo( data_call );
   boxcheck->readData( userinfo->getBoxCheck() );

   // TOP-Huffman
   ahuf = new AHUF();

   // Die QSO-Liste im Hauptfenster
   toplevel->chanListTable->addLine( this, call, mycall, userinfo->getType(), connected );
   toplevel->setMainwinSize();

   if (!connected)
      toplevel->chanListTable->setMode( this, MODE_SETUP );


   // Binaer- und 7plus-Transfers initialisieren
   auto7 = new auto7plus( this );
   binrx = new BinRX( this );
   bintx = NULL;
   didadit = NULL;
   yapp = NULL;

   conncalls = NULL;

   if (connected)
   {
      timer = new QTimer(this);
      connect(timer, SIGNAL(timeout()), SLOT(slotTimer()));
      timer->start(250, true);

      // eine connect-Meldung aussenden
      showConnectMessage(port, call, digis);

      // den Connect-Text aussenden
      sendCText();
      // Bimmeln
      toplevel->playSound( SOUND_CONNECT );
   }
   else
      timer = NULL;
}


Channel::~Channel()
{
   s_conncalls *conntmp;


   // Sendqueue leer machen
   while (sendqueue != NULL) deleteQueueEntry();

   // In dieser Klasse gespeicherte Daten wieder freigeben


   // conncalls
   while  (conncalls != NULL)
   {
      conntmp = conncalls;
      conncalls = conncalls->next;
      free(conntmp);
   }

   // Dateiuebertragungen abbrechen
   if (didadit != NULL) delete didadit;
   if (auto7 != NULL) delete auto7;
   if (binrx != NULL) delete binrx;
   if (bintx != NULL) delete bintx;
   if (yapp != NULL) delete yapp;


   // QSO aus der QSO-Liste des Hauptfensters entfernen
   toplevel->chanListTable->delLine( this );
   toplevel->setMainwinSize();

   // Widgets
   delete panner;
   delete statusbar;
   delete pannerHome;
   delete kontextmenu;

   if (timer != NULL) delete timer;

   if (data_firstcall != NULL) free(data_firstcall);
   if (data_call != NULL) free(data_call);
   if (data_mycall != NULL) free(data_mycall);
   if (data_port != NULL) free(data_port);
   if (data_digis != NULL) free(data_digis);
   if (save_text != NULL) free(save_text);


   // Sonstige Klassen
   delete accels;       // Speicher fuer die Tastatur-Kuerzel freigeben
   delete userinfo;
   delete remote;
   delete ahuf;
   delete password;
//   delete output;
//   delete vorschreib;
   delete boxcheck;
}


void Channel::resizeEvent(QResizeEvent *event)
{
   statusbar->setGeometry(0,0,width(),statusbar->height());
   pannerHome->setGeometry( 0, statusbar->height(), width(), height()-statusbar->height() );
   if (transferDlg == NULL)
      panner->resize(pannerHome->size());
   else
   {
      panner->resize( pannerHome->width(), pannerHome->height()-transferDlg->height() );
      transferDlg->setGeometry( 0, pannerHome->height()-transferDlg->height(), pannerHome->width(), transferDlg->height() );
   }

   QDialog::resizeEvent(event);

   // Dieses Fenster als neue Standardgroesse/Position festlegen
   // (fuer hideWinOnSwitch)
//   toplevel->setQsoWinGeometry( geometry() );
}


//  void Channel::outText( char *text, int len, int colorcode )
// Einen Text im RX-Fenster ausgeben.
void Channel::outText( char *text, int len, int colorcode )
{
   char *str,*tmp;
   int str_len,i;

   str = (char *) malloc( len+1 );
   tmp = (char *) malloc( len+1 );
   memcpy(str, text, len);
   str[len] = '\0';
   str_len = len;

   // Text im Hauptfenster in fett anzeigen
   if (!isActiveWindow())
      toplevel->chanListTable->setNew( this, true );

   while ((str_len>0) && ((i = POS('\xD',str)) != -1))
   {
      memcpy(tmp,str,i);
      tmp[i] = '\0';
      memmove(str,str+i+1,str_len-i-1);
      str_len = str_len-i-1;
      str[str_len] = '\0';

      output->writeText(tmp, colors[colorcode]);
      output->newLine();
   }

   if (str_len>0)
   {
      output->writeText(str, colors[colorcode]);
   }

   free(tmp);
   free(str);
}


//  void Channel::showConnectMessage(char *port, char *call, char *digis)
// Zeigt einen Text an, der z.B. wie folgt aussieht:
//   *** (23cm) CONNECTED to DG4IAD via DB0EA DB0HDB [Connect-Zeit]
void Channel::showConnectMessage(char *port, char *call, char *digis)
{
   char tmp[100];
   struct tm sTm;

   sTm = *localtime(&data_logintime);

   if (digis[0] == '\0')
      sprintf(tmp,"\r*** (%s) %s %s [%.2i.%.2i.%.4i %.2i:%.2i]\r",port,conn_str,call,
                  sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900, sTm.tm_hour,sTm.tm_min);
   else
      sprintf(tmp,"\r*** (%s) %s %s via %s [%.2i.%.2i.%.4i %.2i:%.2i]\r",port,conn_str,call,digis,
                  sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900, sTm.tm_hour,sTm.tm_min);
   outText( tmp, strlen(tmp), config->colors->statusText );
}


//  void Channel::showDisconnectMessage(char *port, char *call, char *digis)
// Zeigt einen Text an, der z.B. wie folgt aussieht:
//   *** (23cm) DISCONNECTED fm DG4IAD via DB0EA DB0HDB
void Channel::showDisconnectMessage(char *port, char *call, char *digis)
{
   char tmp[100];
   struct tm sTm;
   time_t t;

   t = time(0);
   sTm = *localtime(&t);

   if (digis[0] == '\0')
      sprintf(tmp,"\r*** (%s) %s %s [%.2i.%.2i.%.4i %.2i:%.2i]\r",port,disc_str,call,
                  sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900, sTm.tm_hour,sTm.tm_min);
   else
      sprintf(tmp,"\r*** (%s) %s %s via %s [%.2i.%.2i.%.4i %.2i:%.2i]\r",port,disc_str,call,digis,
                  sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900, sTm.tm_hour,sTm.tm_min);
   outText( tmp, strlen(tmp), config->colors->statusText );
}


//  void Channel::macroLine(char *dest, char *src)
// Liest den src-String ein, und wandelt eventuell vorkommende Makros um.
// Das Ergebnis wird nach dest geschrieben.
void Channel::macroLine(char *dest, char *src)
{
  int i,k,count;
  char help[5000];
  struct tm sTm;
  time_t akttime;


  count = strlen(src);

  i = 0;
  k = 0;
  while (i < count)
  {
    switch(src[i])
    {
      case '\n': dest[k] = '\r';
                 k++;
                 break;
      case '%': i++;
                if (i < count)
                   switch(src[i])
                   {
                     case '%': dest[k] = '%';
                               k++;
                               break;
                     case 'V': // Versions-Info mit Patchlevel
                               dest[k] = '\0';
                               sprintf(help,"%i.%i.%i",MAJORLEVEL,MINORLEVEL,PATCHLEVEL);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'v': // Versions-Info ohne Patchlevel
                               dest[k] = '\0';
                               sprintf(help,"%i.%i",MAJORLEVEL,MINORLEVEL);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'n': // Wenn der Name der Gegenstation bekannt ist, wird dieser
                               // ausgesendet, ansonsten das Rufzeichen.
                               dest[k] = '\0';
                               if (userinfo->getName() != NULL)
                                  if (userinfo->getName()[0] != '\0')
                                     strcpy(help,userinfo->getName());
                                  else
                                     strcpy(help,data_call);
                               else
                                  strcpy(help,data_call);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'D': // Einlog-Datum
                              sTm = *localtime(&data_logintime);
                              dest[k] = '\0';
                              sprintf(help,"%.2i.%.2i.%.4i",sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900);
                              memcpy(dest+k,help,strlen(help));
                              k = k + strlen(help);
                               break;
                     case 'd': // Aktuelles Datum
                               akttime = time(NULL);
                               sTm = *localtime(&akttime);
                               dest[k] = '\0';
                               sprintf(help,"%.2i.%.2i.%.4i",sTm.tm_mday,sTm.tm_mon+1,sTm.tm_year+1900);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'T': // Einlog-Zeit
                               sTm = *localtime(&data_logintime);
                               dest[k] = '\0';
                               sprintf(help,"%.2i:%.2i:%.2i",sTm.tm_hour,sTm.tm_min,sTm.tm_sec);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'a': // Away-Zeit
                               sTm = *localtime(&config->away.awaytime);
                               dest[k] = '\0';
                               sprintf(help,"%.2i:%.2i:%.2i",sTm.tm_hour,sTm.tm_min,sTm.tm_sec);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 't': // aktuelle Zeit
                               akttime = time(NULL);
                               sTm = *localtime(&akttime);
                               dest[k] = '\0';
                               sprintf(help,"%.2i:%.2i:%.2i",sTm.tm_hour,sTm.tm_min,sTm.tm_sec);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'm': // eigenes Call
                               dest[k] = '\0';
                               strcpy(help,data_mycall);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'c': // Call der Gegenstation
                               dest[k] = '\0';
                               strcpy(help,data_call);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'p': // Portname
                               dest[k] = '\0';
                               strcpy(help,data_port);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;
                     case 'o': // Cookie
                     case 'O':
                               dest[k] = '\0';
                               getCookie(help);
                               memcpy(dest+k,help,strlen(help));
                               k = k + strlen(help);
                               break;

                     default: dest[k] = '%'; k++;
                              dest[k] = src[i]; k++;
                   }
                break;
      default: dest[k] = src[i];
               k++;
    }
    
    i++;
  }
  
  dest[k] = '\0';
}


//  void Channel::sendMacrofile( char *filename )
// Wertet im uebergebenen File alle Makros aus und sendet es.
void Channel::sendMacrofile( char *filename )
{
  int fd,i,count;
  char tmp[500],str[500],dest[500],output_str[1000];

  if ((fd = open(filename,O_RDONLY)) == -1) return;
  
  while ((count = read(fd,tmp,256)) > 0)
  {
    tmp[count] = '\0';
    output_str[0] = '\0';
    while (tmp[0] != '\0')
    {
      // Das Frame in einzelne Zeilen zerlegen
      if ((i = POS('\r',tmp)) != -1)
      {
        // Return kommt drin vor
        COPY(str,tmp,0,i);
        COPY(tmp,tmp,i,strlen(str)+i);
      }
      else
      {
        // Kein Return - das war das Ende
        strcpy(str,tmp);
        tmp[0] = '\0';
      }

      macroLine( dest,str );
      strcat( output_str,dest );
    }

    sendString(strlen(output_str), output_str, true);
    output_str[0] = '\0';
  }

  ::close(fd);
}



//  void Channel::sendString(int len, char *data)
// Schickt diesen String aus (= traegt ihn in die Sendqueue ein)
void Channel::sendString(int len, const char *data, bool show)
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->len = len;
   tmp->data = (char *)malloc(len);
   memcpy(tmp->data,data,len);
   tmp->show = show;
   tmp->abin = false;
   tmp->comp = COMP_FLAG;
   tmp->type = SENDQ_TEXT;
   tmp->fd = -1;
}


//   void Channel::sendStringComp(int len, const char *data, bool show, int comp)
//
// Sendet den String aus und setzt die comp-Variable auf den uebergebenen
// Wert
void Channel::sendStringComp(int len, const char *data, bool show, int comp)
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->len = len;
   tmp->data = (char *)malloc(len);
   memcpy(tmp->data,data,len);
   tmp->show = show;
   tmp->abin = false;
   tmp->comp = comp;
   tmp->type = SENDQ_TEXT;
   tmp->fd = -1;
}


//  void Channel::sendString( char *data )
// Sendet einen null-terminierten String aus und zeigt ihn auf dem Bildschirm
// an.
void Channel::sendString( const char *data )
{
   sendString( strlen(data), data, true );
}


//  void Channel::sendFile( char *filename )
// Sendet eine Datei aus (traegt sie in die Sendqueue ein).
void Channel::sendFile( char *filename, bool show, bool abin)
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->data = (char *) strdup(filename);
   tmp->show = show;
   tmp->abin = abin;
   tmp->comp = COMP_FLAG;
   tmp->type = SENDQ_FILE;
   tmp->fd = -1;
}


//  void Channel::sendTextFile( char *filename )
// Sendet eine Datei aus (traegt sie in die Sendqueue ein).
void Channel::sendTextFile( char *filename, bool show)
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->data = (char *) strdup(filename);
   tmp->show = show;
   tmp->abin = false;
   tmp->comp = COMP_FLAG;
   tmp->type = SENDQ_TEXTFILE;
   tmp->fd = -1;
}


//  bool Channel::sendqueueSendTextLine()
// Sendet den Text der ersten Zeile aus.
void Channel::sendqueueSendTextLine()
{
   int len=0, tmplen=0;
   char *sendstr=NULL, *tmpstr=NULL;
   bool show = sendqueue->show;
   int comp=sendqueue->comp;
   bool abin = sendqueue->abin;

   if (sendqueue->len <= framesize)
   {
      // Der ganze String kann in einem Rutsch ausgesendet werden.
      sendstr = (char *) malloc( sendqueue->len+1 );
      memcpy(sendstr, sendqueue->data, sendqueue->len);
      len = sendqueue->len;

      // Loescht den ersten Eintrag der Queue
      deleteQueueEntry();
   }
   else
   {
      // Der String muss geteilt werden. Den zu sendenden Teil (framsize
      // bytes) nach sendstr, der Rest wieder nach queue->data schreiben.

      // Die Daten aus ->text an tmpstr/tmplen speichern
      tmpstr = sendqueue->data;
      tmplen = sendqueue->len;

      // den zu sendenden Text aus diesem Speicher nach sendstr/len schreiben
      sendstr = (char *) malloc( framesize );
      memcpy(sendstr, tmpstr, framesize);
      len = framesize;

      // die daten, die noch nicht gesendet wurden, in die sendqueue zurueck
      // schreiben
      sendqueue->len = sendqueue->len - framesize;
      sendqueue->data = (char *) malloc(sendqueue->len);
      memcpy(sendqueue->data,tmpstr+framesize,sendqueue->len);

      free(tmpstr);
   }

   // Sendet den Inhalt von sendstr aus.
   sendFrame( sendstr, len, comp );

   // Hier wird geguckt, ob der zu sendende Text angezeigt werden soll
   if (show)
   {
      sendstr[len] = '\0';
      if (abin)
         outText( sendstr+4, len, config->colors->txText );
      else
         outText( sendstr, len, config->colors->txText );
   }

   free(sendstr);
}


//   void Channel::sendDidadit()
//
// Ein File soll mittels DIDADIT ausgesendet werden.
void Channel::sendDidadit()
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      // Wir brauchen nur einen einzigen DIDADIT-Eintrag
      if (sendqueue->type == SENDQ_DIDADIT)
         return;

      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->data = NULL;
   tmp->show = false;
   tmp->abin = false;
   tmp->comp = COMP_NO;
   tmp->type = SENDQ_DIDADIT;
   tmp->fd = -1;
   tmp->ready = false;
}


void Channel::sendYAPP()
{
   s_sendqueue *tmp;

   if (data_logintime == -1) return;

   if (sendqueue == NULL)
   {
      // Erster Eintrag
      sendqueue = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = sendqueue;
   }
   else
   {
      // Wir brauchen nur einen einzigen YAPP-Eintrag
      if (sendqueue->type == SENDQ_YAPP)
         return;

      tmp = sendqueue;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_sendqueue *)malloc(sizeof(s_sendqueue));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->data = NULL;
   tmp->show = false;
   tmp->abin = false;
   tmp->comp = COMP_NO;
   tmp->type = SENDQ_YAPP;
   tmp->fd = -1;
   tmp->ready = false;
   tmp->len = 0;
}



//  void Channel::sendqueueSendFile( bool textfile )
// Sendet ein File aus. Wenn textfile true ist, wird eine konvertierung
// von 0xA in 0xD gemacht.
// Die Umwandlung ist nur fuer Textdateien geeignet!!
void Channel::sendqueueSendFile( bool textfile )
{
   char *sendstr;
   int len,i;
   bool show = sendqueue->show;
   int comp=sendqueue->comp;
   bool abin;

   abin = sendqueue->abin;

   if (sendqueue->fd == -1)
   {
      // Datei oeffnen
      if ((sendqueue->fd = open(sendqueue->data, O_RDONLY)) == -1)
      {
         deleteQueueEntry();
         return;
      }

      // Gucken, ob wir eine AutoBIN-Datei senden sollen.
      if (abin)
         // Jau, wollen wir. Fenster oeffnen
         bintx->createTransferWin();
   }

   sendstr = (char *) malloc(framesize+1);

   // Ein Frame aus der Datei holen und aussenden.
   if ((len = read(sendqueue->fd, sendstr, framesize)) < framesize)
   {
      // Die Datei ist zu Ende.

      // Gucken, ob wir am Ende einer AutoBIN-Datei angekommen sind.
      if (abin)
      {
         bintx->showEndText();
         kontextmenu->changeItem(klocale->translate("Send &AutoBIN"), mnu_sendabin);
         delete bintx;
         bintx = NULL;
         flags &= ~CH_AUTOBINTX;
         abin = false;
      }

      // Schliessen und naechsten Eintrag in der Queue holen.
      ::close(sendqueue->fd);
      deleteQueueEntry();
   }

   if (len > 0)
   {
      // Konvertierung der Zeilen-End-Markierungen
      if (textfile)
         for (i=0;i<len;i++)
           if (sendstr[i] == '\xA') sendstr[i] = '\xD';

      if (abin)
         bintx->updateTransferWin( len );

      // Sendet den Inhalt von sendstr aus.
      sendFrame( sendstr, len, comp );

      // Hier wird geguckt, ob der zu sendende Text angezeigt werden soll
      if (show)
      {
         sendstr[len] = '\0';
         outText( sendstr, len, config->colors->txText );
      }
   }

   free(sendstr);
}


void Channel::sendqueueDidadit()
{
   char tmp[MAXROHBLOCKSIZE];
   int len;

   if (!sendqueue->ready)
      if (sendqueue->data == NULL)
      {
         // Das naechste DIDADIT-Paket holen
         sendqueue->ready = didadit->getNextPacket( tmp, len );
         if (len == 0)
         {
            deleteQueueEntry();
            return;
         }
         sendqueue->data = (char *) malloc( len );
         memcpy(sendqueue->data, tmp, len);
         sendqueue->len = len;
      }


   // Das DIDADIT-Paket soll ausgesendet werden
   len = framesize;
   if (sendqueue->len > len)
   {
      memcpy(tmp, sendqueue->data, len);
      sendqueue->len -= len;
      memmove(sendqueue->data, sendqueue->data+len, sendqueue->len);
   }
   else
   {
      // Dieses Frame wurde komplett ausgesendet
      memcpy(tmp, sendqueue->data, sendqueue->len);
      len = sendqueue->len;

      free(sendqueue->data);
      sendqueue->data = NULL;
      sendqueue->len = 0;
   }

   // Unkomprimiert aussenden
   sendFrame( tmp, len, COMP_NO );

   if (sendqueue->ready && sendqueue->data == NULL)
      deleteQueueEntry();
}



void Channel::sendqueueYAPP()
{
   char tmp[2000], tmp2[2000];
   int len;


   if (!sendqueue->ready)
      if (sendqueue->len < framesize)
      {
         // Das naechste YAPP-Paket holen
         sendqueue->ready = yapp->getNextPacket( tmp, len );
         if (len == 0)
         {
            deleteQueueEntry();
            return;
         }
         if (sendqueue->data == NULL)
         {
            sendqueue->data = (char *) malloc( len );
            memcpy(sendqueue->data, tmp, len);
            sendqueue->len = len;
         }
         else
         {
            memcpy( tmp2, sendqueue->data, sendqueue->len );
            free(sendqueue->data);
            sendqueue->data = (char *) malloc(len+sendqueue->len);
            memcpy(sendqueue->data, tmp2, sendqueue->len);
            memcpy(sendqueue->data+sendqueue->len, tmp, len);
            sendqueue->len += len;
         }
      }


   // Das YAPP-Paket soll ausgesendet werden
   len = framesize;
   if (sendqueue->len > len)
   {
      memcpy(tmp, sendqueue->data, len);
      sendqueue->len -= len;
      memmove(sendqueue->data, sendqueue->data+len, sendqueue->len);
   }
   else
   {
      // Dieses Frame wurde komplett ausgesendet
      memcpy(tmp, sendqueue->data, sendqueue->len);
      len = sendqueue->len;

      free(sendqueue->data);
      sendqueue->data = NULL;
      sendqueue->len = 0;
   }

   // Unkomprimiert aussenden
   sendFrame( tmp, len, COMP_NO );

   if (sendqueue->ready && sendqueue->data == NULL)
      deleteQueueEntry();
}



//  void Channel::send_a_frame()
// Diese Funktion wird regelmaessig aufgerufen. Es wird
// geguckt, ob etwas zum Senden vorliegt.
//
// Rueckgabe: true, wenn ein Frame gesendet wurde, false wenn nicht.
void Channel::send_a_frame()
{
   switch (sendqueue->type)
   {
      case SENDQ_TEXT:
           sendqueueSendTextLine();
           break;
      case SENDQ_FILE:
           sendqueueSendFile(false);
           break;
      case SENDQ_TEXTFILE:
           sendqueueSendFile(true);
           break;
      case SENDQ_DIDADIT:
           sendqueueDidadit();
           break;
      case SENDQ_YAPP:
           sendqueueYAPP();
           break;
   }
}


void Channel::slotTimer()
{
   int justsent=0;

   while ((squeue < config->maxframe) && (sendqueue != NULL) && (justsent < config->maxframe))
   {
      justsent++;
      send_a_frame();
   }

   timer->start(250, true);
}


void Channel::sendFrame( char *str, int len, int comp )
{
   int count,newlen=0;
   bool is_huff;
   char buffer[FRAMESIZE+1];

   bytesTX += len;

   // Wenn dieses Frame Huffman gepackt sein soll, wird hier gewandelt.
   is_huff = false;
   if (comp != COMP_NO)
   {
      if (((flags & CH_COMP) > 0) || (comp == COMP_SP))
      {
         is_huff = true;
         if ((newlen = comp_stat_huff( str, len, buffer )) == 0)
            is_huff = false;
      }
      if (((flags & CH_HUF_TX) > 0) || (comp == COMP_HUF))
      {
         is_huff = true;
         if ((newlen = ahuf->Komprimieren( true, buffer, str, len )) == 0)
            is_huff = false;
      }
   }

   if (is_huff)
   {
      if ((count = write(txfd, buffer, newlen)) != newlen) send_error = true;
   }
   else
   {
      if ((count = write(txfd, str, len)) != len) send_error = true;
   }
}


//   void Channel::rxFrame(int socket)
//
// Diese Funktion wird aufgerufen, wenn von unserer Verbindung Daten
// gekommen sind.
void Channel::rxFrame(int socket)
{
   char tmp[1000], output[1000];
   int len, tmplen;

   len = read(socket, tmp, 1000);

   if (len > 0)
   {
      if ((flags & CH_COMP) > 0)
      {
         if ((tmplen = decomp_stat_huff( tmp, len, output )) > 0)
         {
            len = tmplen;
            memcpy(tmp, output, len);
            tmp[len] = '\0';
         }
      }

      if ((flags & CH_HUF_RX) > 0)
      {
         if ((tmplen = ahuf->DeKomprimieren( output, tmp, len )) > 0)
         {
            len = tmplen;
            memcpy(tmp, output, len);
            tmp[len] = '\0';
         }
      }

      bytesRX += len;
      work_with_frame(tmp,len);
      return;
   }
}


// Loescht den ersten Eintrag der Queue
void Channel::deleteQueueEntry()
{
   s_sendqueue *tmp;

   if (sendqueue->data != NULL)
      free(sendqueue->data);

   if (sendqueue->fd != -1) ::close(sendqueue->fd);

   tmp = sendqueue;
   sendqueue = sendqueue->next;
   free( tmp );
}


void Channel::checkPi1chlConnect(char *data)
{
   int i = POS(':', data);
   int j = POS('}', data);
   char tmp[100];
   int len;


   if (i == -1 || j == -1) return;
   if (i > j) return;

   if (strstr(data, "} Connected to ") == NULL) return;


   len = strlen(data)-j-15;
   memcpy(tmp, data+j+15, len);
   tmp[len] = '\0';

   if ((i = POS('\r', tmp)) > -1) tmp[i] = '\0';
   checkConnectedTo(tmp);
}


void Channel::checkTNNConnect(char *data)
{
   int i = POS(':', data);
   int j = POS('>', data);
   char tmp[100];
   int len;


   if (i == -1 || j == -1) return;
   if (i > j) return;

   if (strstr(data, "> Connected to ") == NULL) return;


   len = strlen(data)-j-15;
   memcpy(tmp, data+j+15, len);
   tmp[len] = '\0';

   if ((i = POS('\r', tmp)) > -1) tmp[i] = '\0';
   checkConnectedTo(tmp);
}


void Channel::checkTNNReconnect(char *data)
{
   int i = POS(':', data);
   int j = POS('>', data);
   char tmp[100];
   int len;

   if (i == -1 || j == -1) return;
   if (i > j) return;

   if (strstr(data, "> Reconnected to ") == NULL) return;


   len = strlen(data)-j-17;
   memcpy(tmp, data+j+17, len);
   tmp[len] = '\0';

   checkReconnectedTo(tmp);
}


void Channel::work_with_frame(char *neu_str, int neu_str_len)
{
   char *tmp, *data, *str, tmp2[200];
   int datalen, i, str_len;
   bool linemode;


   str = (char *) malloc(neu_str_len+1);
   memcpy(str,neu_str,neu_str_len);
   str_len = neu_str_len;
   str[str_len] = '\0';


  // Gucken, ob in save_text noch was gespeichert ist. Wenn ja, werden
  // diese Daten vor die empfangenen Daten gepackt.
  if (save_text_len != 0)
  {
    tmp = (char *) malloc( str_len+save_text_len+1 );
    memcpy(tmp,save_text,save_text_len);
    memcpy(tmp+save_text_len,str,str_len);
    str_len = str_len + save_text_len;
    free(str);
    str = tmp;
    str[str_len] = '\0';
    free(save_text);
    save_text = NULL;
    save_text_len = 0;
  }


  linemode = (flags & CH_LINEMODE) == CH_LINEMODE;
  while ((str_len > 0) && (((i = POS('\xD',str)) != -1) || (!linemode)))
  {
    // linemode oder nicht?
    if (!linemode)
    {
      data = str;
      str = NULL;
      datalen = str_len;
      str_len = 0;
    }
    else
    {
      data = (char *) malloc(i+2);
      memcpy(data,str,i+1);
      data[i+1] = '\0';
      datalen = i+1;

      str_len = str_len - i-1;
      memmove(str,str+i+1,str_len);
      str[str_len] = '\0';
    }


    // Do something with data and datalen

    // Haben wir jemanden connected?
    if ((userinfo->getFlags() & CH_CHECKCONN) == CH_CHECKCONN)
    {
       if (!strncmp(data,"*** connected to",16) ||
           !strncmp(data,"*** Connected to",16))
       {
         COPY(tmp2,data,17,strlen(data)-17);
         KillSpacesLeft(tmp2);
         KillSpacesRight(tmp2);
         Gross(tmp2);
         i = strlen(tmp2)-1;
         if (tmp2[i] == '\r') tmp2[i] = '\0';

         checkConnectedTo(tmp2);
       }

       // PE1CHL-Digis
       checkPi1chlConnect(data);

       // TNN-Digis
       checkTNNConnect(data);
    }


    if ((userinfo->getFlags() & CH_CHECKDISC) == CH_CHECKDISC)
    {
       if (!strncmp(data,"*** reconnected to",18) ||
           !strncmp(data,"*** Reconnected to",18))
       {
         COPY(tmp2,data,19,strlen(data)-19);
         KillSpacesLeft(tmp2);
         KillSpacesRight(tmp2);
         Gross(tmp2);
         i = strlen(tmp2)-1;
         if (tmp2[i] == '\r') tmp2[i] = '\0';

         checkReconnectedTo(tmp2);
       }

       // TNN-Digis
       checkTNNReconnect(data);
    }


    ///////////////////////////////////////////////////////////////////////
    ////  Text auf dem Bildschirm ausgeben.                            ////
    if (datalen > len_allready_showed)
       {
         if (!binrx->isRxfile() &&
             !auto7->isRxfile() &&
             didadit == NULL &&
             yapp == NULL)
            if ((flags & CH_AUTOBINTX) != 0)
               outText( data+len_allready_showed+4, datalen-len_allready_showed-4, config->colors->rxText );
            else
               outText( data+len_allready_showed, datalen-len_allready_showed, config->colors->rxText );
         len_allready_showed = 0;
       }
    ////  Ab hier koennen Funktionen aufgerufen werden, die etwas      ////
    ////  aussenden                                                    ////
    ///////////////////////////////////////////////////////////////////////


    // Wenn dies eine Mailbox ist, wird geguckt, ob der BoxCheck aktiv
    // ist.
    if (userinfo->getType() == TYPE_MAILBOX)
       boxcheck->proceed(data, datalen);


    // Gucken, welche Software auf der Gegenseite laeuft, wenn dies noch
    // nicht erledigt wurde. (Maximal bis zur 20. Zeile.)
    if (typeCheckLine <= 20)
    {
       typeCheckLine++;
       if (!userinfo->getSwType())
          checkSwType(data);
    }


    // Remote-Befehle?
    if (((userinfo->getFlags() & CH_REMOTE) == CH_REMOTE) &&
        ((flags & CH_DIDADIT) != CH_DIDADIT) &&
        ((flags & CH_DIDADIT) == 0) &&
        ((flags & CH_YAPP) == 0) &&
         (!binrx->isRxfile()) &&
         (!auto7->isRxfile()))
       if (!strncmp(data,"//",2))
          remote->checkLine( data+2 );


    // Alles empfangene an die Passwort-Routinen weitergeben
    password->proceed(data, datalen);

    // auto7plus-Zeugs
    if ((((userinfo->getFlags() & CH_CHECK7PLUS) == CH_CHECK7PLUS) ||
         (auto7->isRxfile())) &&
         (!binrx->isRxfile()))            // Nur 7plus wenn kein AutoBIN und
       auto7->proceed( data, datalen );


    // Wird #HUF# angefordert?
    if ((userinfo->getFlags() & CH_HUFALLOW) != 0)
       checkForHUF(data, datalen);


    // DIDADIT-Empfang
    // Muss vor dem DIDADIT Autosave stehen
    if ((flags & CH_DIDADIT) == CH_DIDADIT)
    {
       switch (didadit->proceed(data, datalen))
       {
          case 0: // Die Uebertragung ist fertig
                  flags &= ~CH_DIDADIT;
                  flags |= CH_LINEMODE;
                  delete didadit;
                  didadit = NULL;
                  updateStatusBar("");
                  kontextmenu->changeItem( klocale->translate("Send D&IDADIT"), mnu_didadit );
                  break;
          case 1: // Abbruch, keine Didadit-Uebertragung
                  flags &= ~CH_DIDADIT;
                  flags |= CH_LINEMODE;
                  delete didadit;
                  didadit = NULL;
                  updateStatusBar("");
                  break;
          case 2: break;
       }
    }


    // AutoBIN RX
    if ((((userinfo->getFlags() & CH_CHECKAUTOBIN) == CH_CHECKAUTOBIN) ||
         (binrx->isRxfile())) &&
         (!auto7->isRxfile()) &&         // Nur AutoBIN, wenn kein 7plus
         ((flags & CH_DIDADIT) == 0))    // und kein DIDADIT
    {
       i = binrx->proceed( data, datalen );
       if (i != datalen && i != 0)
       {
          // Es wurden nicht alle Bytes "gebraucht". Die uebrigen
          // wieder an den str-String haengen
          tmp = (char *) malloc(datalen-i+str_len+1);
          memcpy(tmp,data+i,datalen-i);
          memcpy(tmp+datalen-i,str,str_len);
          str_len = str_len + datalen-i;
          tmp[str_len] = '\0';
          free(str);
          str = tmp;
       }
    }


    // DIDADIT - RX (Autosave)
    if (((userinfo->getFlags() & CH_CHECKDIDADIT) != 0) &&
         (!auto7->isRxfile()) &&          // Nur DIDADIT, wenn kein 7plus
         ((flags & CH_AUTOBINRX) == 0))  // und wenn kein AutoBIN empfangen wird.
    if (!strcmp(data,"#DIDADIT#\xD"))
    {
       didadit = new DIDADIT( this, "", false );
       flags |= CH_DIDADIT;
       flags &= ~CH_LINEMODE;
       kontextmenu->changeItem(klocale->translate("Close D&IDADIT-RX"), mnu_didadit);
       updateStatusBar("");
    }


    // Normale Textfiles speichern, wenn kein AutoBIN-File, kein auto7plus
    // und kein DIDADIT-File empfangen wird
    if (((flags & CH_SAVE) == CH_SAVE) &&  // Soll abgespeichert werden?
        ((flags & CH_AUTOBINTX) == 0) &&   // Kein AutoBIN-TX?
        ((flags & CH_DIDADIT) == 0) &&     // Kein DIDADIT-TX/RX?
        (!auto7->isRxfile()) &&            // Kein Auto7plus-RX?
        (!binrx->isRxfile()))              // Kein AutoBIN-RX?
    {
       write(filerx_fd, data, datalen-1);
       write(filerx_fd, "\n", 1);
    }


    if ((flags & CH_AUTOBINTX) == CH_AUTOBINTX)
    {
       switch (bintx->check_okno(data, datalen))
       {
          case 1: // Ok, File senden
                  sendFile( bintx->getFilename(), false, true );
                  break;
          case 2: // Nein! Abbruch!
                  flags &= ~CH_AUTOBINTX;
                  delete bintx;
                  bintx = NULL;
                  kontextmenu->changeItem(klocale->translate("Send &AutoBIN"), mnu_sendabin);
                  updateStatusBar("");
                  break;
       }
    }


    // YAPP-Transfer?
    if ((flags & CH_YAPP) == CH_YAPP)
    {
       if (yapp->proceed( data, datalen ))
       {
          delete yapp;
          yapp = NULL;
          flags |= CH_LINEMODE;
          flags &= ~CH_YAPP;
          kontextmenu->changeItem(klocale->translate("&YAPP-Transfer"), mnu_yapp);
          updateStatusBar("");
       }
    }


    linemode = (flags & CH_LINEMODE) == CH_LINEMODE;
    free(data);
    datalen = 0;
    data = NULL;
  }


  // Sind noch ein paar Daten in str enthalten? Wenn ja werden die jetzt nach
  // save_text gespeichert
  if (str_len > 0)
  {
    save_text = (char *) malloc(str_len);
    memcpy(save_text,str,str_len);
    save_text_len = str_len;

    free(str);
    str_len = 0;
    str = NULL;

    // Text ausgeben
    if (!auto7->isRxfile() &&
        !binrx->isRxfile())
          outText( save_text+len_allready_showed, save_text_len-len_allready_showed, config->colors->rxText );
    len_allready_showed = save_text_len;
  }
}


void Channel::sendCText()
{
   char tmp2[500],tmp[1000],tmp3[500],topline[50];
   int i;


   // Existiert im userdb-Verzeichnis ein File namens <call>.ctext? Wenn ja wird dieses als
   // ctext ausgesendet.
   COPY(tmp2,data_call,0,3);
   Klein(tmp2);
   strcpy(tmp3,data_call);
   Klein(tmp3);
   if ((i = POS('-', tmp3)) > -1)
      tmp3[i] = '\0';
   sprintf(tmp,"%s/userdb/%s.../%s.ctext",config->maindir,tmp2,tmp3);
   if (!file_exist(tmp)) sprintf(tmp,"%s/ctext",config->maindir);

   // Identifikation der Software senden. Wird nicht angezeigt!
   if (userinfo->getName()[0] == '\0')
      sprintf(topline,"{LinKT-%s-CDH?}\xD", LINKT_VERSION);
   else
      sprintf(topline,"{LinKT-%s-CDH}\xD", LINKT_VERSION);
   sendString(strlen(topline), topline, false);

   sendMacrofile( tmp );

   if (config->away.wo != 0)
   {
      switch (config->away.wo)
      {
         case 1: sprintf(tmp2, "## SysOp is away: %s ##\r", config->away.phone); break;
         case 2: sprintf(tmp2, "## SysOp is away: %s ##\r", config->away.bed); break;
         case 3: sprintf(tmp2, "## SysOp is away: %s ##\r", config->away.text); break;
      }
      macroLine(tmp, tmp2);
      sendString(tmp);
   }
}


void Channel::returnPressedGot()
{
   char text[500],zeichen;
   int len,src,dest;
   int line, col;

   vorschreib->cursorPosition(&line, &col);
   strcpy(text,vorschreib->textLine(line));

   KillSpacesRight(text);
   len = strlen(text);

   text[len] = '\xD';
   text[len+1] = '\0';

   // ^A bis ^Z in ASCII(0) bis ASCII(25) umwandeln
   src = 0;
   dest = 0;
   while (text[src] != '\0')
   {
      if (text[src] == '^')
      {
         src++;
         if (text[src] == '\0')
            text[dest] = text[src];
         else
         {
            zeichen = toupper(text[src]);
            if (zeichen >= 'A' && zeichen <= 'Z')
            {
               text[dest] = zeichen-'A'+1;
            }
            else
            {
               if (text[src] == '^')
                  text[dest] = '^';
            }
         }
      }
      else text[dest] = text[src];
      src++;
      dest++;
   }
   text[dest] = '\0';

   if (binrx->isRxfile() || ((flags & CH_AUTOBINTX) != 0))
   {
      if (config->localEcho)
         insertBINChatLine( text, true );
      else
         insertBINChatLine( text, false );
      return;
   }

   if (config->localEcho)
      sendString( strlen(text), text, true );
   else
      sendString( strlen(text), text, false );
}


//  char * Channel::getCall()
// Das Rufzeichen der Gegenstation auf diesem Kanal zurueckgeben
char * Channel::getCall()
{
   return data_call;
}


//  char * Channel::getMycall()
// Das eigene Rufzeichen auf diesem Kanal zurueckgeben
char * Channel::getMycall()
{
   return data_mycall;
}


char * Channel::getPort()
{
   return data_port;
}


void Channel::setUnack(int value)
{
   char tmp[10];

   sprintf(tmp,"UnAck: %i",value);
   statusbar->changeItem(tmp, ID_STATUS_UNACK);
}


void Channel::setSQueue(int value)
{
   char tmp[10];

   sprintf(tmp,"UnSent: %i",value);
   statusbar->changeItem(tmp, ID_STATUS_UNSENT);
   squeue = value;
}


void Channel::setTries(int value)
{
   char tmp[10];

   sprintf(tmp,"Retry: %i",value);
   statusbar->changeItem(tmp, ID_STATUS_RETRY);
}


void Channel::setStatus(int value)
{
   statusbar->changeItem(TNC_Status[value], ID_STATUS_STATE);

   // Haben wir connected?
   if ((old_status == 1) && (value == 4))
   {
      data_logintime = time(NULL);

      // eine connect-Meldung aussenden
      showConnectMessage(data_port, data_call, data_digis);

      timer = new QTimer(this);
      connect(timer, SIGNAL(timeout()), SLOT(slotTimer()));
      timer->start(250, true);

      // QSO-Liste im Haupfenster
      if (!isActiveWindow())
         toplevel->chanListTable->setNew( this, true );
      toplevel->chanListTable->setMode( this, MODE_INFO );

      // Wenn das Passwort fuer diese Station automatisch beantwortet
      // werden soll, wird dies die ersten 20 Zeilen lang gemacht.
      if ((userinfo->getFlags() & CH_AUTOSEND_PW) != 0)
         password->activatePassword();
   }

   // Verbindungsabbau
   if ((old_status > 1) && (value == 0))
   {
      if (!isActiveWindow())
         toplevel->chanListTable->setNew( this, true );

      // Wenn noch ein Transferfenster offen ist, wirds jetzt auch
      // geschlossen
      auto7->closeTransferwin();

      // Kein Text darf mehr markiert sein
      output->clearAllMarks();

      showDisconnectMessage(data_port, data_call, data_digis);

      // Bimmeln
      toplevel->playSound( SOUND_DISCONNECT );

      delete timer;
      timer = NULL;
      delete sockrx;
      sockrx = NULL;
      flags &= ~CH_DISC;
      toplevel->deleteChannelList( this );
      old_status = value;

      // Eventuell vorhanden Transfer-Fenster schliessen
      if (auto7 != NULL)
      {
         delete auto7;
         auto7 = NULL;
      }
      if (binrx != NULL)
      {
         delete binrx;
         binrx = NULL;
      }
      if (bintx != NULL)
      {
         delete bintx;
         bintx = NULL;
      }

      if (config->closeWinOnDisc) close();
      if (txfd != -1)
      {
         ::close(txfd);
         txfd = -1;
      }
      // Die QSO-Liste im Hauptfenster
      toplevel->chanListTable->setMode( this, MODE_DISC );
      logBookEntry();
   }

   // Wenn wir im IXF-Status sind und CH_DISC aktiv ist, wird disconnected
   if (value == 4)
      if ((flags & CH_DISC) != 0)
      {
         reallyDisconnect();
         timer->stop();
      }

   old_status = value;
}


void Channel::closeEvent(QCloseEvent *e)
{
   char text[100];

   if (old_status != 0)
   {
      sprintf(text, klocale->translate("Do you really want to disconnect from %s?"), data_call);
      if (KMsgBox::yesNo(NULL, klocale->translate("Disconnect?"), text, 0, klocale->translate("&Yes"), klocale->translate("&No")) != 1)
      {
         e->ignore();
         return;
      }

      // Sachen erledigen, die nur bei noch bestehenden Verbindungen erledigt
      // werden sollten.
      discStn();
      closeWinDisc = true;
      logBookEntry();
   }

   if (sendqueue == NULL)
      closeQSOWindow();
}


//  void Channel::newConnect()
// Baut eine neue Verbindung auf.
void Channel::newConnect()
{
   toplevel->slotConnect();
}


//  void Channel::discStation()
// Diese Verbindung soll disconnected werden.
void Channel::discStation()
{
   close();
}


//   void Channel::discStn()
// Die Gegenstation soll Disconnected werden.
void Channel::discStn()
{
   if ((flags & CH_COMP) > 0)
      sendString("//COMP 0\r");

   if ((flags & CH_HUF_TX) > 0)
      sendString("#HUF:NO#\r");

   reallyDisconnect();
}


//   void Channel::discStnAuto()
//
// Diese Verbindung wird von der Gegenstation oder von LinKT selbst beendet.
void Channel::discStnAuto()
{
   if ((flags & CH_COMP) > 0)
      sendString("//COMP 0\r");

   if ((flags & CH_HUF_TX) > 0)
      sendString("#HUF:NO#\r");

   if (sendqueue == NULL)
      reallyDisconnect();
   else
      flags |= CH_DISC;
}


void Channel::reallyDisconnect()
{
   if (txfd != -1)
   {
      ::close(txfd);
      txfd = -1;
   }

   while (sendqueue != NULL)
      deleteQueueEntry();
}


void Channel::closeQSOWindow()
{
    toplevel->deleteChannelList( this );
    if (toplevel->currentChannel == this)
       toplevel->currentChannel = NULL;
    delete this;
}


void Channel::settings()
{
  QDialog *dlg = new SettingsDlg(this);

  QPoint point = this->mapToGlobal (QPoint (0,0));

  QRect pos = this->geometry();
  dlg->setGeometry(point.x() + pos.width()/2  - dlg->width()/2,
		   point.y() + pos.height()/2 - dlg->height()/2, 
		   dlg->width(),dlg->height());

  dlg->exec();

  toplevel->chanListTable->changeType( this, userinfo->getType() );

  delete dlg;
}


void Channel::connCallAdd(char *call)
{
  s_conncalls *tmp;
  
  if (conncalls == NULL)
  {
    conncalls = (s_conncalls *) malloc(sizeof(s_conncalls));
    tmp = conncalls;
  }
  else
  {
    tmp = conncalls;
    while (tmp->next != NULL) tmp = tmp->next;
    
    tmp->next = (s_conncalls *) malloc(sizeof(s_conncalls));
    tmp = tmp->next;
  }

  tmp->next = NULL;
  strcpy(tmp->call,call);
}


//   bool Channel::connCallCheck(char *call)
// Guckt, ob dieses Rufzeichen in der Liste enthalten ist. Wenn ja werden
// alle Calls ab diesem Rufzeichen aus der Liste geloescht.
bool Channel::connCallCheck(char *call)
{
  s_conncalls *tmp,*last;

  // Gucken, ob wir zum allerersten Call zurueck fallen
  if (!strcmp(call,data_firstcall))
  {
    // Jau - alle Eintraege in conncalls loeschen
    tmp = conncalls;

    while (tmp != NULL)
    {
      last = tmp;
      tmp = tmp->next;
      free(last);
    }
    conncalls = NULL;
    
    return true;
  }


  tmp = conncalls;
  last = NULL;

  while (tmp != NULL)
  {
    if (!strcmp(call,tmp->call))
    {
      // Call gefunden. Alles ab diesem (incl.) free'en
      if (last != NULL)
         last->next = NULL;
      else
         conncalls = NULL;
      while (tmp != NULL)
      {
        last = tmp;
        tmp = tmp->next;
        free(last);
      }
      
      return true;
    }
    last = tmp;
    tmp = tmp->next;
  }
  
  return false;
}


bool Channel::validCallsign(char */*call*/)
{
   return true;
}


int Channel::getFlags()
{
   return flags;
}


QWidget * Channel::getVorschreibPtr()
{
   return vorschreib;
}


//   void Channel::sendFile()
// Ein File soll ausgesendet werden.
void Channel::slotSendFile()
{
   char tmp[500];

   if ((flags & CH_AUTOBINTX) != 0)
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("Cannot send a files while sending an AutoBIN-file."));
      return;
   }

   strcpy(tmp, config->dirUp);
   QString f = QFileDialog::getOpenFileName(tmp);

   // Cancel
   if (f.isEmpty()) return;

   strcpy(tmp, (const char *)f);

   if (!file_exist(tmp))
   {
      KMsgBox::message(NULL, klocale->translate("File not found"),
                             klocale->translate("Cannot find specified file."));
      return;
   }

   // Datei aussenden und auf dem Bildschirm anzeigen.
   sendTextFile( tmp, true );
}


void Channel::slotSend7plus()
{
   send7plus *dlg = new send7plus();
   char tmp[1000], fname[1000];
   int i,filecount,startnr;
   char infotext[100];

   if ((flags & CH_AUTOBINTX) != 0)
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("Cannot send 7plus-files while sending an AutoBIN-file."));
      return;
   }

   QPoint point = this->mapToGlobal (QPoint (0,0));

   QRect pos = this->geometry();
   dlg->setGeometry(point.x() + pos.width()/2  - dlg->width()/2,
 		    point.y() + pos.height()/2 - dlg->height()/2,
 		    dlg->width(),dlg->height());

   if (dlg->exec() == QDialog::Accepted)
   {

      sprintf(infotext, "\r\r** LinKT %s: 7plus-autosend **\r\r", LINKT_VERSION);

      filecount = dlg->getFileCount();
      startnr = dlg->startWithFile();
      strcpy(fname, dlg->getFilename());

      COPY(tmp, fname, 0, lPOS('.',fname));

      if (filecount == 1)
      {
         sprintf(fname, "%s.7pl", tmp);
         sendString( dlg->getSendLine(1) );
         sendTextFile( fname, true );
         sendString( infotext );
         sendString( "***END\r" );
      }
      else
      {
         for (i=startnr; i<=filecount; i++)
         {
            sprintf(fname, "%s.p%02x", tmp, i);
            sendString( dlg->getSendLine(i) );
            sendTextFile( fname, true );
            sendString( infotext );
            sendString( "***END\r" );
         }
      }
   }

   delete dlg;
}


void Channel::showKontextMenu()
{
   kontextmenu->popup(QCursor::pos());
}


void Channel::slotSaveText()
{
   char tmp[500];

   if ((flags & CH_SAVE) == CH_SAVE)
   {
      ::close(filerx_fd);
      flags &= ~CH_SAVE;
      kontextmenu->changeItem(klocale->translate("Save &Textfile"), mnu_textfile);
      updateStatusBar("");
   }
   else
   {
      strcpy(tmp, config->dirDown);
      QString f = QFileDialog::getSaveFileName(tmp);

      strcpy(tmp, (const char *)f);

      // Cancel
      if (f.isEmpty()) return;

      if (file_exist(tmp))
      {
         if (KMsgBox::yesNo(NULL, klocale->translate("Overwrite?"),
                                  klocale->translate("The specified file allready exists. Overwrite?"),
                                  0,
                                  klocale->translate("&Yes"),
                                  klocale->translate("&No")) != 1)
            return;
      }

      if ((filerx_fd = open(tmp, O_WRONLY|O_CREAT)) == -1)
      {
         printf("%s\n",strerror(errno));
         KMsgBox::message(NULL, klocale->translate("Failure"),
                                klocale->translate("Cannot create specified file!"));
         return;
      }

      // Zugriffsrechte einstellen
      fchmod(filerx_fd,S_IRUSR|S_IWUSR);

      flags |= CH_SAVE;
      kontextmenu->changeItem(klocale->translate("Close &Textfile-Save"), mnu_textfile);
      updateStatusBar(klocale->translate("Textfile save"));
   }
}


//   void Channel::slotSendAbinFile()
// Ein File soll ausgesendet werden.
void Channel::slotSendAbinFile()
{
   char tmp[500];
   int fd;


   if ((flags & CH_AUTOBINTX) != 0)
   {
      // Upload stoppen
      abortABinTX();
      return;
   }

   strcpy(tmp, config->dirUp);
   QString f = QFileDialog::getOpenFileName(tmp);

   strcpy(tmp, (const char *)f);

   // Cancel
   if (f.isEmpty()) return;

   if ((fd = open(tmp, O_RDONLY)) == -1)
   {
      KMsgBox::message(NULL, klocale->translate("File not found"),
                             klocale->translate("Cannot find specified file."));
      return;
   }
   ::close(fd);

   // #BIN#-Zeile abschicken
   bintx = new BinTX(this, tmp);
   if (bintx->bin_state == 1)
   {
      KMsgBox::message(NULL, klocale->translate("Cannot open file"),
                             klocale->translate("Cannot open specified file."));
      delete bintx;
      bintx = NULL;
      return;
   }
   flags |= CH_AUTOBINTX;
   kontextmenu->changeItem(klocale->translate("Abort &AutoBIN-TX"), mnu_sendabin);
   updateStatusBar(klocale->translate("AutoBIN-Transmission"));
}


//   void Channel::slotSendDIDADIT()
//
// Initiiert die Sendung eines DIDADIT-Files
void Channel::slotSendDIDADIT()
{
   char tmp[500];
   int fd;


   // Abbruch der Uebertragung
   if ((flags & CH_DIDADIT) != 0)
   {
      abortDIDADIT();
      return;
   }


   if ((flags & CH_AUTOBINTX) != 0)
   {
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("Cannot send DIDADIT-files while sending an AutoBIN-file."));
      return;
   }

   strcpy(tmp, config->dirUp);
   QString f = QFileDialog::getOpenFileName(tmp);

   strcpy(tmp, (const char *)f);

   // Cancel
   if (f.isEmpty()) return;

   if ((fd = open(tmp, O_RDONLY)) == -1)
   {
      KMsgBox::message(NULL, klocale->translate("File not found"),
                             klocale->translate("Cannot find specified file."));
      return;
   }
   ::close(fd);


   sendString("#DIDADIT#\xD");
   didadit = new DIDADIT( this, tmp, true );
   flags |= CH_DIDADIT;
   kontextmenu->changeItem(klocale->translate("Close D&IDADIT-TX"), mnu_didadit);
   updateStatusBar(klocale->translate("DIDADIT-Transmission"));
}


void Channel::abortDIDADIT()
{
   char tmp[500];

   // Wenn im Augenblick eine DIDADIT-Sendung ansteht wird sie abgebrochen
   if (sendqueue != NULL)
      if (sendqueue->type == SENDQ_DIDADIT)
         deleteQueueEntry();
   didadit->abortIt();
   delete didadit;
   didadit = NULL;

   flags &= ~CH_DIDADIT;
   flags |= CH_LINEMODE;

   kontextmenu->changeItem( klocale->translate("Send D&IDADIT"), mnu_didadit );

   sprintf(tmp, "\xD<LinKT>: DIDADIT-Transfer aborted.\xD\xD");
   if ((userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      sendString( tmp );
   else
      outText( tmp, strlen(tmp), config->colors->txText );
   updateStatusBar("");
}


void Channel::slotCTRLKeyUp()
{
   toplevel->chanListTable->switchChannel(SWITCH_UP);
}


void Channel::slotCTRLKeyDown()
{
   toplevel->chanListTable->switchChannel(SWITCH_DOWN);
}


int Channel::getTxfd()
{
   return txfd;
}


//   void Channel::updateFontSizes(int qsosize, int txsize)
// Aendert die aktuelle Schriftgroesse auf diesem Kanal und setzt die
// Groesse des
void Channel::updateFont()
{
   QFont f(config->rxfontstr, config->qsofontsize);
   QFont vorschreib_f(config->txfontstr, config->txfontsize);
   int screenwidth;

   if (config->colors->qsowin_bold)
      f.setBold(true);
   output->setFont( f );
   if (config->colors->txwin_bold)
      vorschreib_f.setBold(true);
   vorschreib->setFont ( vorschreib_f );

   // Die Spaltenbreite auf 80 * die Breite des Zeichens 'm' einstellen
   QFontMetrics fm(f);
   screenwidth = 84 * fm.width("m");
   resize(screenwidth, (int)(screenwidth * 0.6));
}


void Channel::slotSendPW()
{
   password->transmit();
}


// Gucken, ob wir zu diesem Rufzeichen connecten koennen
void Channel::checkConnectedTo(char *call)
{
   char tmp[200];
   int i,len;

   strcpy(tmp, call);
   if ((i = POS(':', tmp)) > -1)
   {
      len = strlen(tmp)-i-1;
      memmove(tmp, tmp+i+1,len);
      tmp[len] = '\0';
   }

   if (validCallsign(tmp))
   {
      free(data_call);
      data_call = (char *) strdup(tmp);

      delete userinfo;
      userinfo = new UserInfo( data_call );
      boxcheck->readData( userinfo->getBoxCheck() );
      kontextmenu->setItemEnabled( mnu_boxcheck, false );

      // Fenster-Ueberschrift
      sprintf(tmp,"LinKT: %s <-> %s (port: %s)", data_mycall, data_call, data_port);
      setCaption( tmp );

      // Im Hauptfenster Rufzeichen und Typen updaten
      toplevel->chanListTable->changeCallAndType( this, data_call, userinfo->getType() );

      // Wenn das Passwort fuer diese Station automatisch beantwortet
      // werden soll, wird dies die ersten 20 Zeilen lang gemacht.
      if ((userinfo->getFlags() & CH_AUTOSEND_PW) != 0)
         password->activatePassword();

      connCallAdd( data_call );
      typeCheckLine = 0;
   }
}


// Gucken, ob wir zu diesem Rufzeichen reconnected werden koennen
void Channel::checkReconnectedTo(char *call)
{
   char tmp[200];
   int i,len;

   strcpy(tmp, call);
   if ((i = POS(':', tmp)) > -1)
   {
      len = strlen(tmp)-i-1;
      memmove(tmp, tmp+i+1,len);
      tmp[len] = '\0';
   }


   if (!validCallsign(tmp)) return;

   if (connCallCheck(tmp))
   {
      free(data_call);
      data_call = (char *) strdup(tmp);

      delete userinfo;
      userinfo = new UserInfo( data_call );
      boxcheck->readData( userinfo->getBoxCheck() );
      kontextmenu->setItemEnabled( mnu_boxcheck, false );

      // Fenster-Ueberschrift
      sprintf(tmp,"LinKT: %s <-> %s (port: %s)",data_mycall, data_call, data_port);
      setCaption( tmp );

      // Im Hauptfenster Rufzeichen und Typen updaten
      toplevel->chanListTable->changeCallAndType( this, data_call, userinfo->getType() );
   }
}


void Channel::checkSwType(char *line)
{
   int i;


   for (i=0;i<SW_ANZ;i++)
   {
      switch (swType[i].id)
      {
         case SW_FLEXNET:
              if (!strncmp(line, "PC/FlexNet V", 12) ||
                  !strncmp(line, "RMNC/FlexNet V", 14))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_FLEXNET );
                 userinfo->setType( TYPE_DIGI );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_DIGI );
                 userinfo->setRemotes( REMOTE_STD_DIGI );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_DIGI );
                 return;
              }
              break;
         case SW_AWZNODE:
              if (!strncmp(line, "AWZNode v", 9))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_AWZNODE );
                 userinfo->setType( TYPE_DIGI );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_DIGI );
                 userinfo->setRemotes( REMOTE_STD_DIGI );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_DIGI );
                 return;
              }
              break;
         case SW_TNN:
              if (!strncmp(line, "TNN V", 5))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_TNN );
                 userinfo->setType( TYPE_DIGI );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_DIGI );
                 userinfo->setRemotes( REMOTE_STD_DIGI );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_DIGI );
                 return;
              }
              break;
         case SW_BAYBOX:
              if (!strncmp(line, "BayCom-Mailbox V", 16))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_BAYBOX );
                 userinfo->setType( TYPE_MAILBOX );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_MAILBOX );
                 userinfo->setRemotes( REMOTE_STD_MAILBOX );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_MAILBOX );
                 return;
              }
              break;
         case SW_FBB:
              if (!strncmp(line, "[FBB-", 5))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_FBB );
                 userinfo->setType( TYPE_MAILBOX );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_MAILBOX );
                 userinfo->setRemotes( REMOTE_STD_MAILBOX );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_MAILBOX );
                 return;
              }
              break;
         case SW_DPBOX:
              if (!strncmp(line, "[DP-", 4))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_DPBOX );
                 userinfo->setType( TYPE_MAILBOX );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_MAILBOX );
                 userinfo->setRemotes( REMOTE_STD_MAILBOX );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_MAILBOX );
                 return;
              }
              break;
         case SW_LINKT:
              if (!strncmp(line, "{LinKT-", 7) ||
                  !strncmp(line, "[LinKT#", 7))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_LINKT );
                 userinfo->setType( TYPE_TERMINAL );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_TERMINAL );
                 userinfo->setRemotes( REMOTE_STD_TERMINAL );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_TERMINAL );
                 return;
              }
              break;
         case SW_GP85:
              if (!strncmp(line, "[GP85#", 6))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_GP85 );
                 userinfo->setType( TYPE_TERMINAL );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_TERMINAL );
                 userinfo->setRemotes( REMOTE_STD_TERMINAL );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_TERMINAL );
                 return;
              }
              break;
         case SW_WINGT:
              if (!strncmp(line, "*** WinGT ", 10))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_WINGT );
                 userinfo->setType( TYPE_TERMINAL );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_TERMINAL );
                 userinfo->setRemotes( REMOTE_STD_TERMINAL );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_TERMINAL );
                 return;
              }
              break;
         case SW_MCUT:
              if (!strncmp(line, "{MCUT-", 6))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_MCUT );
                 userinfo->setType( TYPE_TERMINAL );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_TERMINAL );
                 userinfo->setRemotes( REMOTE_STD_TERMINAL );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_TERMINAL );
                 return;
              }
              break;
         case SW_XPACKET:
              if (!strncmp(line, "[XPACKET-", 9))
              {
                 // Digi - entsprechende Flags und den Typen setzen
                 userinfo->autoSave = false;
                 userinfo->setSwType( SW_XPACKET );
                 userinfo->setType( TYPE_TERMINAL );
                 userinfo->setFlags( (userinfo->getFlags() & ~FLAGS_SAVE) | FLAGS_STD_TERMINAL );
                 userinfo->setRemotes( REMOTE_STD_TERMINAL );
                 userinfo->autoSave = true;
                 userinfo->writeData();
                 toplevel->chanListTable->changeType( this, TYPE_TERMINAL );
                 return;
              }
              break;
      }
   }
}


void Channel::slotAway()
{
   toplevel->slotAway();
}


void Channel::checkSIDFlags( char *line )
{
   int i,k,len;

   // Gucken, ob dies ein Header mit der Syntax {Software-Version-Flags}
   // ist. Wenn ja werden die Flags 'rausgesucht.
   if ((i = POS('{', line)) < 0) return;
   if ((k = POS('}', line)) < 0) return;
   if (i > k) return;
   while ((i = POS('-', line)) > -1)
   {
      len = strlen(line);
      memmove(line, line+i+1, len);
      line[len] = '\0';
   }
   if ((k = POS('}', line)) > -1)
      line[k] = '\0';

   // Sooo... Die Flags sind jetzt in line und werden durchsucht.
   for (i=0; line[i] != '\0'; i++)
   {
      switch (line[i])
      {
         case '?': // Unser Name ist der Gegenstation unbekannt.
                   break;
      }
   }
}

/*
//   void Channel::slotDropEvent( KDNDDropZone *zone )
//
// Wird aufgerufen, wenn etwas auf das Output-Fenster fallen gelassen wurde.
void Channel::slotDropEvent( KDNDDropZone *zone )
{
   printf("drop!\n");
}
*/


void Channel::logBookEntry()
{
   char tmp[500];
   time_t zeit;
   struct tm loti_now, loti_start;
   char mon[12][4]={"jan","feb","mar","apr","may","jun","jul","aug",
                    "sep","okt","nov","dez"};
   int fd;


   zeit = time(NULL);
   loti_now = *localtime(&zeit);
   loti_start = *localtime(&data_logintime);

   sprintf(tmp, "%s/log/%s%i.log", config->maindir, mon[loti_now.tm_mon], loti_now.tm_year+1900);

   if ((fd = open(tmp, O_WRONLY|O_APPEND)) == -1)
      if (errno == ENOENT)
         if ((fd = open(tmp, O_WRONLY|O_CREAT|O_APPEND)) == -1)
            return;
         else
         {
            // Zugriffsrechte einstellen
            fchmod(fd,S_IRUSR|S_IWUSR);

            strcpy(tmp, "\n    port         starttime               endtime         rxbytes   txbytes      mycall     connpath\n");
            write(fd, tmp, strlen(tmp));
            strcpy(tmp, "-------------------------------------------------------------------------------------------------------------\n");
            write(fd, tmp, strlen(tmp));
         }
      else
         return;

   sprintf(tmp, "%10s  %.2i.%.2i.%.4i %.2i:%.2i:%.2i    %.2i.%.2i.%.4i %.2i:%.2i:%.2i  %8li  %8li  %10s <-> %s %s\n",
                data_port,
                loti_start.tm_mday, loti_start.tm_mon+1, loti_start.tm_year+1900, loti_start.tm_hour, loti_start.tm_min, loti_start.tm_sec,
                loti_now.tm_mday, loti_now.tm_mon+1, loti_now.tm_year+1900, loti_now.tm_hour, loti_now.tm_min, loti_now.tm_sec,
                bytesRX, bytesTX, data_mycall, data_call, data_digis);
   write(fd, tmp, strlen(tmp));
   ::close(fd);
}


//   void Channel::setIfaceConnection()
//
// Markiert diese Verbindung als eine Verbindung, die von einem externen
// Programm aufgebaut wurde.
void Channel::setIfaceConnection(Interface *i)
{
   externalConnection = true;

   iface = i;
}


void Channel::activateCheckWin()
{
   boxcheck->showWindow();
}


void Channel::checkForHUF(char *data, int len)
{
   char tmp[100];
   int i;

   if (strncmp(data, "#HUF:", 5)) return;

   memcpy(tmp, data+5, len-5);
   tmp[len-5] = '\0';

   if ((i = POS('#', tmp)) > -1)
      tmp[i] = '\0';


   if (!strcmp(tmp, "ON"))
   {
      flags |= CH_HUF_TX|CH_HUF_RX;
      sendStringComp(9, "\r#HUF:1#\r", true, COMP_NO);
      return;
   }

   if (!strcmp(tmp, "OFF"))
   {
      flags &= ~(CH_HUF_TX|CH_HUF_RX);
      sendStringComp(9, "\r#HUF:0#\r", true, COMP_HUF);
      return;
   }

   if (!strcmp(tmp, "NO"))
   {
      flags &= ~(CH_HUF_TX|CH_HUF_RX);
      return;
   }

   if (!strcmp(tmp, "1"))
   {
      flags |= CH_HUF_RX;
      return;
   }

   if (!strcmp(tmp, "0"))
   {
      flags &= ~CH_HUF_RX;
      flags &= ~CH_HUF_TX;
      return;
   }
}


void Channel::slotHufComp()
{
   char tmp[] = "\r#HUF:OFF#\r";


   if (((flags & CH_HUF_RX) != 0) ||
       ((flags & CH_HUF_TX) != 0))
   {
      // #HUF# ist aktiviert - deaktivieren?
      if (KMsgBox::yesNo( NULL,
                          klocale->translate("#HUF#-Compression?"),
                          klocale->translate("Do you want to deactivate the #HUF#-online-compression?"),
                          0,
                          klocale->translate("&Yes"), klocale->translate("&No")) == 1)
      {
         // #HUF# deaktivieren
         sendStringComp(strlen(tmp), tmp, false, COMP_HUF);
         flags &= ~CH_HUF_TX;
         return;
      }
      return;
   }

   // #HUF# ist noch nicht aktiviert - aktivieren?
   if (KMsgBox::yesNo( NULL,
                       klocale->translate("#HUF#-Compression?"),
                       klocale->translate("Do you want to activate the #HUF#-online-compression?"),
                       0,
                       klocale->translate("&Yes"), klocale->translate("&No")) == 1)
   {
      // #HUF# aktivieren
      sendStringComp(10, "\r#HUF:ON#\r", true, COMP_NO);
      flags |= CH_HUF_TX;
      return;
   }
}


//   void Channel::getCookie(char *str)
//
// Holt ein Cookie und packt es nach str. Das File heisst "cookies"
// und wird zuerst unter .kde/share/apps/linkt/ und wenn da nicht
// vorhanden in $(KDEDIR)/share/apps/linkt/cookies gesucht.
void Channel::getCookie(char *str)
{
   FILE *f;
   char tmp[500];
   int count, use, len=0, i;
   bool usethis=false;


   strcpy(str, "Sorry, the cookies-file cannot be found.");
   str[0] = '\0';

   sprintf(tmp, "%s/cookies", config->maindir);
   if ((f = fopen(tmp,"r")) == NULL)
   {
      sprintf(tmp, "%s/linkt/cookies", (const char *)mykapp->kde_datadir());
      if ((f = fopen(tmp,"r")) == NULL) return;
   }

   count=-1;
   while (fgets(tmp, 499, f) != NULL)
      if (!strncmp(tmp, "@@", 2)) count++;

   use = 1+(int) ((float)count*rand()/(RAND_MAX+1.0));


   rewind(f);

   count = 0;
   while (fgets(tmp, 499, f) != NULL)
   {
      if (!strncmp(tmp, "@@", 2))
      {
         if (usethis)
         {
            fclose(f);
            str[len] = '\0';
            return;
         }
         count++;
      }
      if (usethis)
      {
         if ((i = POS('\r', tmp)) > 0) tmp[i] = '\0';
         if ((i = POS('\n', tmp)) > 0) tmp[i] = '\0';

         if (len > 0)
         {
            i = strlen(tmp);
            str[len] = '\xD';
            memcpy(str+len+1, tmp, i);
            len += i+1;
         }
         else
         {
            len = i;
            memcpy(str, tmp, i);
         }
      }
      if ((count == use) && !usethis)
         usethis = true;
   }


   fclose(f);
   str[len] = '\0';
}


void Channel::updateBoxcheckCfg()
{
   boxcheck->readData( userinfo->getBoxCheck() );
}


void Channel::slotSendQuoted()
{
   const char *str = QApplication::clipboard()->text();
   int i,pos;
   char tmp[5000];

   if (str == NULL) return;

   pos = 0;
   strcpy(tmp, "> ");
   i = strlen(tmp);
   while (str[pos] != '\0')
   {
      if (str[pos] == '\n')
      {
         tmp[i] = '\r';
         tmp[i+1] = '\0';
         sendString( tmp );
         strcpy(tmp, "> ");
         i = strlen(tmp);
         pos++;
      }
      else
      {
         tmp[i] = str[pos];
         i++;
         pos++;
      }
   }

   tmp[i] = '\r';
   tmp[i+1] = '\0';
   sendString( tmp );
}


void Channel::slotInsertQuoted()
{
   const char *str = QApplication::clipboard()->text();
   int i,pos,line;
   char tmp[5000];

   if (str == NULL) return;

   vorschreib->cursorPosition( &line, &i );
   line++;

   pos = 0;
   strcpy(tmp, "> ");
   i = strlen(tmp);
   while (str[pos] != '\0')
   {
      if (str[pos] == '\n')
      {
         tmp[i] = '\0';
         vorschreib->insertLine(tmp, line);
         line++;
         strcpy(tmp, "> ");
         i = strlen(tmp);
         pos++;
      }
      else
      {
         tmp[i] = str[pos];
         i++;
         pos++;
      }
   }

   tmp[i] = '\0';
   vorschreib->insertLine(tmp, line);
}


void Channel::showKontextVorschrMenu()
{
   kontextvorschr->popup(QCursor::pos());
}


//   void Channel::updateStatusBar()
//
// Guckt selbst, was im Augenblick gemacht wird, und setzt entsprechend
// den Text in der Statuszeile.
void Channel::updateStatusBar()
{
   if ((flags & CH_DIDADIT) != 0)
   {
      if (didadit->didaditTx())
         statusbar->changeItem( klocale->translate("DIDADIT-Transmission"), ID_GENERAL );
      else
         statusbar->changeItem( klocale->translate("DIDADIT-Autosave"), ID_GENERAL );
      return;
   }

   if (auto7->isRxfile())
   {
      statusbar->changeItem( klocale->translate("7plus-Autosave"), ID_GENERAL );
      return;
   }

   if (binrx->isRxfile())
   {
      statusbar->changeItem( klocale->translate("AutoBIN-Autosave"), ID_GENERAL );
      return;
   }

   if ((flags & CH_AUTOBINTX) != 0)
   {
      statusbar->changeItem( klocale->translate("AutoBIN-Transmission"), ID_GENERAL );
      return;
   }

   if ((flags & CH_SAVE) != 0)
   {
      statusbar->changeItem( klocale->translate("Textfile save"), ID_GENERAL );
      return;
   }


   statusbar->changeItem( "", ID_GENERAL );
}


void Channel::updateStatusBar( const char *str )
{
   statusbar->changeItem( str, ID_GENERAL );
}


void Channel::slotYAPP()
{
   char filename[2000];
   int xfer;


   if ((flags & CH_YAPP) != 0)
   {
      if (KMsgBox::yesNo( NULL,
                          klocale->translate("YAPP-Transfer"),
                          klocale->translate("Do you want to deactivate the YAPP-Transfer?"),
                          0,
                          klocale->translate("&Yes"), klocale->translate("&No")) == 1)
      {
         // YAPP deaktivieren
         abortYAPP();
         return;
      }
      return;
   }


   DlgYAPP *dlg = new DlgYAPP( this );

   dlg->setGeometry( (mykapp->desktop()->width()-dlg->width())/2,
                     (mykapp->desktop()->height()-dlg->height())/2,
 		               dlg->width(),dlg->height());

   if (dlg->exec() == 1)
   {
      // Ok wurde angeklickt
      dlg->getData( xfer, filename );
      switch (xfer)
      {
         case 1: // RX
                 yapp = new YAPP( this );
                 if (filename[0] == '\0')
                    yapp->receiveYAPP( NULL );
                 else
                    yapp->receiveYAPP( filename );
                 flags |= CH_YAPP;
                 flags &= ~CH_LINEMODE;
                 break;
         case 2: // TX
                 yapp = new YAPP( this );
                 switch (yapp->sendYAPP( filename ))
                 {
                    case 0: // Alles ok, 'raus damit
                            flags |= CH_YAPP;
                            flags &= ~CH_LINEMODE;
                            break;
                    case 1: // File konnte nicht geoeffnet werden
                            KMsgBox::message(NULL, klocale->translate("Cannot open file"),
                                                   klocale->translate("Cannot open specified file."));
                            break;
                 }
                 break;
      }
   }

   kontextmenu->changeItem(klocale->translate("Abort &YAPP"), mnu_yapp);
   delete dlg;
}


void Channel::abortYAPP()
{
   if ((flags & CH_YAPP) == 0) return;

   // Wenn im Augenblick eine YAPP-Sendung ansteht wird sie abgebrochen
   if (sendqueue != NULL)
      if (sendqueue->type == SENDQ_YAPP)
         deleteQueueEntry();

   yapp->abortTransfer();
   delete yapp;
   yapp = NULL;

   flags &= ~CH_YAPP;
   flags |= CH_LINEMODE;

   updateStatusBar("");
   kontextmenu->changeItem(klocale->translate("&YAPP-Transfer"), mnu_yapp);
}


void Channel::abortABinTX()
{
   char tmp[100];


   flags &= ~CH_AUTOBINTX;
   flags |= CH_LINEMODE;
   delete bintx;
   bintx = NULL;
   kontextmenu->changeItem(klocale->translate("Send &AutoBIN"), mnu_sendabin);
   updateStatusBar("");

   sendString("#ABORT#\r");

   strcpy(tmp, "\r<LinKT>: AutoBIN-Transmission aborted.\r");
   if ((userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      sendString( tmp );
   else
      outText( tmp, strlen(tmp), config->colors->txText );

   if (sendqueue != NULL)
      if (sendqueue->type == SENDQ_FILE)
      {
         if (sendqueue->fd != -1)
            ::close(sendqueue->fd);
         deleteQueueEntry();
      }
}


void Channel::enableBoxcheckMenu()
{
   kontextmenu->setItemEnabled( mnu_boxcheck, true );
}


//   void Channel::withoutTransInfo()
//
// Fenster ohne TransferInfo-Widget
void Channel::withoutTransInfo()
{
   transferDlg = NULL;
   panner->resize(pannerHome->size());
}



//   void Channel::withTransInfo()
//
// Fenster mit TransferInfo-Widget
void Channel::withTransInfo( QWidget *win )
{
   transferDlg = win;
   panner->resize( pannerHome->width(), pannerHome->height()-transferDlg->height() );
   transferDlg->setGeometry( 0, pannerHome->height()-transferDlg->height(), pannerHome->width(), transferDlg->height() );
}


//   void Channel::insertBINChatLine( char *text, bool out )
//
// Packt an den Anfang der BIN-Uebertragung eine Textzeile hin, die mit
// "SP\-" davor ausgesendet wird.
void Channel::insertBINChatLine( char *text, bool out )
{
   s_sendqueue *tmp;
   char *data;
   int len = strlen(text);

   data = (char *) malloc(len+5);
   strcpy(data, "SP\\-");
   memmove(data+4, text, len);
   len += 4;
   data[len] = '\0';

   // Ein neues Element vor die Sendqueue packen
   tmp = (s_sendqueue *)malloc(sizeof(s_sendqueue));
   tmp->next = sendqueue;
   sendqueue = tmp;

   tmp->len = len;
   tmp->data = (char *)malloc(len);
   memcpy(tmp->data, data, len);
   tmp->show = out;
   tmp->abin = true;
   tmp->comp = COMP_FLAG;
   tmp->type = SENDQ_TEXT;
   tmp->fd = -1;

   free(data);
}


#include "channel.moc"

