/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "output.h"
#include "toolbox.h"
#include "global.h"
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>

#include <qfont.h>
#include <qfontmetrics.h>
#include <qapplication.h>
#include <qclipboard.h>


#define ABSTAND_ZEILEN 2
#define ABSTAND_OBEN 3
#define ABSTAND_LINKS 3
#define BREITE_VSCROLL 20

#define MARKTIMER 250           // Millisekunden



OutputLine::OutputLine(QFont f, QColor colback)
{
   line = NULL;
   font = f;
   markline = false;
   background = colback;
}


OutputLine::~OutputLine()
{
}


void OutputLine::addString( char *str, QColor color, int style )
{
   s_outputline *tmp;


   if (line == NULL)
   {
      // Erster Eintrag
      line = (s_outputline *)malloc(sizeof(s_outputline));
      tmp = line;
   }
   else
   {
      tmp = line;
      while (tmp->next != NULL) tmp = tmp->next;

      tmp->next = (s_outputline *)malloc(sizeof(s_outputline));
      tmp = tmp->next;
   }

   tmp->next = NULL;
   tmp->color = color;
   tmp->text = (char *) strdup(str);
   tmp->style = style;
}


void OutputLine::markLine( bool mark )
{
   markline = mark;
}


void OutputLine::drawString( int x, int y, int w, int h, QPainter *painter )
{
   s_outputline *outline;
   QFontMetrics fm(font);

   outline = line;
   while (outline != NULL)
   {
      if (markline)
      {
         painter->setPen(colors[config->colors->qsoWinMarkColor]);
         painter->setBackgroundColor(colors[config->colors->qsoWinMarkBack]);
         painter->setBackgroundMode( OpaqueMode );
      }
      else
      {
         // Erstmal den Hintergrund fuer diese Zeile malen
//         painter->setPen(colors[config->colors->background]);
//         painter->drawRect(0,y,
         painter->setPen(outline->color);
         painter->setBackgroundColor( background );
         painter->setBackgroundMode( TransparentMode );
      }


      if (outline->style == STYLE_ITALIC)
      {
         QFont f(painter->font());
//         f.setBold(false);
         f.setBold(true);
         f.setItalic(true);
         painter->setFont(f);
      }

      if (outline->style == STYLE_BOLD)
      {
         QFont f(painter->font());
         f.setBold(true);
         f.setItalic(true);
//         f.setItalic(false);
         painter->setFont(f);
      }

      if (outline->style == STYPE_BOLDITALIC)
      {
         QFont f(painter->font());
         f.setBold(true);
         f.setItalic(true);
         painter->setFont(f);
      }

      if (outline->style == STYLE_NORMAL)
      {
         QFont f(painter->font());
         f.setBold(false);
         f.setItalic(false);
         painter->setFont(f);
      }


      painter->drawText( x+ABSTAND_LINKS, y, w, h, AlignLeft, outline->text );
      x += fm.width(outline->text);

      outline = outline->next;
   }

   painter->setBackgroundColor( background );
}


void OutputLine::getLine(char *l)
{
   s_outputline *tmp=line;

   l[0] = '\0';
   while (tmp != NULL)
   {
      strcat(l, tmp->text);
      tmp = tmp->next;
   }
}


int OutputLine::getLineLength()
{
   s_outputline *tmp=line;
   int length=0;

   while (tmp != NULL)
   {
      length += strlen(tmp->text);
      tmp = tmp->next;
   }

   return length;
}


/**************************************************************************/
/**************************************************************************/
/**************************************************************************/


OutputWidget::OutputWidget( QWidget* parent )
             : QTableView( parent )
{
   setNumCols(1);
   setNumRows(0);
   setCellWidth(width());
   setCellHeight(1);

   setAutoUpdate(true);
   setTableFlags( Tbl_clipCellPainting | Tbl_vScrollBar );
   setBackgroundMode( PaletteBase );
   setFrameStyle( QFrame::WinPanel|QFrame::Sunken );

   rows = new QList<OutputLine>();
   rows->setAutoDelete(true);

   lenlastline = 0;
   scrolltimer = NULL;
   firstline = -1;

   vscroll = verticalScrollBar();

   newLine();
}


OutputWidget::~OutputWidget()
{
}


void OutputWidget::paintCell( QPainter *p, int row, int col )
{
    int w = cellWidth( col );			// width of cell in pixels
    int h = cellHeight( row );			// height of cell in pixels
    OutputLine *outline;


    p->setFont(font);

    outline = rows->at(row);
    outline->drawString( 0, 0, w, h, p );
}


void OutputWidget::newLine()
{
   OutputLine *outline;
   bool add_line=false,unten_bleiben=false;


   if ((vscroll->value() == vscroll->maxValue()) || (vscroll->maxValue() == 0))
      unten_bleiben = true;

   outline = new OutputLine(font, background);
   rows->append(outline);

   if ((int)rows->count() > config->buffersize)
   {
      rows->removeFirst();
      if (firstline > -1)
      {
         firstline--;
         if (firstline == -1) firstline = 0;
         lastline--;
         if (lastline == -1)
            clearAllMarks();
      }
   }
   else
   {
      add_line = true;
      setNumRows(rows->count());
   }

   if (unten_bleiben)
   {
      int t = totalHeight();
      int v = viewHeight();
      if (t > v)
         if (add_line)
            setYOffset(t-v);
         else
            scroll(0,cellHeight(1));
   }

   if ((int)rows->count() > config->buffersize)
      rows->removeFirst();

   lenlastline = 0;
}


void OutputWidget::writeText( char *text, QColor color, int style )
{
   int i;
   char *str, *tmp;

   str = (char *) strdup(text);
   tmp = (char *) malloc(strlen(text)+1);

   strcpy(str,text);

   do
   {
      i = strlen(str);
      if (i > 80-lenlastline)
      {
         memcpy(tmp, str, 80-lenlastline);
         tmp[80-lenlastline] = '\0';
         memmove(str, str+80-lenlastline, i-80+lenlastline);
         str[i-80+lenlastline] = '\0';
      }
      else
      {
         memcpy(tmp, str, i);
         tmp[i] = '\0';
         str[0] = '\0';
      }
      lenlastline = 0;

      rows->last()->addString(tmp, color, style);
      updateCell(rows->count()-1, 0);
      if (str[0] != '\0')
         newLine();
   } while (str[0] != '\0');

   lenlastline = strlen(tmp);
}


void OutputWidget::resizeEvent(QResizeEvent *e)
{
   QTableView::resizeEvent(e);

   setCellWidth(width());
}


void OutputWidget::setFont(QFont & f)
{
   font = f;

   QFontMetrics fm(font);
   setCellHeight(fm.height()+ABSTAND_ZEILEN);
}


void OutputWidget::updateCurrentLine()
{
   updateCell( rows->count()-1, 0 );
}


void OutputWidget::mousePressEvent( QMouseEvent *e)
{
   if (e->button() == RightButton)
      emit kontextMenu();

   if (e->button() == LeftButton)
   {
      clearAllMarks();
      marking = true;
      mark_new = true;
      markdirection = 0;
   }

   QTableView::mousePressEvent(e);
}


void OutputWidget::mouseReleaseEvent( QMouseEvent *e )
{
   char *tmp,tmp2[500];
   int len=0,i;


   if (e->button() == LeftButton)
   {
      marking = false;
      if (firstline > -1)
      {
         if (scrolltimer != NULL)
         {
            scrolltimer->stop();
            delete scrolltimer;
            scrolltimer = NULL;
         }

         // Schreibt den markierten Text in die Zwischenablage
         for (i=firstline;i<=lastline;i++)
            len += rows->at(i)->getLineLength()+1;

         tmp = (char *) malloc(len+1);

         tmp[0] = '\0';
         for (i=firstline;i<=lastline;i++)
         {
            rows->at(i)->getLine( tmp2 );
            strcat(tmp, tmp2);
            strcat(tmp, "\n");
         }

         QApplication::clipboard()->setText(tmp);

         free(tmp);
      }
   }

   QTableView::mouseReleaseEvent(e);
}


void OutputWidget::mouseMoveEvent( QMouseEvent *e )
{
   int i,row;


   if (marking)
   {
      row = findRow(e->pos().y());
      if (row != -1)
      {
         if (mark_new || firstline == -1)
         {
            firstline = row;
            lastline = row;
         }

         if (markdirection == 0)
            if (firstline <= row)
               markdirection = MARKDIR_TOPDOWN;
            else
               markdirection = MARKDIR_DOWNTOP;

         if (row < firstline)
            markdirection = MARKDIR_DOWNTOP;


         if (markdirection == MARKDIR_TOPDOWN)
         {
            // Gucken, ob gerade eine oder mehrere Zeilen wieder freigegeben
            // wurden.
            if (lastline > row)
            {
               // Ja! Alles zwischen firstline (incl) und row(excl.) freigeben
               for (i=row+1; i<=lastline; i++)
               {
                  rows->at(i)->markLine( false );
                  if (rowIsVisible(i))
                     updateCell(i, 0);
               }
               lastline = row;
            }

            // Gucken, ob wir gerade ein paar Zeilen markiert haben
            if ((row > firstline) || mark_new)
            {
               for (i=firstline; i<=row; i++)
               {
                  rows->at(i)->markLine( true );
                  if (rowIsVisible(i))
                     updateCell(i, 0);
               }
               lastline = row;
            }
         }


         // Gucken, ob wir ganz unten mit unserer Markierung angelangt
         // sind. Wenn ja wird das ganze etwas heruntergescrollt
         if (row == lastRowVisible())
         {
            scrolltimer = new QTimer(this);
            connect(scrolltimer, SIGNAL(timeout()), SLOT(slotTimerUnten()));
            scrolltimer->start(MARKTIMER, true);
         }


         // Gucken, ob wir ganz oben mit unserer Markierung angelangt
         // sind. Wenn ja wird das ganze etwas heraufgescrollt
         if (row != 0)
            if (row == topCell())
            {
               scrolltimer = new QTimer(this);
               connect(scrolltimer, SIGNAL(timeout()), SLOT(slotTimerOben()));
               scrolltimer->start(MARKTIMER, true);
            }


         // Gucken, ob wir gerade ueber dem oberen Ende 'rausscrollen
         if (markdirection == MARKDIR_DOWNTOP)
         {
            // Oben soll was markiert werden
            if (row < firstline)
            {
               for (i=row; i<firstline; i++)
               {
                  rows->at(i)->markLine( true );
                  if (rowIsVisible(i))
                     updateCell(i, 0);
               }
               firstline = row;
            }

            // Wieder nach unten scrollen? -> alles obere "demarkieren"
            if (row > firstline)
            {
               for (i=firstline; i<row; i++)
               {
                  rows->at(i)->markLine( false );
                  if (rowIsVisible(i))
                     updateCell(i, 0);
               }
               firstline = row;
            }
         }

         if (firstline == lastline)
            markdirection = MARKDIR_TOPDOWN;

         mark_new = false;
      }
   }

   QTableView::mouseMoveEvent(e);
}


void OutputWidget::slotTimerUnten()
{
   unsigned int row = lastRowVisible();

   if (row < rows->count()-1)
   {
      setTopCell(topCell()+1);
      rows->at(row)->markLine( true );
      updateCell(row, 0);
      lastline = row;
   }

   if ((scrolltimer != NULL) && ((unsigned int)row < rows->count()-1))
      scrolltimer->start(MARKTIMER, true);
}


void OutputWidget::slotTimerOben()
{
   int row = topCell();

   if (row > 0)
   {
      setTopCell(topCell()-1);
      rows->at(row)->markLine( true );
      updateCell(row, 0);
      firstline = row;
   }

   if ((scrolltimer != NULL) && (row != 0))
      scrolltimer->start(MARKTIMER, true);
}


void OutputWidget::slotPageUp()
{
   ((QScrollBar *)vscroll)->subtractPage();
}


void OutputWidget::slotPageDown()
{
   ((QScrollBar *)vscroll)->addPage();
}


void OutputWidget::clearAllMarks()
{
   int i;

   if (firstline == -1) return;

   for (i=firstline; i<=lastline; i++)
   {
      rows->at(i)->markLine( false );
      if (rowIsVisible(i))
         updateCell(i, 0);
   }
   firstline = -1;
   QApplication::clipboard()->clear();
}


void OutputWidget::setBackgroundColor( QColor & colback )
{
   background = colback;

   QTableView::setBackgroundColor( colback );
}



#include "output.moc"

