/* $Header: /.automount/sherluck/sh/sy/alan/dlg/tiger/src/RCS/tigaprs.h,v 1.21 1997/11/12 20:11:13 alan Exp $
 * tigaprs.h: data structures for converting TIGER/Line(R) 1994 to APRS.
 *
 * Copyright (c) 1996  E. Alan Crosswell
 *
 * Alan Crosswell, N2YGK
 * 144 Washburn Road
 * Briarcliff Manor, NY 10510, USA
 * n2ygk@weca.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * (See the file "COPYING" that is included with this source distribution.)
 *
 */
struct latlon {
  int lat;			/* latitude, as an integer */
  int lon;			/* longitude */
};

struct feat {			/* Census Feature Class Code (CFCC), e.g. A41*/
  char class;			/* 'A' */
  u_int level;			/* 41 */
};

struct name {			/* feature name */
  char fedirp[3];		/* direction prefix */
  char fename[31];		/* name */
  char fetype[5];		/* type, e.g. "road" */
  char fedirs[3];		/* direction suffix */
};

/*
 * For lookup purposes, hash segments by:
 *
 * 1. TLID in order to find the seg quickly when reading other record types.
 * 2. FROM lat/lon and TO lat/lon to determine segment adjacency.
 * 3. combined FROM/TO lat/lon for single-sided segments only (county bdrys).
 *
 * The hash function yields a hash bucket which is then singly linked for
 * all links that hash to the same bucket.
 *
 * For road construction purposes, segments are also members of a doubly-linked
 * list which gets edited as roads get built up.
 *
 * The list of single-sided segments is maintained seperately 'cuz it has
 * many fewer segs and we don't need to waste space.
 *
 * Be careful of segs[] moving!
 */
struct seg {			/* TIGER calls this a "complete chain" */
  struct seg *ps,*ns;		/* linear previous, next ptrs (in a road) */
  struct seg *tlns;		/* TLID: linear next pointer (lookup) */
  struct seg *frns,*tons;	/* Lat/lon: from&to hash chain ptrs (lookup) */
  u_long tlid;			/* unique id */
  struct feat feat;		/* feature class */
  struct name name;
  struct latlon from;		/* start of road segment (from rec type 1) */
  struct latlon to;		/* end (not sure we need these) */
  int first;			/* first element in array of way points */
  int last;			/* last element index. */
  int flags;			
};
#define SEG_SKIP 0x01		/* (temporarily) ignore this seg */
#define SEG_DUP 0x02		/* seg is a dupe -- always ignore it */
#define SEG_IGN (SEG_SKIP|SEG_DUP) /* ignore these */
#define SEG_REV 0x04		/* read this segment in reverse */
#define SEG_SS 0x08		/* segment is single-sided */
#define SEG_FF 0x10		/* is eligible for faster fuzzing */
#define SEG_FS 0x20		/* endpoints can be fuzzed too */
#define SEG_LKFR 0x40		/* lock this seg's FROM (SEG_FS overrides)*/
#define SEG_LKTO 0x80		/* lock this seg's TO (SEG_FS overrides)*/
#define SEG_DONE 0x100		/* chain is complete (all way points seen) */

struct sshash {			/* hash table chain entry */
  struct sshash *nh;		/* Single-sided search (combined lat/lon) */
  struct seg *s;		/* the seg we are hashing */
};

struct road {
  int roadid;			/* just to help debugging; */
  struct road *nr, *pr;		/* linear next, previous chain */
  struct road *fpr,*fnr,*tpr,*tnr; /* from&to hash chain previous, next ptrs */
  struct seg *head;		/* one end of the road */
  struct seg *tail;		/* the other end */
  int flags;
};
#define RD_SKIP 0x01		/* skip this road */

/* 
 * LABELS:  Several flavors:
 *  I. Area polygon-based:
 *     1. Cities, Towns, etc.
 *        - Record type P identifies polygon center lat/lon.
 *        - Link to record type S (geographic entity codes).
 *        - Use record type C to entity map codes to names.
 *     2. Area landmarks (parks, cemteries, etc.)
 *        - Record type P.
 *        - Record type 8 links polygon to area landmark.
 *        - Record type 7 names area landmarks.
 *     Note: To get fancy and color in the polys, need type I records which
 *           link to type 1's.
 *
 *  Areas are built up from several elementary polygons lumped together.
 *  They are lumped by virtue of being linked to by a type A or S entity
 *  record with matching entity code (e.g. FIPS-55 code).
 *
 *  II. Point landmarks:
 *     1. Record type 7 directly contains the lat/lon.
 */
struct poly {			/* an elementary polygon */
  struct poly *np;		/* polyid hash chain */
  char cenid[6];		/* census idcode (A->P)*/
  int polyid;			/* polygon id (from A/S->P) */
  struct latlon ctr;		/* internal point of this polygon */
  int flags;
};
#define POLY_SKIP 0x01		/* ignore */


/*
 * an area is a grouping of polygons that share a common fips-55 code
 */
struct arealink {
  struct arealink *nal;		/* next area link */
  struct poly *p;		/* pointer to the poly that is in this area */
};

struct area {			/* one for each fips-55 code */
  struct area *na;		/* next area in list for given area type */
  struct arealink *link;	/* the list of links */
  int fips;			/* fips 55 code for this area */
  char *name;			/* area name */
  char *classcode;		/* fips class code */
};

struct kgl {			/* key geographic location */
  struct kgl *nk;		/* next area in list for given area type */
  struct arealink *link;	/* the 1-element list of links to polys */
  struct feat cfcc;		/* the cfcc for this feature */
  char name[31];		/* kgl name */
};

struct fips {
  struct fips *pnl;		/* place list */
  int fips;			/* fips55 code */
  char classcode[3];		/* fips55 class code */
  char name[67];
};

struct land {			/* landmark */
  struct land *nl;		/* next in list */
  int state;			/* state code */
  int county;			/* county code */
  int landid;			/* landmark id number */
  struct feat cfcc;		/* cfcc code */
  char name[31];		/* landmark name */
  union {			/* either a point or area landmark */
    struct latlon ctr;		/* point landmark location */
    struct arealink *link;	/* or area landmark polygon group */
  } u;
  int flags;
};

#define LAND_POINT 0x01		/* is a point landmark */

#define APRSLMAX 12
#define LMAXLINES 6
#define MAXL 60

struct label {
  struct label *next;		/* next label */
  struct latlon ll;		/* location of the label */
  double range;			/* range to display it at */
  struct feat cfcc;		/* the cfcc for the label */
  char symbol[5];		/* the symbol if any */
  char name[MAXL];		/* label name */
  int len;			/* length */
  int flags;
};
#define LABEL_IGN 1		/* ignore */

enum map_overlap {onot=0,oe,ose,os}; /* this is a {e,se,s} overlap map of... */
enum map_quad {not=0,nw,ne,sw,se}; /* this is a quadrant map of... */

struct map {			/* a (potential) DOS/APRS output map */
  int mapid;			/* map number */
  int flags;			/* flags: */
#define MAP_PRINT 1		/*  map has been printed: put in maplist */
  double range;			/* radius of map in miles */
  double detail;		/* level of detail in miles */
  int ppdy;			/* pixel per degree in Y */
  int pts;			/* current number of would-be output points */
  int fuzz;			/* current fuzz factor in use for this map */
  double water_filt;		/* current water filtering percentage */
  double slopefuzz;		/* current slope ratio */
  struct latlon max;		/* maximum lat/lon == map origin (NW corner) */
  struct latlon min;		/* minimum lat/lon (SE corner) */
  struct latlon ctr;		/* center of map */
  struct map *nw,*ne,*sw,*se;	/* quadrants of this map (if a parent) */
  enum map_quad qd;		/* this map is one of the quadrants of ... */
  struct map *parent;		/* ...this parent map */
  struct map *oe,*ose,*os;	/* overlaps of this base map (if a base) */
  enum map_overlap overlap;	/* this map is one of the overlaps of... */
  struct map *base;		/* ...this base map */
  struct label *labels,*lastl;	/* list of labels (and last one) for this map */
};


