/* $Header: /.automount/sherluck/sh/sy/alan/dlg/tiger/src/RCS/tig2aprs.c,v 1.58 1997/11/12 20:11:13 alan Exp $
 * tig2aprs: tiger-to-aprs map generator.
 * Reads US Census TIGER/Line(R) data and writes DOS APRS maps.
 *
 * Copyright (c) 1996,1997  E. Alan Crosswell
 *
 * Alan Crosswell, N2YGK
 * 144 Washburn Road
 * Briarcliff Manor, NY 10510, USA
 * n2ygk@weca.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * (See the file "COPYING" that is included with this source distribution.)
 *
 * Code to calculate mapping from lon/lat to x/y is derived from MAKEMAP1.BAS
 * in the APRS distribution, by KB4XF Jack Cavanagh, of Fredericksburg, VA
 * and modified by WB4APR Bob Bruninga.
 *
 * Reads from stdin so the files can be compressed or whatever.  Just
 * cat together the various files and pipe them in.
 *
 * Since the TIGER/Line coordinate system uses longitude/latitude as
 * integers with an implicit decimal point, all processing might as well
 * be done with integer arithmetic until final output into the APRS file
 * at which point scaling from lat/lon to pixel coordinates has to be done.
 *
 *  Tiling algorithm recursively subdivides the map until an output point
 *  count < maxpts (3000 for DOS APRS) is reached at which point the map
 *  is "good enough" and can be written out.  This way, sparse areas can
 *  be squeezed into a large map while more dense areas will result in
 *  smaller and smaller maps.  Give up when we are down to 1 mile!
 *
 *  Example: Top left quadrant is too dense, so it is subdivided into
 *    four new maps.  The other three quads are sparse enough as is.
 *
 *     +------+------+--------------+
 *     |      |      |              |
 *     |      |      |              |
 *     |      |      |              |
 *     +------+------+              |
 *     |      |      |              |
 *     |      |      |              |
 *     |      |      |              |
 *     +------+------+--------------+
 *     |             |              |
 *     |             |              |
 *     |             |              |
 *     |             |              |
 *     |             |              |
 *     |             |              |
 *     |             |              |
 *     +-------------+--------------+
 *
 * To do:
 *  - if a water feature has been filtered out, make sure to filter
 *    out its name from the lables.
 *  - NAD27 to NAD83 (WGS84) datum correction for 1994 only.
 *  - Possibly use APRS special symbol for road names
 *  - gprof to speed up
 *  - make defcuts table user-configurable from a config file.
 *  - clean up cfccrange,fipsrange,defcuts.
 *  - look at Tiger binary file format used by http://tiger.census.gov map
 *    generator.
 *  - make more modular & add support for other input and output formats
 */
static char revision[] = "$Revision: 1.58 $";
#define rev &revision[11]
#define BANNER "tig2aprs %s (c) 1996,1997 Alan Crosswell N2YGK\r\n"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#define TGR_LL 1		/* use struct tgr_ll */
#include "tgr.h"
#include "tigaprs.h"
#ifdef DBMALLOC
#include <malloc.h>
#endif

static int debug = 0;
#define DEBUG_LABELS 0x01
#define DEBUG_POLY   0x02
#define DEBUG_AREA   0x04
#define DEBUG_KGL    0x08
#define DEBUG_FUZZ   0x10
static int verbose = 0;
static int join=0;
static int all_tlid=0;
static int fename_match=0;
static int color_match=0;
static int data_file=0;
static int overlap=0;
static double water_filt=-1.0,max_filt=-1.0;
static double slopefuzz=-1.0,maxslope=-1.0;
static int maxpts = 3000;
static int pixper = -1;
static int label_places = 0;
static int label_landmarks = 0;
static int label_kgls = 0;
static char *namepfx = "map";
static FILE *maplist = NULL;
static char maplistname[50];
static char cmdline[200];
static int maplistmax = 120;	/* 169 is the hard max. Leave some room. */
static int currd = 0;
static char tigver[10];

union inbuf {
  struct tgr1 t1;
  struct tgr2 t2;
  char buf[100];
} in;
struct latlon ctr = {41120000,73770000}; /* default to my county ctr */
struct latlon max,min;
struct latlon inmax = {-999999999,-999999999};
struct latlon inmin = {999999999,999999999};
static double range = 32.0;	/* default to 32 miles */
static double detail = -1;	/* level of detail, in terms of mileage */
static double tile = -1;	/* smallest tile radius I will accept */
int fuzz = 1;			/* (x,y),(x+-fuzz,y+-fuzz) considered same */
int maxfuzz = 20;		/* don't fuzz worse than this */

struct seg *segs;		/* TIGER calls these "complete chains" */
int curseg;
int maxseg;
#define SEGS 20000

struct latlon *Pts;		/* array of lat/lon points */
int curpt;
int maxpt;
#define PTS (2*SEGS)

struct poly *polys;		/* array of polygons */
int curpoly;
int maxpoly;
#define POLYS 10000

struct fips *fipscodes;		/* array of fipss */
int curfips;
int maxfips;
#define FIPS 1000

struct land *landmarks;		/* array of landmarks */
int curland;
int maxland;
#define LANDS 1000

int ptsout = 0;

struct map *rootmap;
struct map *map_init(double,double,int,int,enum map_quad,struct map *,enum map_overlap,struct map *);
u_long strntoul(char *,int,int);
enum name_eq_opt {none=0,color,name} name_eq_global = none;
int name_eq(struct seg *, struct seg *,enum name_eq_opt);
int add_road_hash(struct road *);
int drop_road_hash(struct road *);
int add_hashfrom(struct road *,int);
int add_hashto(struct road *,int);
int drop_hashfrom(struct road *,int);
int drop_hashto(struct road *,int);
int rehash_road_head(struct road *, struct seg *);
int rehash_road_tail(struct road *, struct seg *);
int invisible(struct map *);

main(argc,argv)
int argc;
char **argv;
{
  int invis;

  do_args(argc,argv);		/* parse args */
  init();			/* set up initial values */
  read_in();			/* read in the TIGER/Line data */
  invis = invisible(rootmap);
  if (verbose)
    fprintf(stderr,"map%03d: %d off-map features filtered out\n",
	    rootmap->mapid,invis);
  kill_dups();			/* get rid of overlapping single-sided segs */
#ifdef DDD
  fprint_segs(stderr);
#endif
  link_segs();			/* try coalescing some segs together */
  if (join)
    join_roads();		/* then do it again by reversing roads */
  lock_roads();			/* "lock" important segs against fuzzing */
  do_tile(rootmap);		/* and tile it out */
  do_rootmaplist(rootmap);	/* and generate the maplist */
  if (verbose)
    fprintf(stderr,"done.\n");
  exit(0);
}

/*
 * Tile out the map, recursively subdividing into quadrants down to
 * the lower limit set by 'tile'.
 */
do_tile(m)
struct map *m;
{
  int i,pts = -1; 
  struct map *mmm[4];		/* 4 overlapped maps */

  bzero(mmm,sizeof(mmm));
  mmm[0] = m;

  if (overlap) {		/* have to do this 4 times */
    int latdelta = (m->max.lat - m->min.lat)/2;
    int londelta = (m->max.lon - m->min.lon)/2;
    
    if (verbose)
      fprintf(stderr,"map%03d: initializing overlap maps....\n",m->mapid);
    mmm[1] = map_init(m->range,m->detail,m->ctr.lat,m->ctr.lon-londelta,0,0,oe,m); /* east */
    m->oe = mmm[1];
    mmm[2] = map_init(m->range,m->detail,m->ctr.lat-latdelta,m->ctr.lon-londelta,0,0,ose,m); /* se */
    m->ose = mmm[2];
    mmm[3] = map_init(m->range,m->detail,m->ctr.lat-latdelta,m->ctr.lon,0,0,os,m); /* south */
    m->os = mmm[3];
    if (verbose)
      fprintf(stderr,"map%03d: overlaps: e=map%03d se=map%03d s=map%03d\n",
	      m->mapid,mmm[1]->mapid,mmm[2]->mapid,mmm[3]->mapid);
  }

  /* Try to reduce map to below maxpts.  Have to do this for each overlay
     as well to make sure they all fit. */
  if (verbose)
    fprintf(stderr,"map%03d: Possibly tiling.  Reducing first.\n",m->mapid);

  for (pts = -1,i = 0; i < ((overlap)?4:1) && mmm[i]; i++) {
    int n;

    n = reduce_map(mmm[i]);	/* xxx this trashes global seg flags! */
    if (n < maxpts)
      print_map(mmm[i],"map");	/* so print the map now */
    if (n > pts)		/* and maybe need to cleanup later */
      pts = n;			/* find worst case among the 4 overlaps */
  }

  if (pts > maxpts) {		/* undo what we just did! and then tile */
    for (i = 0; i < ((overlap)?4:1); i++) {
      if (mmm[i]) {
	if (mmm[i]->pts < maxpts)
	  unlink_map(mmm[i],"map"); /* pretty stupid */
	if (i > 0)
	  free_map(mmm[i]);	/* don't free the 'm' map */
      }
    }
    if (m->range/2.0 >= tile) { /* OK to split and recurse */
      double newrange = m->range/2L;
      int latdelta = (m->max.lat - m->min.lat)/4;
      int londelta = (m->max.lon - m->min.lon)/4;

      if (verbose)
	fprintf(stderr,"map%03d: Base map (or an overlap) has too many pts.  Tiling....\n",
		m->mapid,m->pts);

      m->nw = map_init(newrange,m->detail,m->ctr.lat+latdelta,m->ctr.lon+londelta,nw,m,0,0);
      m->ne = map_init(newrange,m->detail,m->ctr.lat+latdelta,m->ctr.lon-londelta,ne,m,0,0);
      m->sw = map_init(newrange,m->detail,m->ctr.lat-latdelta,m->ctr.lon+londelta,sw,m,0,0);
      m->se = map_init(newrange,m->detail,m->ctr.lat-latdelta,m->ctr.lon-londelta,se,m,0,0);
      if (verbose)
	fprintf(stderr,"map%03d: split into nw=%03d,ne=%03d,sw=%03d,se=%03d\n",
		m->mapid,m->nw->mapid,m->ne->mapid,m->sw->mapid,m->se->mapid);
      do_tile(m->nw);
      do_tile(m->ne);
      do_tile(m->sw);
      do_tile(m->se);
    } else {			/* couldn't reduce it enough */
      fprintf(stderr,"map%03d: WARNING: too many points (%d) and already at the lowest tile size.\n",
	      m->mapid,pts);
    }
  }
}

int
reduce_map(m)
struct map *m;
{
  int invis = invisible(m);
  int tryfuzz,trymax,trymin;

  invis += water_filter(m);
  count_roads(m);

  if (m->pts >= maxpts) {	/* need to fuzz down */
    int lastpts,lastfuzz;
    float lastslopefuzz;

    if (verbose)
      fprintf(stderr,"map%03d: tossing some data to make it fit. (pts=%d, fuzz=%d)\n",
	      m->mapid,m->pts,m->fuzz);
    if (data_file && m->pts < maxpts*2) { /* first save a map.dat */
      print_map(m,"dat");
    }
    
    /*
     * trying fuzzing it down(???):
     *  1. do a minimum slopefuzz & see if that's enough.
     *  2. increment the water_filter to maximum.
     *  3. increment slopefuzz to maximum.
     *  4. increment fuzz to maximum.
     */
    if (slopefuzz > 0)
      m->slopefuzz = slopefuzz;
    for (lastpts = -1; m->pts != lastpts && m->pts >= maxpts; lastpts = m->pts) {
      if (count_roads(m) < maxpts) { /* min slopefuzz test */
	break;
      }
      if (verbose)
	fprintf(stderr,"map%03d: start fuzzing. pts now %d\n",m->mapid,m->pts);
      for (; m->pts >= maxpts && m->water_filt < max_filt; 
	   m->water_filt *= 1.10) {
	invis += water_filter(m);
	if (verbose)
	  fprintf(stderr,"map%03d: water filt %.3f. pts now %d\n",
		  m->mapid, m->water_filt,m->pts);
	if (count_roads(m) < maxpts) {
	  break;
	}
      }
      /* nest slope fuzzing inside proximity fuzzing? */
      /* do a faster search:
	 start at the most fuzzed to see if we will have any chance
	 of success.  Then binary search from there. */
      trymax = tryfuzz = maxfuzz;
      trymin = fuzz;
      for (;;) {
	m->fuzz = tryfuzz;
	m->slopefuzz = maxslope;
	count_roads(m);
	if (verbose)
	  fprintf(stderr,"map%03d: fuzz %d, slope fuzz %.3f. pts now %d\n",
		  m->mapid, m->fuzz, m->slopefuzz, m->pts);
	if (m->pts >= maxpts) {
	  /* too many pts, find next midpoint between tryfuzz & maxfuzz */
	  if (tryfuzz >= maxfuzz) { /* failure to reduce enough */
	    if (verbose)
	      fprintf(stderr,"map%03d: unable to reduce pts=%d below maxpts\n",
		      m->mapid, m->pts);
	    break;
	  } else {
	    if (trymin == trymax) { /* can't go any more */
	      if (debug&DEBUG_FUZZ)
		fprintf(stderr,"map%03d: reached bottom min=max=%d lastpts=%d\n",
			m->mapid,trymin,lastpts);
	      m->pts = lastpts;
	      m->fuzz = lastfuzz;
	      m->slopefuzz = lastslopefuzz;
	      break;
	    } else {
	      trymin = tryfuzz+1;
	      tryfuzz = (int)rint((double)trymax-(double)(trymax-tryfuzz)/2.0);
	      if (debug&DEBUG_FUZZ)
		fprintf(stderr,"map%03d: pushing tryfuzz up to %d min=%d max=%d\n",
			m->mapid,tryfuzz,trymin,trymax);
	      continue;
	    }
	  }
	} else { /* pts < maxpts, possibly too far */
	  /* start by sliding slopefuzz back down to minimum */
	  for (m->slopefuzz = maxslope/1.10;
	       m->pts < maxpts && m->slopefuzz >= slopefuzz; 
	       m->slopefuzz /= 1.10) {
	    count_roads(m);
	    if (verbose)
	      fprintf(stderr,"map%03d: fuzz %d, slope fuzz %.3f. pts now %d\n",
		      m->mapid, m->fuzz, m->slopefuzz, m->pts);
	    lastpts = m->pts;
	    lastslopefuzz = m->slopefuzz;
	  }
	  /* falling out here *may* have found the sweet spot
	     where pts has just exceeed maxpts */
	  lastfuzz = m->fuzz;
	  if (m->pts >= maxpts) {
	    /* found it: increase slopefuzz one notch and be done! */
	    m->slopefuzz *= 1.10 * 1.10;
	    m->pts = lastpts;
#define SANITY 1
#ifdef SANITY
	    count_roads(m);
	    if (m->pts > maxpts) {	/* sanity check */
	      fprintf(stderr,"map%03d: ERROR: INSANE! slopefuzz now %.3f but pts=%d!!!\n",
		      m->mapid,m->slopefuzz,m->pts);
#endif /*SANITY*/
	    }
	    break;
	  } else { /* slopefuzz wasn't enuf to get near maxpts: try harder */
	    /* split the difference between fuzz and tryfuzz */
	    if (tryfuzz <= fuzz) {
	      /* can't fuzz or slopefuzz any less than this! */
	      if (debug&DEBUG_FUZZ)
		fprintf(stderr,"map%03d: got minimal fuzz=%d\n",
			m->mapid,tryfuzz);
	      m->slopefuzz *= 1.10; /* ??? */
	      break;
	    }
	    trymax = tryfuzz-1;	/* don't go any higher than this next time */
	    tryfuzz = (int)rint((double)trymin-1.0 
					 + (double)(tryfuzz-trymin)/2.0);
	    if (debug&DEBUG_FUZZ)
	      fprintf(stderr,"map%03d: pushing tryfuzz down to %d min=%d max=%d\n",
		      m->mapid,tryfuzz,trymin,trymax);
	    continue;
	  }
	}
      }
    }
  }
  if (verbose)
    fprintf(stderr,"map%03d: final fuzz %d, slope fuzz %.3f. pts %d\n",
	    m->mapid, m->fuzz, m->slopefuzz, m->pts);
  return m->pts;
}

/*
 * finally print the map out in APRS/DOS format.  If overlap is set,
 * then print three half-way overlapping maps so that scrolling from
 * map to map will work when a series of maps are put together.
 */
print_map(m,suffix)
struct map *m;
char *suffix;
{
  int domaplist = !strcmp(suffix,"map");
  int before = m->pts;

  if (m->pts > 0) {
    if (verbose)
      fprintf(stderr,"map%03d: printing %s%03d.%s...",
	      m->mapid,namepfx,m->mapid,suffix);
    map_open(m,suffix);		/* reopen stdout with new filename */
    print_roads(m);		/* print roads out to a DOS/APRS map */
    print_labels(m);		/* print labels */
    if (domaplist)
      m->flags |= MAP_PRINT;
    if (m->pts != before) {
      fprintf(stderr,"map%03d: ERROR: point count discrepancy before=%d after=%d\n",
	      m->mapid,before,m->pts);
    }
    if (verbose)
      fprintf(stderr,"%d points\n",m->pts);
  }
}

map_open(m,suffix)
struct map *m;
char *suffix;
{
  char namebuf[500];

  sprintf(namebuf,"%s%03d.%s",namepfx,m->mapid,suffix);
  freopen(namebuf,"w",stdout);
}

unlink_map(m,suffix)
struct map *m;
char *suffix;
{
  char namebuf[500];

  sprintf(namebuf,"%s%03d.%s",namepfx,m->mapid,suffix);
  unlink(namebuf);
  if (strcmp(suffix,"map")==0)
    m->flags &= ~MAP_PRINT;
}

free_map(m)
struct map *m;
{
  if (m) {
    switch(m->qd) {
    case nw:
      m->parent->nw = 0;	/* clobber parent reference to me */
      break;
    case ne:
      m->parent->ne = 0;
      break;
    case sw:
      m->parent->sw = 0;
      break;
    case se:
      m->parent->se = 0;
      break;
    }
    switch(m->overlap) {
    case oe:
      m->base->oe = 0;		/* clobber base reference to me */
      break;
    case ose:
      m->base->ose = 0;
      break;
    case os:
      m->base->os = 0;
      break;
    }
    free(m);
  }
}

do_rootmaplist(m)
struct map *m;
{
  FILE *maplist[5],*open_maplist();
  int count[4],tot;
  int i, n;
  char *qd[4] = {"nw","ne","sw","se"};

  if (!m)
    return;

  tot = count[0] = count_maplist(m->nw,0);
  tot += (count[1] = count_maplist(m->ne,0));
  tot += (count[2] = count_maplist(m->sw,0));
  tot += (count[3] = count_maplist(m->se,0));
  if (tot > maplistmax) {
    if (verbose)
      fprintf(stderr,"map%03d: maplist would have %d maps;  splitting up....\n",
	      m->mapid,tot);
    for (i = 0; i < 4; i++) {
      maplist[i] = open_maplist(maplistname,i+1);
      if (verbose)
	fprintf(stderr,"map%03d: %s sublist has %d maps.\n",
		m->mapid,qd[i],count[i]);
    }
    n = 4;
  } else {
    maplist[0] = maplist[1] = maplist[2] = maplist[3] 
      = open_maplist(maplistname,0);
    n = 1;
  }
  for (i = 0; i < n; i++) {
    print_maplist(maplist[i],m); /* print my map entry */
    print_maplist(maplist[i],m->oe); /* and my overlaps */
    print_maplist(maplist[i],m->ose);
    print_maplist(maplist[i],m->os);
  }
  do_maplist(maplist[0],m->nw);	/* and recurse down quadrants */
  do_maplist(maplist[1],m->ne);
  do_maplist(maplist[2],m->sw);
  do_maplist(maplist[3],m->se);
  for (i = 0; i < n; i++)
    close(maplist[i]);
}

do_maplist(maplist,m)		/* descend the map tree */
FILE *maplist;
struct map *m;
{
  if (!m)
    return;
  print_maplist(maplist,m);	/* print my map entry */
  print_maplist(maplist,m->oe); /* and my overlaps */
  print_maplist(maplist,m->ose);
  print_maplist(maplist,m->os);
  do_maplist(maplist,m->nw);	/* and recurse down quadrants */
  do_maplist(maplist,m->ne);
  do_maplist(maplist,m->sw);
  do_maplist(maplist,m->se);
}

int
count_maplist(m,n)
struct map *m;
int n;
{
  if (!m)
    return n;
  if (m->flags&MAP_PRINT)
    ++n;
  if (m->oe && m->oe->flags&MAP_PRINT)
    ++n;
  if (m->ose && m->oe->flags&MAP_PRINT)
    ++n;
  if (m->os && m->oe->flags&MAP_PRINT)
    ++n;
  n = count_maplist(m->nw,n);
  n = count_maplist(m->ne,n);
  n = count_maplist(m->sw,n);
  n = count_maplist(m->se,n);
  return n;
}

FILE *
open_maplist(name,n)
char *name;
int n;
{
  static char namebuf[100];
  FILE *maplist = NULL;
  int i;
  
  if (n) {
    sprintf(namebuf,"%s%d",name,n);
  } else {
    strcpy(namebuf,name);
  }
  maplist = fopen(namebuf,"w");
  if (maplist == NULL)
    perror(namebuf);
  return maplist;
}

print_maplist(maplist,m)	/* print (or count) a maplist entry */
FILE *maplist;
struct map *m;
{
  if (!m)
    return;
  if (!(m->flags & MAP_PRINT))
    return;
  fprintf(maplist,"%s%03d.map,%d.%06d,%d.%06d,%f, ",
	  namepfx,m->mapid,
	  m->ctr.lat/1000000,m->ctr.lat%1000000,
	  m->ctr.lon/1000000,m->ctr.lon%1000000,
	  m->range);
  if (m->overlap) {
    char *ovs[3] = {"E","SE","S"};
    fprintf(maplist,"%s overlap of %s%03d\r\n",
	    ovs[((int)m->overlap) - 1],namepfx,m->base->mapid);
  } else if (m->parent) {
    char *qp;
    if (m->parent->nw == m)
      qp = "NW";
    else if (m->parent->ne == m)
      qp = "NE";
    else if (m->parent->sw == m)
      qp = "SW";
    else if (m->parent->se == m)
      qp = "SE";
    fprintf(maplist,"%s quadrant of %s%03d\r\n",
	    qp,namepfx,m->parent->mapid);
  } else {
    fprintf(maplist,"\r\n");
  }
}

/* table of input functions and the order data must appear */
void type1(struct tgr1 *), end_type1(void);
void type2(struct tgr2 *), end_type2(void);
void typeP(struct tgrP *), end_typeP(void);
void typeC(struct tgrC *), end_typeC(void);
void typeS(struct tgrS *);
void type7(struct tgr7 *), end_type7(void);
void type8(struct tgr8 *);
void type9(struct tgr9 *);

struct {
  char type;
  void (*infunc)();
  void (*endfunc)();
} functab[] = {
  {'1',type1,end_type1},
  {'2',type2,end_type2},
  {'P',typeP,end_typeP},
  {'C',typeC,end_typeC},
  {'S',typeS,NULL},
  {'7',type7,end_type7},
  {'8',type8,NULL},
  {'9',type9,NULL},
};

read_in()
{
  int i,recidx = -1;
  int skip_type = 0, type_OK = 0;
  char curtype = '\0';
  int type1_done = 0;
  char *mygets(char *, int, FILE *);

  while (mygets(in.buf,sizeof(in),stdin)) {
    if (curtype != in.buf[0]) {	/* type break */
      if (recidx >= 0 && functab[recidx].endfunc)
	functab[recidx].endfunc();
      curtype = in.buf[0];
      for (type_OK = 0, ++recidx; 
	   recidx < sizeof(functab)/sizeof(functab[0]); recidx++)
	if (functab[recidx].type == curtype) {
	  ++type_OK;
	  skip_type = 0;
	  break;
	}
      if (!type_OK) {
	fprintf(stderr,"Record type %c not recognized; skipping it\n",curtype);
	++skip_type;
      }
    }
    if (!skip_type)
      functab[recidx].infunc(&in);
  }
  if (recidx >= 0 && functab[recidx].endfunc) /* get the post-EOF end */
    functab[recidx].endfunc();

  if (verbose) {
    fprintf(stderr,"input: max coords: %ld/%ld\n",inmax.lat,inmax.lon);
    fprintf(stderr,"input: min coords: %ld/%ld\n",inmin.lat,inmin.lon);
  }

  if (verbose > 2) {
    for (i = 0; i < curseg; i++) {
      int j;
      fprintf(stderr,
	      "seg[%d] id=%d class=%c%d name=%s %s %s %s from=%d/%d to=%d/%d\n",
	      i,segs[i].tlid,segs[i].feat.class,segs[i].feat.level,
	      segs[i].name.fedirp,segs[i].name.fename,segs[i].name.fetype,
	      segs[i].name.fedirs,segs[i].from.lat,segs[i].from.lon,
	      segs[i].to.lat,segs[i].to.lon);
      fprintf(stderr,"Way points (%d to %d):",segs[i].first,segs[i].last);
      for (j = segs[i].first; j >= 0 && j <= segs[i].last; j++) {
	if (j%5 == 0)
	  fprintf(stderr,"\n\t");
	fprintf(stderr," %d/%d",Pts[j].lat,Pts[j].lon);
      }
      fprintf(stderr,"\n");
    }
  }
}

/* GNU fgets() on Linux seems to consider \r a newline!  Jerks... */
char *
mygets(s,l,f)
char *s;
int l;
FILE *f;
{
  int c;
  char *cp;

  for (cp = s, c = fgetc(f);
       l > 0 && c != EOF && c != '\n'; 
       --l,c = fgetc(f))
    *cp++ = c;

  if (l > 0)
    *cp = '\0';
  return (c == EOF)?NULL:s;
}

/*
 * Simple hash function to speed up tlid and lat/lon searches.  Actually
 * three hash tables in one (TLID, FROM, TO).  
 * The TLID hash "table" (actually the "->nh" links) is useful for finding
 * segs by TLID.
 * The FROM and TO "tables" (->fnh and ->tnh links) are useful for finding
 * colocated segments since all identical Lat/lons will hash to the same
 * chain (duh).
 */

#define SHASHSIZE 0xFFF		/* must share the hashsize for all 3 lists */
#define TLIDHASH(x) ((x)&SHASHSIZE)
#define SSHASH(x) (((x)->from.lat+(x)->from.lon+(x)->to.lat+(x)->to.lon)&SHASHSIZE)
#define LLHASH(x) (((x).lat+(x).lon)&SHASHSIZE)

struct sshash *ss_hash = NULL;	/* single-sided hash buckets */
struct seg *seghash = NULL;	/* all other hash buckets */

init_seg_hash()			/* alloc up the table of hash chain anchors */
{
  ss_hash = (struct sshash *)calloc(SHASHSIZE+1,sizeof(struct sshash));
  seghash = (struct seg *)calloc(SHASHSIZE+1,sizeof(struct seg));
  if (seghash == NULL || ss_hash == NULL) {
    fprintf(stderr,"couldn't calloc seghash!\n");
    exit(1);
  }
}


struct seg *
find_tl(tl)
u_long tl;
{
  static struct seg *lastseg = NULL;
  static u_long last_tl = 0;
  struct seg *hl;

  if ((last_tl == tl) && lastseg)
    return lastseg;
  for (hl = seghash[TLIDHASH(tl)].tlns; hl; hl = hl->tlns) {
    if (hl->tlid == tl) {
      last_tl = tl;
      return (lastseg = hl);
    }
  }
  return NULL;
}

add_seg_hash(segidx)		/* add a seg to the hash table */
int segidx;
{
  struct seg *s = &segs[segidx];
  int tlidx = TLIDHASH(s->tlid);
  int fromidx = LLHASH(s->from);
  int toidx = LLHASH(s->to);

  if (!seghash)
    init_seg_hash();

  if (find_tl(s->tlid)) {	/* don't add duplicates */
    if (verbose > 2)
      fprintf(stderr,"ignoring segs[%0x] duplicate for tlid %d\n",s,s->tlid);
    return;
  }
  s->tlns = seghash[tlidx].tlns; /* add to the TLID list */
  seghash[tlidx].tlns = s;

  s->frns = seghash[fromidx].frns; /* add to FROM list */
  seghash[fromidx].frns = s;

  s->tons = seghash[toidx].tons; /* add to TO list */
  seghash[toidx].tons = s;

  if (s->flags&SEG_SS) {	/* only if single sided do we */
    int ssidx = SSHASH(s);	/* add to single-sided search list */
    struct sshash *hp = (struct sshash *)calloc(1,sizeof(struct sshash));
    if (hp == NULL) {
      fprintf(stderr,"couldn't calloc hash table entry!\n");
      exit(1);
    }
    hp->s = s;
    hp->nh = ss_hash[ssidx].nh; 
    ss_hash[ssidx].nh = hp;
  }
}

struct seg *
find_tlid(tlid)			/* return ptr to seg w/tlid if any. */
char *tlid;
{
  struct tgr1 *t1;
  u_long tl = strntoul(tlid,10,sizeof(t1->tlid));

  return find_tl(tl);
}

/*
 * Gee, algebra wasn't always this hard for me.  I'm losing my
 * marbles.  Too much RF I guess.
 *
 * Givens: 
 * 
 * (ll1.lat,ll1.lon) = one endpoint of segment
 * (ll2.lat,ll2.lon) = other endpoint of segment
 * max.lat = northernmost latitude (top edge of map)
 * min.lat = southernmost latitude (bottom edge)
 * max.lon = westernmost longitude (left edge)
 * min.lon = easternmost longitude (right edge)
 * 
 * Solve for:
 * 
 * top = top intercept longitude (max.lat,top)
 * bot = bottom intercept longitude (min.lat,bot)
 * lft = left intercept latitude (lft,max.lon)
 * rgt = right intercept latitude (rgt,min.lon)
 *
 * ... and then ... remember that these lines are of finite length!
 * 
 * ll2.lat-ll1.lat   ll2.lat-max.lat            
 * --------------- = --------------- 
 * ll2.lon-ll1.lon     ll2.lon-top                    
 * 
 *            (ll2.lat-max.lat)(ll2.lon-ll1.lon)
 * => top = - ---------------------------------- - ll2.lon
 *                   (ll2.lat-ll1.lat)
 * 
 * ........................................................
 * 
 * ll2.lat-ll1.lat   ll2.lat-min.lat
 * --------------- = ----------------- 
 * ll2.lon-ll1.lon     bot-ll2.lon
 * 
 *            (ll2.lat-min.lat)(ll2.lon-ll1.lon)
 * => bot =   ---------------------------------- + ll2.lon
 *                    (ll2.lat-ll1.lat)
 * 
 * ........................................................
 * 
 * ll2.lat-ll1.lat      ll2.lat-rgt
 * --------------- =  ---------------
 * ll2.lon-ll1.lon    ll2.lon-min.lon
 * 
 * 
 *             (ll2.lat-ll1.lat)(ll2.lon-min.lon)
 * => rgt = -  ---------------------------------- + ll2.lat
 *                     (ll2.lon-ll1.lon)
 *                    
 * ........................................................
 * 
 * ll2.lat-ll1.lat      lft-ll2.lat
 * --------------- =  ---------------
 * ll2.lon-ll1.lon    max.lon-ll2.lon
 * 
 *           (ll2.lat-ll1.lat)(max.lon-ll2.lon)
 * => lft =  ---------------------------------- + ll2.lat
 *                   (ll2.lon-ll1.lon)
 * ........................................................
 * 
 * Some speedups that prevent core dumps (-: 
 *    if ll2.lon == ll1.lon, then the line is vertical and
 *      top = bot = ll2.lon = ll1.lon
 *      *and* rgt and lft are infinite
 *      
 *    if ll2.lat == ll1.lat, then the line is horizontal and
 *      rgt = lft = ll2.lat = ll1.lat
 *      *and* top and bot are infinite
 * 
 * If any 2 of the following are true, then the line crosses the map
 * and we now know the intercepts:
 *   top between min.lon & max.lon
 *   bot between min.lon & max.lon
 *   lft between min.lat & max.lat
 *   rgt between min.lat & max.lat
 */

#define INF -1

int
toplon(m,ll1,ll2)
struct map *m;
struct latlon *ll1,*ll2;
{
  int lon2_lon1 = ll2->lon - ll1->lon;
  int lat2_lat1 = ll2->lat - ll1->lat;
  struct latlon r;

  if (lon2_lon1 == 0)
    r.lon = ll2->lon;
  else if (lat2_lat1 == 0)
    return INF;
  else 
    r.lon = (ll2->lat-m->max.lat)*(lon2_lon1)/(lat2_lat1) - ll2->lon;
  r.lat = max.lat;
  return (between(&r,ll1,ll2))?r.lon:INF;
}

int
botlon(m,ll1,ll2)
struct map *m;
struct latlon *ll1,*ll2;
{
  int lon2_lon1 = ll2->lon - ll1->lon;
  int lat2_lat1 = ll2->lat - ll1->lat;
  struct latlon r;

  if (lon2_lon1 == 0)
    r.lon = ll2->lon;
  else if (lat2_lat1 == 0)
    return INF;
  else 
    r.lon = (ll2->lat-m->min.lat)*(lon2_lon1)/(lat2_lat1) + ll2->lon; 
  r.lat = min.lat;
  return (between(&r,ll1,ll2))?r.lon:INF;
}

int
rgtlat(m,ll1,ll2)
struct map *m;
struct latlon *ll1,*ll2;
{
  int lon2_lon1 = ll2->lon - ll1->lon;
  int lat2_lat1 = ll2->lat - ll1->lat;
  struct latlon r;

  if (lat2_lat1 == 0)
    r.lat = ll2->lat;
  else if (lon2_lon1 == 0)
    return INF;
  else
    r.lat = (lat2_lat1)*(ll2->lon-m->min.lon)/(lon2_lon1) + ll2->lat;
  r.lon = min.lon;
  return (between(&r,ll1,ll2))?r.lat:INF;
}

int
lftlat(m,ll1,ll2)
struct map *m;
struct latlon *ll1,*ll2;
{
  int lon2_lon1 = ll2->lon - ll1->lon;
  int lat2_lat1 = ll2->lat - ll1->lat;
  struct latlon r;

  if (lat2_lat1 == 0)
    r.lat = ll2->lat;
  else if (lon2_lon1 == 0)
    return INF;
  else
    r.lat = (lat2_lat1)*(m->max.lon-ll2->lon)/(lon2_lon1) + ll2->lat;
  r.lon = max.lon;
  return (between(&r,ll1,ll2))?r.lat:INF;
}

between(p,ll1,ll2)
struct latlon *p;		/* is this point */
struct latlon *ll1,*ll2;	/* these two points? */
{
  return 
    (p->lat >= ll1->lat && p->lat <= ll2->lat 
     && p->lon >= ll1->lon && p->lon <= ll2->lon) ||
    (p->lat >= ll2->lat && p->lat <= ll1->lat
     && p->lon >= ll2->lon && p->lon <= ll1->lon);
}

crosses(m,ll1,ll2)		/* test: does this segment cross the map? */
struct map *m;
struct latlon *ll1,*ll2;
{
  int lft,rgt,top,bot;		/* intercepts */

  if (isvis(m,ll1) || isvis(m,ll2)) /* any point actually in the map? */
    return 1;

  /* If any one edge is intercepted (then another must be as well but we
     don't care which one at this point): The segment crosses the map. */
  if ((lft = lftlat(m,ll1,ll2)) >= m->min.lat && lft <= m->max.lat) {
    if (1 || verbose > 3)
      fprintf(stderr,"crossing %d/%d to %d/%d on left edge at lat %d\n",
	      ll1->lat,ll1->lon,ll2->lat,ll2->lon,lft);
    return 1;
  }
  if ((rgt = rgtlat(m,ll1,ll2)) >= m->min.lat && rgt <= m->max.lat) {
    if (1 || verbose > 3)
      fprintf(stderr,"crossing %d/%d to %d/%d on right edge at lat %d\n",
	      ll1->lat,ll1->lon,ll2->lat,ll2->lon,rgt);
    return 1;
  }
  if ((top = toplon(m,ll1,ll2)) >= m->min.lon && top <= m->max.lon) {
    if (1 || verbose > 3)
      fprintf(stderr,"crossing %d/%d to %d/%d on top edge at lon %d\n",
	      ll1->lat,ll1->lon,ll2->lat,ll2->lon,top);
    return 1;
  }
  if ((bot = botlon(m,ll1,ll2)) >= m->min.lon && bot <= m->max.lon) {
    if (1 || verbose > 3)
      fprintf(stderr,"crossing %d/%d to %d/%d on bottom edge at lon %d\n",
	      ll1->lat,ll1->lon,ll2->lat,ll2->lon,bot);
    return 1;
  }
  return 0;			/* doesn't cross the map */
}

invisible(m)			/* filter out invisible (off-map) segs */
struct map *m;
{
  int i,j,invis,vis;
  struct latlon prevLL;

  vis = invis = 0;
  for (i=0; i < curseg; i++) {
    if (verbose > 3)
      fprintf(stderr,"invisible: testing segs[%d] tlid %ld\n",i,segs[i].tlid);
    /* do easy test first to see if From or To point is in the map... */
    if (isvis(m,&segs[i].from) || isvis(m,&segs[i].to)) {
      segs[i].flags &= ~SEG_SKIP; /* turn off skip flag */
      ++vis;
      continue;
    }
    /* A little harder: 1st see if any point is in the map.  If not,
       try to see if any adjacent pair defines a line that crosses
       the map.  (Don't forget to include the from and to points.)*/
    for (prevLL=segs[i].from, j = segs[i].first;
	 j >= 0 && j < segs[i].last; j++) {
      if (isvis(m,&Pts[j]) || crosses(m,&prevLL,&Pts[j])) {
	++vis;
	segs[i].flags &= ~SEG_SKIP; /* turn off skip flag */
	break;
      }
      prevLL = Pts[j];
    }
    if (vis == 0 && crosses(m,&prevLL,&segs[i].to)) {
      ++vis;
      segs[i].flags &= ~SEG_SKIP; /* turn off skip flag */
    }
    if (vis == 0) {		/* finally give up on this seg */
      if (verbose > 3)
	fprintf(stderr,"invisible: skipping segs[%d] tlid %ld\n",i,segs[i].tlid);
      segs[i].flags |= SEG_SKIP;
      ++invis;
    }
  }
  return invis;
}

int
isvis(m,ll)			/* is point within map window? */
struct map *m;
struct latlon *ll;
{
  int r = ((ll->lat <= m->max.lat) && (ll->lat >= m->min.lat)
	   && (ll->lon <= m->max.lon) && (ll->lon >= m->min.lon));
  if (verbose > 3)
    fprintf(stderr,"isvis=%d ll=%d/%d max=%d/%d min=%d/%d\n",
	    r,ll->lat,ll->lon,m->max.lat,m->max.lon,m->min.lat,m->min.lon);
  return r;
}

/* kill overlapping single-sided chains.  The &*(% census repeats county
   boundary chains in each county file, so when you concatenate them, you
   have to search for the overlaps (by matching to/from I guess) */
kill_dups()			
{
  int killed = 0;
  int i;
  struct sshash *h1,*h2;

  if (verbose)
    fprintf(stderr,"Eliminating duplicate single-sided chains...");

  /* search the single-sided hash chain for hits */
  for (i = 0; i <= SHASHSIZE; i++) { /* for each of the buckets... */
    for (h1 = ss_hash[i].nh; h1 && h1->nh; h1 = h1->nh) {
      if (h1->s->flags&SEG_DUP) /* already been marked as dupe */
	continue;
      for (h2 = h1->nh; h2; h2 = h2->nh) { /* search for my twin */
	if (h1->s->feat.class == h2->s->feat.class
	    && h1->s->feat.level == h2->s->feat.level
	    && h1->s->from.lat == h2->s->from.lat
	    && h1->s->to.lat == h2->s->to.lat) {
	  h2->s->flags |= SEG_DUP;
	  ++killed;
	  break;
	}
      } /* ... search for my twin */
    } /* ... walk the chain for this bucket */
  } /* ... for each of the buckets */
  if (verbose)
    fprintf(stderr," %d found.\n",killed);
}

struct road anchor;
struct road *find_road_from(struct seg *);
struct road *find_road_to(struct seg *);
struct road *find_flipped_road_to(struct seg *);
struct road *find_flipped_road_from(struct seg *);

water_filter(m)			/* filter out "small" (water) features */
struct map *m;
{
  int killed = 0;
  struct road *r;

  if (m->water_filt < 0)	/* filtering is disabled */
    return 0;

  for (r = anchor.nr; r; r = r->nr) { /* loop over roads...*/
    struct latlon maxi = {-999999999,-999999999};
    struct latlon mini = {999999999,999999999};
    struct seg *l;
    int lat,lon,width,maplat,maplon,mapwidth;
    if (r->head->feat.class != 'H')
      continue;			/* ignore non-water */
    for (l = r->head; l && l->flags&SEG_IGN; l = l->ns)
      ;	/* skip ignored segments */
    for (; l; l = l->ns) {
      int i,j,n,direction;
      find_maxmin(&maxi,&mini,&l->from);
      n=abs(l->first-l->last);
      direction=(l->flags&SEG_REV)?-1:1;
      for (i = 0, j = l->first; j >= 0 && i < n; j += direction,++i) {
	find_maxmin(&maxi,&mini,&Pts[j]);
      }
      find_maxmin(&maxi,&mini,&l->to);
    }
    lat=maxi.lat-mini.lat;
    lon=maxi.lon-mini.lon;
    width=(lat>lon)?lat:lon;
    maplat=m->max.lat-m->min.lat;
    maplon=m->max.lon-m->min.lon;
    mapwidth=(maplat>maplon)?maplat:maplon;
    if (verbose > 2)
      fprintf(stderr,"road %d (%d) is %d which is %4.2f%% of map's %d\n",
	      r->roadid,r->head->tlid,width,
	      100.0*(double)width/(double)mapwidth,mapwidth);
    if (((double)width/(double)mapwidth) < m->water_filt) {
      r->flags |= RD_SKIP;
      ++killed;
    } else {
      r->flags &= ~RD_SKIP;
    }
  }
  return killed;
}



void
type1(t1)
struct tgr1 *t1;
{
  struct seg *s;

  if (!segs) {
    segs = (struct seg *) calloc(maxseg=SEGS,sizeof(struct seg));
    if (!segs) {
      fprintf(stderr,"calloc of segs failed!\n");
      exit(1);
    }
    curseg = 0;
  }
  if (curseg+1 >= maxseg) {
    struct seg *news = 
      (struct seg *)realloc(segs,(maxseg+=SEGS)*sizeof(struct seg));
    if (news)
      segs = news;
    else {
      maxseg-=SEGS;
      fprintf(stderr,"realloc of segs failed, ignoring additional segs.\n");
    }
  }
  if (!*tigver) {
    if (strncmp(t1->version,"0024",4) == 0)
      strcpy(tigver,"1995");
    else if (strncmp(t1->version,"0021",4) == 0)
      strcpy(tigver,"1994");
    else
      sprintf(tigver,"v%4.4s",t1->version);
    if (verbose)
      fprintf(stderr,"TIGER/Line files version: %s\n",tigver);
  }
  s = &segs[curseg];

  bzero(s,sizeof(*s));		/* clean out any garbage */
  if (verbose > 2)
    fprintf(stderr,"id: %10.10s CFCC %3.3s name: %2.2s %30.30s %4.4s %2.2s\n",
	    t1->tlid,t1->cfcc,t1->fedirp,t1->fename,t1->fetype,t1->fedirs);

  s->feat.class = *t1->cfcc;
  s->feat.level = (t1->cfcc[1]-'0')*10 + (t1->cfcc[2]-'0');
  strtrim(s->name.fedirp,t1->fedirp,sizeof(s->name.fedirp)-1);
  strtrim(s->name.fename,t1->fename,sizeof(s->name.fename)-1);
  strtrim(s->name.fetype,t1->fetype,sizeof(s->name.fetype)-1);
  strtrim(s->name.fedirs,t1->fedirs,sizeof(s->name.fedirs)-1);
  if (*t1->side1 == '1')
    s->flags |= SEG_SS;
  s->from.lat = atoil(&t1->fr[LAT],LAT_LEN);
  s->from.lon = -atoil(&t1->fr[LON],LON_LEN);
  s->to.lat = atoil(&t1->to[LAT],LAT_LEN);
  s->to.lon = -atoil(&t1->to[LON],LON_LEN);
  find_maxmin(&inmax,&inmin,&s->from);
  find_maxmin(&inmax,&inmin,&s->to);
  if (!cut_keep(rootmap,s))	/* check to see if this record meets the cut */
    return;
  s->tlid = strntoul(t1->tlid,10,sizeof(t1->tlid));
  s->first = s->last = -1;	/* indicates not yet set */
  curseg++;
}

/*
 * hash the type 1 segs in one fell swoop and get tighter memory use
 * this way.
 */
void
end_type1()
{
  int i;

  if (verbose)
    fprintf(stderr,"%d segments loaded.\nbuilding segment hash indexes....\n",
	    curseg);
  init_seg_hash();		/* alloc up the seg hash tables now? */
  for (i = 0; i < curseg; i++)
    add_seg_hash(i);		/* add this seg to hash tables */
}

find_maxmin(maxi,mini,ll)
struct latlon *maxi,*mini,*ll;
{
  if (ll->lat > maxi->lat)
    maxi->lat = ll->lat;
  if (ll->lat < mini->lat)
    mini->lat = ll->lat;
  if (ll->lon > maxi->lon)
    maxi->lon = ll->lon;
  if (ll->lon < mini->lon)
    mini->lon = ll->lon;
}

/*
 * N.B. Assumes the rtsq's are in increasing order for a given tlid.
 * (as documented in the manual).  However, got burned for single-sided
 * chains since they reappear once for each county file, and with a huge
 * gap between them.  So, the quick and dirty fix is to mark a seg DONE
 * after the last way point for it from the first county has been seen.
 */

void
type2(t2)
struct tgr2 *t2;
{
  long i,lon,lat;
  struct seg *s;
  static struct seg *prev = NULL;

  if (verbose > 2)
    fprintf(stderr,"id: %10.10s #%3.3s",t2->tlid,t2->rtsq);
  if ((s = find_tlid(t2->tlid)) == NULL) { /* ignore tlid's we didn't select */
    if (verbose > 2)
      fprintf(stderr," ignored.\n");
    return;
  }
#ifdef DDD
  if (s->tlid == 114045538)
    verbose=3;
  else
    verbose=1;
#endif

  if (s->flags&SEG_DONE) {
    if (verbose > 2)
      fprintf(stderr," single-sided segment is already done.\n");
    return;
  }

  if (prev && prev != s) {
    if (verbose > 2)
      fprintf(stderr," ends segs[%0x] tlid %d.\n",prev,prev->tlid);
    prev->flags |= SEG_DONE;
  }
  
  if (s->first < 0) {		/* no first way point yet */
    s->first = curpt;		/* this will be the first way point */
    s->last = s->first;
  }
  if (!Pts) {
    Pts = (struct latlon *) calloc(maxpt=PTS,sizeof(struct latlon));
    curpt = 0;
    if (!Pts) {
      fprintf(stderr,"Couldn't calloc pts\n");
      exit(1);
    }
  } else if (curpt+1 >= maxpt) {
    struct latlon *newp = 
      (struct latlon *)realloc(Pts,(maxpt+=PTS)*sizeof(struct latlon));
    if (newp)
      Pts = newp;
    else {
      maxpt-=PTS;
      fprintf(stderr,"realloc of pts failed, ignoring additional pts.\n");
    }
  }
  if (verbose > 2)
    fprintf(stderr,"segs[%0x] tlid %s seq %3.3s 1st way %d, last so far %d\n",
	    s,t2->tlid,t2->rtsq,s->first,s->last);

  for (i = 0; i < 10; i++) {
    lon = -atoil(&t2->ll[i][LON],LON_LEN);
    lat = atoil(&t2->ll[i][LAT],LAT_LEN);
    if (lon == 0 || lat == 0) /* end of list */
      break;
    else {
      Pts[curpt].lon = lon;
      Pts[curpt].lat = lat;
      if (verbose > 2)
	fprintf(stderr,"\tsegs[%0x],Pts[%d] = %d/%d\n",s,curpt,lat,lon);
      s->last = curpt;
      ++curpt;
    }
  }
  if (verbose > 2)
    fprintf(stderr,"\tnow %d way points (%d is last)\n",
	    (s->last-s->first+1),curpt-1);
  prev = s;			/* remember the previous segment */
}

void
end_type2()
{
  if (verbose)
    fprintf(stderr,"%d points loaded.\n",curpt);
}

void 
typeP(tP)
struct tgrP *tP;
{
  struct poly *p;

  if (!polys) {
    polys = (struct poly *) calloc(maxpoly=POLYS,sizeof(struct poly));
    if (!polys) {
      fprintf(stderr,"calloc of polys failed!\n");
      exit(1);
    }
    curpoly = 0;
  } else if (curpoly+1 >= maxpoly) {
    struct poly *newp = 
      (struct poly *)realloc(polys,(maxpoly+=POLYS)*sizeof(struct poly));
    if (newp)
      polys = newp;
    else {
      maxpoly-=POLYS;
      fprintf(stderr,"realloc of polys failed, ignoring additional.\n");
    }
  }
  p = &polys[curpoly];

  bzero(p,sizeof(*p));		/* clean out any garbage */
  if (debug&DEBUG_POLY)
    fprintf(stderr,"cenid: %5.5s polyid %10.10s lat/lon %9.9s/%10.10s\n",
	    tP->cenid,tP->polyid,&tP->ctr[LAT],&tP->ctr[LON]);

  p->ctr.lat = atoil(&tP->ctr[LAT],LAT_LEN);
  p->ctr.lon = -atoil(&tP->ctr[LON],LON_LEN);
  if (!isvis(rootmap,&p->ctr))
      return;
  strncpy(p->cenid,tP->cenid,sizeof(tP->cenid));
  p->polyid = strntoul(tP->polyid,10,sizeof(tP->polyid));
  curpoly++;
}

#define PHASHSIZE 0xFFF
#define POLYHASH(x) ((x)&PHASHSIZE)

struct poly *polyhash = NULL;	/* all other hash buckets */

init_poly_hash()		/* alloc up the table of hash chain anchors */
{
  polyhash = (struct poly *)calloc(PHASHSIZE+1,sizeof(struct poly));
  if (polyhash == NULL) {
    fprintf(stderr,"couldn't calloc polyhash!\n");
    exit(1);
  }
}

add_poly_hash(polyidx)		/* add a poly to the hash table */
int polyidx;
{
  struct poly *p = &polys[polyidx];
  int pidx = POLYHASH(p->polyid);

  if (!polyhash)
    init_poly_hash();

  p->np = polyhash[pidx].np;	/* add to the polyid list */
  polyhash[pidx].np = p;
}

void
end_typeP()
{
  int i;

  if (verbose)
    fprintf(stderr,"%d polygon centers loaded.\nbuilding hash index....\n",
	    curpoly);
  init_poly_hash();		/* alloc up the poly hash tables now? */
  for (i = 0; i < curpoly; i++)
    add_poly_hash(i);		/* add this poly to hash tables */
}

struct poly *
find_poly(cenid,polyid)
char *cenid,*polyid;
{
  struct tgrP *tP;
  u_long pid = strntoul(polyid,10,sizeof(tP->polyid));
  struct poly *pp;
  static struct poly *lastpoly = NULL;
  static u_long last_pid;
  static char last_cid[sizeof(pp->cenid)];

  if (strncmp(last_cid,cenid,sizeof(last_cid)-1)==0 && last_pid==pid && lastpoly)
    return lastpoly;
  for (pp = polyhash[POLYHASH(pid)].np; pp; pp = pp->np) {
    if (strncmp(pp->cenid,cenid,sizeof(pp->cenid)-1)==0 && pp->polyid == pid) {
      strncpy(last_cid,cenid,sizeof(last_cid));
      last_pid = pid;
      return (lastpoly = pp);
    }
  }
  return NULL;
}

void 
typeC(tC)
struct tgrC *tC;
{
  struct fips *f;

  if (!fipscodes) {
    fipscodes = (struct fips *) calloc(maxfips=FIPS,sizeof(struct fips));
    if (!fipscodes) {
      fprintf(stderr,"calloc of fipscodes failed!\n");
      exit(1);
    }
    curfips = 0;
  } else if (curfips+1 >= maxfips) {
    struct fips *newf = 
      (struct fips *)realloc(fipscodes,(maxfips+=FIPS)*sizeof(struct fips));
    if (newf)
      fipscodes = newf;
    else {
      maxfips-=FIPS;
      fprintf(stderr,"realloc of fipscodes failed, ignoring additional.\n");
    }
  }
  f = &fipscodes[curfips];

  bzero(f,sizeof(*f));		/* clean out any garbage */
  f->fips = atoil(tC->fips,sizeof(tC->fips));
  if (!f->fips)
    return;
  f->classcode[0] = tC->fipscc[0];
  f->classcode[1] = tC->fipscc[1];
  strtrim(f->name,tC->name,sizeof(f->name));
  curfips++;
  if (verbose > 2)
    fprintf(stderr,"fips code %d, class %2.2s name %s\n",
	    f->fips,tC->fipscc,f->name);
}

#define FHASHSIZE 0xFFF
#define FIPSHASH(x) ((x)&FHASHSIZE)

struct fips *fipshash = NULL;

init_fips_hash()		/* alloc up the table of hash chain anchors */
{
  fipshash = (struct fips *)calloc(FHASHSIZE+1,sizeof(struct fips));
  if (fipshash == NULL) {
    fprintf(stderr,"couldn't calloc fipshash!\n");
    exit(1);
  }
}

add_fips_hash(fipsidx)		/* add a label to the hash table */
int fipsidx;
{
  struct fips *f = &fipscodes[fipsidx];
  int pidx = FIPSHASH(f->fips);

  if (!fipshash)
    init_fips_hash();

  f->pnl = fipshash[pidx].pnl;	/* add to the place list */
  fipshash[pidx].pnl = f;
}

void
end_typeC()
{
  int i;

  if (verbose)
    fprintf(stderr,"%d fips codes loaded.\nbuilding hash index....\n",
	    curfips);
  init_fips_hash();		/* alloc up the fips hash tables now */
  for (i = 0; i < curfips; i++)
    add_fips_hash(i);		/* add this label to hash tables */
}

struct fips *
find_fips(fips)
int fips;
{
  struct fips *fl;
  static struct fips *lastf = NULL;
  static int last_fips;

  if (last_fips == fips)
    return lastf;
  for (fl = fipshash[FIPSHASH(fips)].pnl; fl; fl = fl->pnl) {
    if (fl->fips == fips) {
      last_fips = fips;
      return (lastf = fl);
    }
  }
  return NULL;
}

struct area *
find_area(a,fips)		/* find area w/given fips code */
struct area *a;
int fips;
{
  for (; a; a = a->na)
    if (a->fips == fips)
      return a;
  return NULL;
}

struct area *
add_area(a,fips)		/* add new fips code to given area list */
struct area *a;
int fips;
{
  struct area *newa = (struct area *)calloc(1,sizeof(struct area));
  struct fips *f;

  if (!newa) {
    fprintf(stderr,"Couldn't calloc an area\n");
    return NULL;
  }
  newa->fips = fips;
  if (f = find_fips(fips)) {
    newa->classcode = f->classcode;
    newa->name = f->name;
  }
  newa->na = a->na;		/* insert into list */
  a->na = newa;
  return newa;  
}

/* lists for each area type we care about */
static struct area areas[4] = {{0,0,0},{0,0,0},{0,0,0},{0,0,0}};

#define CITY 0
#define MCD 1
#define SMCD 2
#define PLACE 3

void
typeS(tS)
struct tgrS *tS;
{
  struct poly *p;
  int code[4],i;

  bzero(code,sizeof(code));
  if (debug&DEBUG_AREA)
    fprintf(stderr,"typeS cenid: %5.5s %10.10s",tS->cenid,tS->polyid);
  if ((p = find_poly(tS->cenid,tS->polyid)) == NULL) {
    if (debug&DEBUG_AREA)
      fprintf(stderr," ignored.\n");
    return;
  }
  if (debug&DEBUG_AREA)
    fprintf(stderr," water=%c,city=%5.5s,mcd=%5.5s,smcd=%5.5s,pl=%5.5s\n",
	    tS->water[0],tS->fccity,tS->fmcd,tS->fsmcd,tS->fpl);
  code[CITY] = atoil(tS->fccity,sizeof(tS->fccity));
  code[MCD] = atoil(tS->fmcd,sizeof(tS->fmcd));
  code[SMCD] = atoil(tS->fsmcd,sizeof(tS->fsmcd));
  code[PLACE] = atoil(tS->fpl,sizeof(tS->fpl));
  for (i = 0; i < sizeof(code)/sizeof(code[0]); i++) {
    if (code[i]) {		/* non-blank fips-55 code */
      struct arealink *l = 
	(struct arealink *)calloc(1,sizeof(struct arealink));
      struct area *a = find_area(&areas[i],code[i]);

      if (!l) {
	fprintf(stderr,"failed to calloc arealink\n");
	exit(1);
      }
      if (!a)
	a = add_area(&areas[i],code[i]);

      l->p = p;			/* point at the poly */
      l->nal = a->link;		/* insert into this area's link list */
      a->link = l;
    }
  }
}

void 
type7(t7)
struct tgr7 *t7;
{
  struct land *l;

  if (!landmarks) {
    landmarks = (struct land *) calloc(maxland=LANDS,sizeof(struct land));
    if (!landmarks) {
      fprintf(stderr,"calloc of landmarks failed!\n");
      exit(1);
    }
    curland = 0;
  } else if (curland+1 >= maxland) {
    struct land *newl = 
      (struct land *)realloc(landmarks,(maxland+=LANDS)*sizeof(struct land));
    if (newl)
      landmarks = newl;
    else {
      maxland-=LANDS;
      fprintf(stderr,"realloc of landmarks failed, ignoring additional.\n");
    }
  }
  l = &landmarks[curland];

  bzero(l,sizeof(*l));		/* clean out any garbage */
  l->cfcc.class = *t7->cfcc;
  l->cfcc.level = (t7->cfcc[1]-'0')*10 + (t7->cfcc[2]-'0');
  if (!feat_keep(&l->cfcc))
    return;
  l->landid = atoil(t7->land,sizeof(t7->land));
  if (!l->landid)
    return;
  l->state = atoil(t7->state,sizeof(t7->state));
  l->county = atoil(t7->county,sizeof(t7->county));
  strtrim(l->name,t7->laname,sizeof(l->name)-1);
  l->u.ctr.lat = atoil(&t7->la[LAT],LAT_LEN);
  l->u.ctr.lon = -atoil(&t7->la[LON],LON_LEN);
  if (l->u.ctr.lat) {
    if (!isvis(rootmap,&l->u.ctr))
      return;
    l->flags |= LAND_POINT;
  }
  curland++;
  if (verbose > 2)
    fprintf(stderr,"landmark %d, cfcc %c%2.2d name %s\n",
	    l->landid,l->cfcc.class,l->cfcc.level,l->name);
}

#define LHASHSIZE 0xFF
#define LANDHASH(x) ((x)&LHASHSIZE)

struct land *landhash = NULL;

init_land_hash()		/* alloc up the table of hash chain anchors */
{
  landhash = (struct land *)calloc(LHASHSIZE+1,sizeof(struct land));
  if (landhash == NULL) {
    fprintf(stderr,"couldn't calloc landhash!\n");
    exit(1);
  }
}

add_land_hash(landidx)		/* add a label to the hash table */
int landidx;
{
  struct land *l = &landmarks[landidx];
  int pidx = LANDHASH(l->landid);

  if (!landhash)
    init_land_hash();

  l->nl = landhash[pidx].nl;	/* add to the hash list */
  landhash[pidx].nl = l;
}

void
end_type7()
{
  int i;

  if (verbose)
    fprintf(stderr,"%d landmarks loaded.\nbuilding hash index....\n",
	    curland);
  init_land_hash();		/* alloc up the land hash tables now */
  for (i = 0; i < curland; i++)
    add_land_hash(i);		/* add this label to hash tables */
}

struct land *
find_land(state,county,land)	/* return ptr to landmark if any. */
int state,county,land;
{
  struct land *ll;
  static int last_st = 0,last_co = 0 ,last_la = 0;
  static struct land *last_ll = NULL;

  if (last_ll && last_st == state && last_co == county && last_la == land)
    return last_ll;
  for (ll = landhash[LANDHASH(land)].nl; ll; ll = ll->nl) {
    if (ll->landid == land && ll->state == state && ll->county == county) {
      last_st = state;
      last_co = county;
      last_la = land;
      return ll;
    }
  }
  return NULL;
}

void
type8(t8)
struct tgr8 *t8;
{
  int state,county,land;
  struct land *l;

  state=atoil(t8->state,sizeof(t8->state));
  county=atoil(t8->county,sizeof(t8->county));
  land=atoil(t8->land,sizeof(t8->land));
  if (l = find_land(state,county,land)) {
    struct poly *p = find_poly(t8->cenid,t8->polyid);
    if (p) {
      struct arealink *al = 
	(struct arealink *)calloc(1,sizeof(struct arealink));

      if (!al) {
	fprintf(stderr,"failed to calloc arealink\n");
	exit(1);
      }
      al->p = p;		/* point at the poly */
      al->nal = l->u.link;	/* insert into this area's link list */
      l->u.link = al;		/* which is hung off this landmark */
    }
  }
}

/* key geographic locations (or, "Where's the shopping mall?") */

static struct kgl kgls;

void
type9(t9)
struct tgr9 *t9;
{
  struct poly *p;
  int nlen,polyid;
  struct kgl *k;
  struct arealink *al;
  char buf[31];
  struct feat cfcc;

  if (debug&DEBUG_KGL)
    fprintf(stderr,"type9 cenid: %5.5s %10.10s",t9->cenid,t9->polyid);
  if ((p = find_poly(t9->cenid,t9->polyid)) == NULL) {
    if (debug&DEBUG_KGL)
      fprintf(stderr," ignored.\n");
    return;
  }
  cfcc.class = *t9->cfcc;
  cfcc.level = (t9->cfcc[1]-'0')*10 + (t9->cfcc[2]-'0');
  if (!feat_keep(&cfcc)) {
    if (debug&DEBUG_KGL)
      fprintf(stderr," ignored.\n");
    return;
  }
  if (debug&DEBUG_KGL)
    fprintf(stderr," cfcc=%3.3s\n",t9->cfcc);
  polyid = atoil(t9->polyid,sizeof(t9->polyid));
  k = (struct kgl *)calloc(1,sizeof(struct kgl));
  al = (struct arealink *)calloc(1,sizeof(struct arealink));
  if (!al || !k) {
    fprintf(stderr,"failed to calloc arealink and/or kgl\n");
    exit(1);
  }
  k->nk = kgls.nk;		/* insert at head of kgl list */
  kgls.nk = k;
  strtrim(k->name,t9->kglname,sizeof(t9->kglname));
  k->cfcc = cfcc;
  k->link = al;			/* the one and only arealink for this KGL */
  al->p = p;			/* point at the poly */
  al->nal = 0;			/* only one arealink for a KGL */
}

do_args(argc,argv)
int argc;
char **argv;
{
  extern int optind;
  extern char *optarg;
  char *lonptr;
  int c;
  int argerr = 0;
  int i;

  for (i = 1, *cmdline='\0'; i < argc; i++) { /* save cmdline to document */
    strncat(cmdline,argv[i],sizeof(cmdline)-1);
    strncat(cmdline," ",sizeof(cmdline)-1);
  }

  while ((c=getopt(argc,argv,"vjnToaCDc:r:f:F:t:p:M:l:d:w:W:s:S:R:L:g:")) != EOF)
    switch(c) {
    case 'v':
      ++verbose;
      break;
    case 'j':			/* join road chains be flipping them */
      ++join;
      break;
    case 'n':			/* match features by name as well as CFCC */
      ++fename_match;
      name_eq_global = name;	/* initially match on names */
      break;
    case 'C':			/* match features by resulting APRS color */
      ++color_match;
      break;
    case 'D':
      ++data_file;		/* print a *.DAT file before fuzzing maps */
      break;
    case 'T':			/* print all road's segment TLIDs in comment */
      ++all_tlid;
      break;
    case 'o':			/* make two more overlapping maps */
      ++overlap;
      break;
    case 'w':
      water_filt = strtod(optarg,NULL)/100.0;
      break;
    case 'W':
      max_filt = strtod(optarg,NULL)/100.0;
      break;
    case 'c':			/* map center lat/lon in decimal degrees */
      ctr.lat = (int)(1000000*strtod(optarg,&lonptr));
      ++lonptr;			/* skip over the comma */
      ctr.lon = (int)(1000000*strtod(lonptr,NULL));
      break;
    case 'r':			/* radius for map(s) in miles */
      range = strtod(optarg,NULL);
      break;
    case 'd':			/* level of detail cutoffs wanted (miles?) */
      detail = strtod(optarg,NULL);
      break;
    case 't':			/* recursively subdivide if too many points */
      tile = strtod(optarg,NULL); /* down to this radius and no further */
      break;
    case 'f':			/* fuzz factor for (x,y)==(x1,y1) test */
      fuzz = atoi(optarg);
      break;
    case 'F':			/* max fuzz factor for (x,y)==(x1,y1) test */
      maxfuzz = atoi(optarg);
      break;
    case 's':
      slopefuzz = strtod(optarg,NULL); /* used for smoothing straight lines */
      break;
    case 'S':
      maxslope = strtod(optarg,NULL); /* no fuzzier than this */
      break;
    case 'p':			/* filename prefix for map files */
      namepfx = optarg;
      break;
    case 'M':
      maxpts = atoi(optarg);	/* override the default of 3000 */
      break;
    case 'R':
      pixper = atoi(optarg);	/* override the default */
      break;
    case 'l':			/* filename for maplist file */
      strncpy(maplistname,optarg,sizeof(maplistname));
      break;
    case 'm':
      maplistmax = atoi(optarg); /* max number of entries in a maplist */
      break;
    case 'L':
      if (strcmp(optarg,"places") == 0)
	++label_places;
      else if (strcmp(optarg,"landmarks") == 0)
	++label_landmarks;
      else if (strcmp(optarg,"kgls") == 0)
	++label_kgls;
      break;
    case 'a':			/* all the usual suspects */
      ++verbose;
      ++join;
      ++fename_match;
      name_eq_global = name;
      ++color_match;
      water_filt=.05;
      max_filt=.10;
      slopefuzz=.05;
      maxslope=.2;
      ++label_places;
      break;
    case 'g':
      if (strcmp(optarg,"label") == 0)
	debug |= DEBUG_LABELS;
      else if (strcmp(optarg,"poly") == 0)
	debug |= DEBUG_POLY;
      else if (strcmp(optarg,"area") == 0)
	debug |= DEBUG_AREA;
      else if (strcmp(optarg,"kgl") == 0)
	debug |= DEBUG_KGL;
      else if (strcmp(optarg,"fuzz") == 0)
	debug |= DEBUG_FUZZ;
      break;
    default:
      argerr++;
      break;
    }
  if (argerr || optind < argc) {
    fprintf(stderr,"Usage: %s [-vjnoaTD] -c lat,lon -r range -d detail -t min_tile\n",
	    argv[0]);
    fprintf(stderr,"       -f fuzz -F maxfuzz -w filt%% -W maxfilt%% -p map_prefix\n");
    fprintf(stderr,"       -s slopefuzz -F maxslope -l maplist -m maps_per_list -M maxAPRSpoints\n");
    fprintf(stderr,"       -L places|landmarks|kgls -g debug_opt\n");
    fprintf(stderr," -v = verbose\n");
    fprintf(stderr," -j = flip and join road segments\n");
    fprintf(stderr," -n = match segments by name as well as CFCC\n");
    fprintf(stderr," -o = make three overlapping maps\n");
    fprintf(stderr," -a = all the usual (same as -vjnC -w .05 -W .1 -s .05 -S .2)\n");
    fprintf(stderr," -T = print all a road's TLIDs in comment\n");
    fprintf(stderr," -D = create a *.dat file before fuzzing a map\n");
    fprintf(stderr," -c = map center lat,lon in decimal degrees\n");
    fprintf(stderr," -r = map radius in miles\n");
    fprintf(stderr," -d = level of map detail in miles\n");
    fprintf(stderr," -t = make tiles no smaller than this radius in miles\n");
    fprintf(stderr," -f = initial map fuzziness (reducing # of points)\n");
    fprintf(stderr," -F = worst map fuzziness\n");
    fprintf(stderr," -w = toss lakes smaller than x%% of map\n");
    fprintf(stderr," -W = worst lake fuzz%%\n");
    fprintf(stderr," -s = line smoothing factor\n");
    fprintf(stderr," -S = worst line smoothing factor\n");
    fprintf(stderr," -p = filename prefix for map files (default is 'map')\n");
    fprintf(stderr," -l = maplist filename\n");
    fprintf(stderr," -m = max map names per maplist before splitting up\n");
    fprintf(stderr," -M = max APRS/DOS map points (default 2999)\n");
    fprintf(stderr," -R = resolution (a/k/a pixels per degree)\n");
    fprintf(stderr," -L = label places, landmarks, or key geographic locations (kgls)\n");
    fprintf(stderr," -g = debug poly, kgl, area, label, fuzz\n");
    
    exit(1);
  }
}

struct cuts {
  int lo,hi;			/* lo/hi values */
  int fl;			/* flags */
#define NN 1			/* cut unnamed features */
#define SS 2			/* keep single-sided (county bdry) only */
#define FF 4			/* fuzz this feature faster than others */
#define FS 8			/* fuzz segment begin/end points too */
};

#define X {-1,-1,0}

struct cuts A1[] = {{0,99,0},X}; /* roads */
struct cuts B1[] = {{0,99,0},X}; /* railroads */
struct cuts C1[] = {X};		/* transmission lines */
struct cuts D1[] = {{10,10,0},{28,44,0},{50,71,0},{80,89,0},X};	/* landmarks */
struct cuts E1[] = {X};		/* physical feature */
struct cuts F1[] = {{10,13,SS|FF|FS},X}; /* show county lines */
struct cuts G1[] = {{0,99,0},X}; /* special-case (non-CENSUS) */
/* water */
struct cuts H1[] = {{1,1,FF|FS},{11,11,FF|FS},{30,31,FF|FS},
		    {40,41,FF|FS},{50,53,FF|FS},{0,99,SS|FS|FF},X};

struct cuts A2[] = {{0,49,0},{60,69},{0,99,SS|FS|FF},X}; /* roads */
struct cuts B2[] = {{0,19,0},X}; /* railroads */
struct cuts C2[] = {X};		/* transmission lines */
struct cuts D2[] = {{10,10,0},{28,44,0},{50,69,0},{80,89,0},X};	/* landmarks */
struct cuts E2[] = {X};		/* physical feature */
struct cuts F2[] = {{10,19,SS|FF|FS},X}; /* show county lines */
struct cuts G2[] = {{0,99,0},X}; /* special-case (non-CENSUS) */
/* water */
struct cuts H2[] = {{1,1,FF|FS},{11,11,FF|FS},{30,31,FF|FS},
		    {40,41,FF|FS},{50,53,FF|FS},{0,99,SS|FS|FF},X};

struct cuts A4[] = {{0,39,0},{0,99,SS|FS|FF},X};
struct cuts B4[] = {X};
struct cuts C4[] = {X};
struct cuts D4[] = {{31,31,0},{51,51,0},{83,83,0},X};
struct cuts E4[] = {X};
struct cuts F4[] = {{10,19,SS|FF|FS},X};
struct cuts G4[] = {{0,99,0},X}; /* special-case (non-CENSUS) */
struct cuts H4[] = {{1,1,FF|FS},{0,99,SS|FS|FF},X}; 

struct cuts A8[] = {{0,29,0},{0,99,SS|FS|FF},X};
struct cuts B8[] = {X};
struct cuts C8[] = {X};
struct cuts D8[] = {{31,31,0},{51,51,0},{83,83,0},X};
struct cuts E8[] = {X};
struct cuts F8[] = {{10,19,SS|FF|FS},X};
struct cuts G8[] = {{0,99,0},X}; /* special-case (non-CENSUS) */
struct cuts H8[] = {{1,1,FF|FS},{0,99,SS|FS|FF},X}; 

struct cuts A16[] = {{10,19,FS},{0,99,SS|FS|FF},X};
struct cuts B16[] = {X};
struct cuts C16[] = {X};
struct cuts D16[] = {{31,31,0},{41,41,0},{51,51,0},{83,83,0},X};
struct cuts E16[] = {X};
struct cuts F16[] = {{10,19,SS|FS|FF},X};
struct cuts G16[] = {{0,99,0},X}; /* special-case (non-CENSUS) */
#ifdef foo
struct cuts H16[] = {{1,1,FF|FS},{0,99,SS|FS|FF},X}; 
#else
struct cuts H16[] = {{1,1,0},X}; 
#endif
struct cutoff {
  double r;
  struct cuts *c['H'-'A'+1];	/* has these default cut-off values */
} defcut[] = {
  { 16, {A16, B16, C16, D16, E16, F16, G16, H16}}, 
  { 8, {A8, B8, C8, D8, E8, F8, G8, H8}}, 
  { 4, {A4, B4, C4, D4, E4, F4, G4, H4}}, 
  { 2, {A2, B2, C2, D2, E2, F2, G2, H2}}, 
  { 0, {A1, B1, C1, D1, E1, F1, G1, H1}}, /* most detailed */
};

struct cutoff *cut = &defcut[0];

cfccrange(f,cut,show)
struct feat *f;
double *cut,*show;
{
  switch (f->class) {
  case 'D':			/* landmarks */
    switch(f->level) {
    case 31:
      *show=8.0;		/* hospital */
      *cut=0.0;
      break;
    case 40:			/* unknown educational or religious */
    case 41:			/* sorority or fraternity */
    case 42:			/* convent or monastery */
      *show=2.0;
      *cut=0.0;
      break;
    case 43:			/* educational institution */
      *show=8.0;
      *cut=0.0;
      break;
    case 44:			/* religious institution */
      *show=2.0;
      *cut=0.0;
      break;
    case 51:
      *show=4.0;		/* airport */
      *cut=1.0;
      break;
    case 54:
      *show=8.0;		/* harbor */
      *cut=1.0;
      break;
    case 81:			/* state forests, etc */
    case 82:
    case 83:
    case 84:
      *show=-1.0;		/* ??? */
      *cut=100.0;
      break;
    default:
      *show=2.0;
      *cut=0.0;
      break;
    }
    break;
  case 'G':			/* special case */
    *show = 999.0;
    *cut = 0.0;
    break;
  case 'H':
    *show = -1.0;
    *cut = 100.0;
    break;
#ifdef notdef
    if (f->level >= 20 && f->level < 40) {
      *show=1.0;
      *cut=0.0;
    } else {
      *show=2.0;
      *cut=0.0;
    }
    break;
#endif
  default:
    *show=2.0;
    *cut=0.0;
    break;
  }
}

init()				/* calculate scaling factors, etc. */
{
  int i;

  *tigver = '\0';
  /* make the range be 5% more than a power of 2 */
  if ((range/2.0 - ((int)range)/2) <= (.05 * (range/2.0-((int)range)/2)))
    range = ((int) range) * 1.05;
  if (detail < 0)		/* simplified case */
    detail = range;
  if (tile < 0)
    tile = range;		/* in effect, no tiling */
  else if ((tile/2.0 - ((int)tile)/2) <= (.05 * (tile/2.0-((int)tile)/2)))
    tile = ((int) tile) * 1.05;

  for (i = 0; i < sizeof(defcut)/sizeof(defcut[0]); i++) {
    if (detail >= defcut[i].r) {
      cut = &defcut[i];		/* use this cutoff table */
      break;
    }
  }
  if (verbose) {
    fprintf(stderr,BANNER,rev);
    fprintf(stderr,"*range=%.3f detail=%.3f tile=%.3f\n",range,detail,tile);
    fprintf(stderr,"*water=%.3f:%.3f slope=%.3f:%.3f fuzz=%d:%d\n",
	    water_filt,max_filt,slopefuzz,maxslope,fuzz,maxfuzz);
    fprintf(stderr,"*matching on:");
    if (fename_match)
      fprintf(stderr," names");
    if (color_match)
      fprintf(stderr," colors");
    fprintf(stderr,"\n");
    fprintf(stderr,"*Using feature cutoffs for range %.1f:\n",cut->r);
    for (i = 'A'; i <= 'H'; i++) {
      struct cuts *c;
      fprintf(stderr,"*%c: ",i);
      for (c = cut->c[i-'A']; c->lo >= 0; c++)
	fprintf(stderr,"%02d:%02d(%02x), ",c->lo,c->hi,c->fl);
      fprintf(stderr,"\n");
    }
  }
  rootmap = map_init(range,detail,ctr.lat,ctr.lon,0,0,0,0); /* initialize output map */
  init_road_hash();
}

struct map *
map_init(r,d,ctrlat,ctrlon,qd,parent,overlap,base)
double r,d;			/* range, level of detail */
int ctrlat,ctrlon;
enum map_quad qd;		/* is this a quadrant of this... */
struct map *parent;		/* ... parent map */
enum map_overlap overlap;	/* is this an overlap of this... */
struct map *base;		/* ... base map */
{
  struct map *m = (struct map *)calloc(1,sizeof(struct map));  
  int i,j,invis;
  static int mapnum = 0;
  double radx,rady;		/* radius in minutes */

  if (m == NULL) {
    fprintf(stderr,"croak\n");
    exit(1);
  }
  m->mapid = mapnum++;
  m->parent = parent;
  m->overlap = overlap;
  m->base = base;
  m->fuzz = fuzz;		/* initial fuzz value is the global one */
  m->water_filt = water_filt;
  m->slopefuzz = -1.0;		/* doesn't get set until first needed */
  m->range = r;
  m->detail = d;
  m->ctr.lat = ctrlat;
  m->ctr.lon = ctrlon;
  rady = m->range/60.0;		/* radius in degrees (1 mile per minute) */
  /* by default scale for DOS screen unless overridden by pixper */
  m->ppdy = (pixper>0)?pixper:(int) 525.0/(m->detail/60.0)+0.5; 
  m->max.lat = m->ctr.lat + (int)(rady*1000000);
  m->min.lat = m->ctr.lat - (int)(rady*1000000);
  /* DOS screen aspect ratio is 4:3 */
  radx = 4.0*(m->range)/
    (cos(3.14159*(((double)m->max.lat)/(180.0*1000000.0)))*3.0*60.0);
  m->max.lon = m->ctr.lon + (int)(radx*1000000);
  m->min.lon = m->ctr.lon - (int)(radx*1000000);

  if (verbose) {
    if (m->overlap && m->base)
      fprintf(stderr,"map%03d:   overlaps: map%03d\n",
	      m->mapid,m->base->mapid);
    if (m->parent)
      fprintf(stderr,"map%03d:quadrant of: map%03d\n",
	      m->mapid,m->parent->mapid);
    fprintf(stderr,"map%03d:     center: %d/%d\n",
	    m->mapid,m->ctr.lat,m->ctr.lon);
    fprintf(stderr,"map%03d: max coords: %d/%d\n",
	    m->mapid,m->max.lat,m->max.lon);
    fprintf(stderr,"map%03d: min coords: %d/%d\n",
	    m->mapid,m->min.lat,m->min.lon);
    fprintf(stderr,"map%03d:      range: %f miles (%f/%f degrees)\n",
	    m->mapid,m->range,rady,radx);
    fprintf(stderr,"map%03d: pixels per: %d\n",m->mapid,m->ppdy);
  }
  return m;
}

cut_keep(m,s)
struct map *m;
struct seg *s;
{
  struct cuts *c;
  struct feat *f = &s->feat;
  struct name *n = &s->name;

  if (verbose > 2)
    fprintf(stderr,"cut_keep: feature %c%02d %d/%d -> %d/%d",
	    f->class,f->level,s->from.lat,s->from.lon,s->to.lat,s->to.lon);

  /* xxx -- need something like a call to invisible().  Settle for crosses. */
  if (!crosses(m,&s->from,&s->to) /* ends completely off-map */
      || f->class < 'A' || f->class > 'H') {
    if (verbose > 2)
      fprintf(stderr," ignored\n");
    return 0;			/* don't keep garbage */
  }

  for (c = cut->c[f->class-'A']; c->lo >= 0; c++) {
    if (c->fl&FF)		/* not really where this belongs, but... */
      s->flags |= SEG_FF;	/* feature is eligible for faster fuzz */
    if (c->fl&FS)
      s->flags |= SEG_FS;	/* OK to fuzz seg endpoints too */
    if ((c->fl&NN && *n->fename == '\0') /* ignore unnamed features */
	|| (c->fl&SS && !(s->flags&SEG_SS))) /* keep only cnty boundaries */
      break;			
    if (c->lo <= f->level && f->level <= c->hi) {
      if (verbose > 3)
	fprintf(stderr," kept\n");
      return 1;
    }
  }
  if (verbose > 3)
    fprintf(stderr," ignored\n");
  return 0;			/* didn't make the cut */
}

feat_keep(f)
struct feat *f;
{
  struct cuts *c;

  for (c = cut->c[f->class-'A']; c->lo >= 0; c++) {
    if (c->lo <= f->level && f->level <= c->hi) {
      if (verbose > 3)
	fprintf(stderr," kept\n");
      return 1;
    }
  }
  return 0;
}

strtrim(to,from,len)
char *to,*from;
int len;
{
  strncpy(to,from,len);
  while (--len >= 0 && (to[len] == ' ' || to[len] == '\r' || to[len] == '\n'))
    to[len] = '\0';
}

/* from APRS MAPMAKIN.TXT:
 0 - Black (normal background)    * 8 - dark gray (Railroads)
 1 - dim blue (ferrys, etc)         9 - Bright Blue
 2 - dim grn (Admin areas, Parks) *10 - Bright Green (Interstates)
*3 - dim cyan (Rivers)            *11 - Bright Cyan (Big rivers, Coasts)
 4 - deep red (state roads)       *12 - Bright Red   (major roads)
 5 - dim violet (custom features)  13 - Bright Violet(special routes/events)
*6 - dim orange (state/cnty lines) 14 - Bright Yellow (Cities,airports)
*7 - gray     (back roads)         15 - Bright White  (Labels and CALLS)
*/

int
aprscolor(f)			/* return APRS color code for given feature */
struct feat *f;			/* see MAPMAKIN.TXT */
{
  switch (f->class) {
  case 'A':			/* roads */
    if (f->level <= 19)
      return 10;		/* bright green (Interstates) */
    if (f->level <= 29)
      return 12;		/* bright red (major roads) */
    if (f->level <= 39)
      return 4;			/* deep red (state roads) */
    if (f->level <= 49)
      return 7;
    if (f->level == 65)		/* ferries */
      return 1;			/* dim blue (ferrys, etc) */
    return 7;			/* gray (back roads) */
  case 'B':			/* railroads */
    return 8;
  case 'C':			/* transmission lines */
    return 5;			/* for lack of a better color */
  case 'D':			/* landmarks */
    if (f->level == 31)		/* hospital */
      return 9;			/* bright blue */
    if (f->level <= 49)
      switch(f->level) {
      case 40:			/* unknown educational or religious */
	return 5;
      case 41:			/* sorority or fraternity */
	return 2;
      case 42:			/* convent or monastery */
	return 1;
      case 43:			/* educational institution */
	return 2;
      case 44:			/* religious institution */
	return 1;
      default:
	return 7;
      }
    if (f->level <= 59)
      return 14;		/* cities,airports */
    return 2;			/* admin areas, parks */
  case 'E':			/* fences */
    return 6;
    break;
  case 'F':			/* legal boundaries */
    return 6;
  case 'G':			/* my special case */
    return 13;			/* bright purple */
  case 'H':			/* water */
    if (f->level <= 2)
      return 11;		/* big rivers, coasts */
    return 3;			/* rivers */
  }
  return 0;			/* ??? */
}


/* from aprs/readme/symbols.txt: 
       ! - 3 vert bars (EMERGENCY)
       " - RAIN
       # - DIGI
       $ - SUN (always yellow)
       % - DX CLUSTER
       & - HF GATEway
D5[15]  ' - AIRCRAFT (small)
       ( - CLOOUDY
       ) - Hump
       * - SNOW
B*     + - Cross
       , - reverse L shape
xx     - - QTH
               D3[2-5],D22
       . - X
       / - Dot
       0-9 Numerial Boxes
       : - FIRE
D28    ; - Portable tent
       < - Advisory flag
D52    = - RAILROAD ENGINE
       > - CAR (SSID-9)
       ? - GRID SQUARE (six digit.  Not shown below 8 miles)
       @ - HURRICANE or tropical storm
       A-J LETTERED CIRCLES ...............ALTERNATE WX DEFINITIONS:
D4[0-4] K - School                          A - reserved
       L - Lighthouse                      B - Blowing Snow
       M - MacAPRS                         C - reserved
       N - Navigation Buoy                 D - Drizzle
       O - BALLOON                         E - Smoke
       P - Police                          F - Freezing rain
       Q - QUAKE                           G - Snow Shower
D23    R - RECREATIONAL VEHICLE            H - Haze
       S - Space/Satellite                 I - Rain Shower
       T - THUNDERSTORM                    J - Lightening
D53    U - BUS
       V - VORTAC Nav Aid
       W - National WX Service Site
       X - HELO  (SSID-5)
       Y - YACHT (sail SSID-6)
       Z - UNIX X-APRS
       [ - RUNNER
       \ - TRIANGLE   (DF)     
       ] - BOX with X (PBBS's)
       ^ - LARGE AIRCRAFT
       _ - WEATHER SURFACE CONDITIONS (always blue)
       ` - Satellite Ground Station
       a - AMBULANCE
       b - BIKE
       c - DX spot by callsign
       d - Dual Garage (Fire dept)
       e - SLEET
       f - FIRE TRUCK
       g - GALE FLAGS
D31    h - HOSPITAL
       i - IOTA (islands on the air)
       j - JEEP (SSID-12)
       k - TRUCK (SSID-14)
       l - AREAS (box,circle,line,triangle) See below
       m - MILEPOST (box displays 2 letters if in {35})
       n - small triangle
       o - small circle
       p - PARTLY CLOUDY
       q - GRID SQUARE (4 digit.  Not shown below 128 miles)
D7x    r - ANTENNA
D54    s - SHIP (pwr boat SSID-8)
       t - TORNADO
       u - TRUCK (18 wheeler)
       v - VAN (SSID-15)
       w - FLOODING(water)
       x - diamond (NODE)
       y - YAGI @ QTH
       z - WinAPRS
       { - FOG
       | - reserved (Stream Switch)
       } - diamond with cross
 */

char *
aprssymbol(f)
struct feat *f;
{
  static char symbol[3];	/* "$xc" x=symbol, c=color */
  char s = '\0';

  *symbol = '\0';
  switch (f->class) {
  case 'B':			/* railroads */
    s = '+';			/* cross */
    break;
  case 'D':			/* landmarks */
    switch(f->level) {
    case 22:
    case 32:
    case 33:
    case 34:
    case 35:
      s = '-';			/* QTH */
      break;
    case 23:			/* trailer park */
      s = 'R';			/* RV */
      break;
    case 28:			/* campground */
      s = ';';			/* tent */
      break;
    case 31:
      s = 'h';			/* hospital */
      break;
    case 40:
    case 41:
    case 42:
    case 43:
    case 44:
      s = 'K';			/* school */
      break;
    case 51:			/* airport */
    case 55:			/* marine air terminal */
      s = '\'';			/* small aircraft */
      break;
    case 52:
      s = '=';			/* railroad engine */
      break;
    case 53:
      s = 'U';			/* bus */
      break;
    case 54:
      s = 's';			/* ship */
      break;
    case 70:			/* tower */
    case 71:
      s = 'r';			/* antenna (sort of like a tower) */
      break;
    default:
      if (f->level < 80 || f->level >89) 
      s = '}';			/* diamond */
      break;
    }
    break;
  default:
    break;
  }
  if (s)
    sprintf(symbol,"$%c%X",s,aprscolor(f));
  return symbol;
}


/*
 * link up segments into roads.  Each road has two ends ('from' and 'to')
 *  which segments can be glommed onto.
 *
 *struct road anchor <-> road <-> road <-> road
 *                      /    \      ...
 *                from /      \ to
 *                    /        \
 *                   v          v
 *                  link<-...->link
 *                   |          |
 *                   v          v
 *                  seg        seg
 */
link_segs()
{
  int i;

  bzero(&anchor,sizeof(anchor)); 

  if (verbose)
    fprintf(stderr,"linking segments...\n");
  for (i = 0; i < curseg; i++) {
    struct road *f;
    if (segs[i].flags&SEG_IGN)	/* skip filtered segments! */
      continue;
    if (verbose > 2)
      fprintf(stderr,"link_segs: segs[%d] id=%d from=%d/%d to=%d/%d\n",
	      i,segs[i].tlid,segs[i].from.lat,segs[i].from.lon,
	      segs[i].to.lat,segs[i].to.lon);
    if (f = find_road_from(&segs[i]))
      ins_seg_before(f,&segs[i]);
    else if (f = find_road_to(&segs[i]))
      ins_seg_after(f,&segs[i]);
    else if (f = find_flipped_road_from(&segs[i])) {
	flip_seg(&segs[i]);
	ins_seg_before(f,&segs[i]);
    } else if (f = find_flipped_road_to(&segs[i])) {
      flip_seg(&segs[i]);
      ins_seg_after(f,&segs[i]);
    } 
    else
      ins_seg_new(&segs[i]);
  }
  if (verbose)
    fprintf(stderr,"... %d segments were combined into %d roads\n",curseg,currd);
}

/* hash chains for roads, indexed by lat/lon */
static struct road *roadhash = NULL;
#define RHASHSIZE 0xFFF
#define ROADHASH(x) (((x).lat+(x).lon)&RHASHSIZE)
/* basic list manipulation functions */

add_hashfrom(r,i)		/* push this road onto roadhash[i].from list */
struct road *r;
int i;
{
  int shouldbe = ROADHASH(r->head->from);
  if (i != shouldbe)
    abort();
  if (verbose > 2)
    fprintf(stderr,"add_hashfrom: adding road %d to roadhash[%d]\n",
	    r->roadid,i);
  if (r->fnr = roadhash[i].fnr)
    r->fnr->fpr = r;		/* chain back if not the last one */
  r->fpr = &roadhash[i];
  roadhash[i].fnr = r;
}

add_hashto(r,i)			/* push this road onto roadhash[i].to list */
struct road *r;
int i;
{
  int shouldbe = ROADHASH(r->tail->to);

  if (i != shouldbe)
    abort();
  if (verbose > 2)
    fprintf(stderr,"add_hashto: adding road %d to roadhash[%d]\n",
	    r->roadid,i);
  if (r->tnr = roadhash[i].tnr)
    r->tnr->tpr = r;
  r->tpr = &roadhash[i];
  roadhash[i].tnr = r;
}

drop_hashfrom(r,i)		/* remove road from roadhash[i].from list */
struct road *r;
int i;
{
  struct road *rp;

  if (verbose > 2)
    fprintf(stderr,"drop_hashfrom: dropping road %d from roadhash[%d]\n",
	    r->roadid,i);
  /* somewhat idiot-proofed just to make sure I am doing it right */
  for (rp = roadhash[i].fnr; rp; rp = rp->fnr) {
    if (rp == r) {		/* found road to unlink */
      r->fpr->fnr = r->fnr;	/* set prev road to skip to my next */
      if (r->fnr)
	r->fnr->fpr = r->fpr;	/* set next road to skip to my prev */
      r->fnr = r->fpr = NULL;	/* unlinked from the hashfrom chain */
      return;
    }
  }
  fprintf(stderr,"drop_hashfrom road %d is not on roadhash[%d]!\n",
	  r->roadid,i);
}

drop_hashto(r,i)		/* remove road from roadhash[i].to list */
struct road *r;
int i;
{
  struct road *rp;

  if (verbose > 2)
    fprintf(stderr,"drop_hashto: dropping road %d from roadhash[%d]\n",
	    r->roadid,i);
  for (rp = roadhash[i].tnr; rp; rp = rp->tnr) {
    if (rp == r) {		/* found road to unlink */
      r->tpr->tnr = r->tnr;	/* set prev road to skip to my next */
      if (r->tnr)
	r->tnr->tpr = r->tpr;	/* set next road to skip to my prev */
      r->tnr = r->tpr = NULL;	/* unlinked from the hashto chain */
      return;
    }
  }
  fprintf(stderr,"drop_hashto road %d is not on roadhash[%d]!\n",
	  r->roadid,i);
}

init_road_hash()
{
  roadhash = (struct road *)calloc(RHASHSIZE+1,sizeof(struct road));
  if (roadhash == NULL) {
    fprintf(stderr,"couldn't calloc roadhash!\n");
    exit(1);
  }
}

add_road_hash(r)		/* add road to hash lists */
struct road *r;
{
  int fromidx = ROADHASH(r->head->from);
  int toidx = ROADHASH(r->tail->to);

  /* link the road in to front of the from & to hash chains */
  add_hashfrom(r,fromidx);
  add_hashto(r,toidx);
}

drop_road_hash(r)		/* remove road from hash lists */
struct road *r;
{
  int fromidx = ROADHASH(r->head->from);
  int toidx = ROADHASH(r->tail->to);

  drop_hashfrom(r,fromidx);
  drop_hashto(r,toidx);
}

/*
 * rehash a road by removing it's old head from a roadhash[] list
 *  and moving it to the new hash position 
 */
rehash_road_head(r,olds)
struct road *r;			/* the road w/modified head */
struct seg *olds;		/* the old head of this road */
{
  int wasidx = ROADHASH(olds->from);
  int newidx = ROADHASH(r->head->from);

  drop_hashfrom(r,wasidx);	/* find r and unlink from old chain */
  add_hashfrom(r,newidx);	/* add r into new chain */
}

rehash_road_tail(r,olds)
struct road *r;			/* the road w/modified tail */
struct seg *olds;		/* the old tail of this road */
{
  int wasidx = ROADHASH(olds->to);
  int newidx = ROADHASH(r->tail->to);

  drop_hashto(r,wasidx);	/* find r and unlink from old chain */
  add_hashto(r,newidx);		/* add r into new chain */
}

fprint_road_hash(f)
FILE *f;
{
  int i;

  for (i = 0; i <= RHASHSIZE; i++) {
    fprint_one_roadhash(f,i);
  }
}

fprint_one_roadhash(f,i)
FILE *f;
int i;
{
  struct road *rp;

  if (roadhash[i].fnr) {
    fprintf(f,"roadhash[%d].from =",i);
    for (rp = roadhash[i].fnr; rp; rp = rp->fnr)
      fprintf(f," %d",rp->roadid);
    fprintf(f,"\n");
  }
  if (roadhash[i].tnr) {
    fprintf(f,"roadhash[%d].to =",i);
    for (rp = roadhash[i].tnr; rp; rp = rp->tnr)
      fprintf(f," %d",rp->roadid);
    fprintf(f,"\n");
  }
}

struct road *
find_road_from(sp)
struct seg *sp;
{
  struct road *r;
  int fromidx = ROADHASH(sp->to);

  for (r = roadhash[fromidx].fnr; r; r = r->fnr) {
    if (sp->to.lat == r->head->from.lat 
	&& sp->to.lon == r->head->from.lon
	&& name_eq(sp,r->head,name_eq_global)) {
      if (verbose > 2)
	fprintf(stderr,"find_road_from: %d/%d at %0x\n",
		sp->to.lat,sp->to.lon,r);
      return r;
    }
  }
  return NULL;
}

struct road *
find_road_to(sp)
struct seg *sp;
{
  struct road *r;
  int toidx = ROADHASH(sp->from);

  for (r = roadhash[toidx].tnr; r; r = r->tnr) {
    if (sp->from.lat == r->tail->to.lat 
	&& sp->from.lon == r->tail->to.lon
	&& name_eq(sp,r->tail,name_eq_global)) {
      if (verbose > 2)
	fprintf(stderr,"find_road_to: %d/%d at %0x\n",
		sp->from.lat,sp->from.lon,r);
      return r;
    }
  }
  return NULL;
}

struct road *
find_flipped_road_to(sp)
struct seg *sp;
{
  struct road *r;
  int toidx = ROADHASH(sp->to);

  for (r = roadhash[toidx].tnr; r; r = r->tnr) {
    if (sp->to.lat == r->head->to.lat 
	&& sp->to.lon == r->head->to.lon
	&& name_eq(sp,r->head,name_eq_global)) {
      if (verbose > 2)
	fprintf(stderr,"find_flipped_road_to: %d/%d at %0x\n",
		sp->to.lat,sp->to.lon,r);
      return r;
    }
  }
  return NULL;
}

struct road *
find_flipped_road_from(sp)
struct seg *sp;
{
  struct road *r;
  int fromidx = ROADHASH(sp->from);

  for (r = roadhash[fromidx].fnr; r; r = r->fnr) {
    if (sp->from.lat == r->tail->from.lat 
	&& sp->from.lon == r->tail->from.lon
	&& name_eq(sp,r->tail,name_eq_global)) {
      if (verbose > 2)
	fprintf(stderr,"find_flipped_road_from: %d/%d at %0x\n",
		sp->from.lat,sp->from.lon,r);
      return r;
    }
  }
  return NULL;
}

ins_seg_new(s)			/* add a new road that is not linked to */
struct seg *s;
{
  struct road *r = (struct road *)calloc(1,sizeof(struct road));

  if (verbose > 2)
    fprintf(stderr,"ins_seg new road %0x, seg %0x tlid %d\n",
	   r,s,s->tlid);

  r->roadid = currd++;
  r->head = r->tail = s;	/* new road consists of a single link */
  add_road_hash(r);		/* add to hash lists */
  if (anchor.nr)		/* for now, continue to add to linear list */
    anchor.nr->pr = r;		/* point previous 1st road back at me */
  r->nr = anchor.nr;		/* and put the previous head after me */
  r->pr = &anchor;		/* point back at head of list */
  anchor.nr = r;		/* point head at me */
}

ins_seg_after(r,s)		/* insert seg s w/new link after tail link */
struct road *r;
struct seg *s;
{
  if (verbose > 2)
    fprintf(stderr,"ins seg %0x after road %0x, seg tlid %d\n",
	   s,r,s->tlid);
  s->ps = r->tail;		/* prev to this link is current end link */
  s->ns = 0;			/* this link is the new end link */
  r->tail->ns = s;		/* hang this link off previous end link */
  r->tail = s;			/* make this be the new tail */
  rehash_road_tail(r,s->ps);	/* rehash new tail lat/lon */
}

ins_seg_before(r,s)		/* insert seg s w/new link before head link */
struct road *r;
struct seg *s;
{
  if (verbose > 2)
    fprintf(stderr,"ins seg %0x after road %0x, seg tlid %d\n",
	   s,r,s->tlid);
  s->ns = r->head;		/* next to this link is current head link */
  s->ps = 0;			/* this link is the new head link */
  r->head->ps = s;		/* put this link in front of prev head link */
  r->head = s;			/* make this be the new head */
  rehash_road_head(r,s->ns);	/* rehash new head lat/lon */
}

int
name_eq(s1,s2,options)
struct seg *s1,*s2;
enum name_eq_opt options;
{
  struct name *n1 = &s1->name, *n2 = &s2->name;

  if (options==color) {		/* colors are enuf, CFCC matching not req'd */
    if (aprscolor(&s1->feat) == aprscolor(&s2->feat))
      return 1;
  }
  if (s1->feat.class != s2->feat.class)
    return 0;
  if (s1->feat.level != s2->feat.level)
    return 0;
  if (options==name) {
    if (strncmp(n1->fedirp,n2->fedirp,sizeof(n1->fedirp)))
      return 0;
    if (strncmp(n1->fename,n2->fename,sizeof(n1->fename)))
      return 0;
    if (strncmp(n1->fetype,n2->fetype,sizeof(n1->fetype)))
      return 0;
    if (strncmp(n1->fedirs,n2->fedirs,sizeof(n1->fedirs)))
      return 0;
  }
  return 1;			/* they match */
}

/*
 * Check for any roads that could link up by flipping them (reversing
 * order of points).
 */
static int road_dropped;

join_roads()
{
  struct road *i,*j;
  int still_going;		/* keep relinking. */
  int passes = 0;
  int hashidx;
  int fename_was = fename_match;
  int color_was = color_match; 
  enum name_eq_opt opt = (fename_match)?name:none; /* 1st passes match names */

  road_dropped = 0;
  do {
    ++passes;
    if (verbose)
      fprintf(stderr,"Joining roads, pass %d\n",passes);
    if (verbose > 2)
      fprint_road_hash(stderr);
    /*
     * search each hash chain (hashfrom,hashto) for these matches:
     *
     *  head to tail		(from -- to)
     *  head to head		(from -- from)
     *  tail to head		(to -- from)
     *  tail to tail		(to -- to)
     *
     * In head to tail & vice-versa cases, note that the matching lat/lon
     * for something on hashfrom[hasidx] will be on hashto[hashidx] since
     * hashidx is a many-to-one function of lat/lon.
     */
    for (still_going = 0, hashidx = 0; hashidx <= RHASHSIZE; hashidx++) {
      int lc;			/* list changed flag */

      /* cruise the hashfrom list for this hash index... */
      for (i = roadhash[hashidx].fnr; i; i = i->fnr) {
	for (lc = 0,j = roadhash[hashidx].tnr; j; j = j->tnr) {/* head -- tail */
	  if (i != j && i->head->from.lat == j->tail->to.lat
	      && i->head->from.lon == j->tail->to.lon
	      && name_eq(i->head,j->tail,opt)) {
	    lc = still_going = 1;
	    join_head_tail(i,j); /* tack them together */
	    break;		/* j has been freed */
	  }
	}
	if (lc)			/* the hash chain has changed out from under */
	  break;
	for (lc = 0,j = roadhash[hashidx].fnr; j; j = j->fnr) { /* head -- head */
	  if (i != j && i->head->from.lat == j->head->from.lat
	      && i->head->from.lon == j->head->from.lon
	      && name_eq(i->head,j->head,opt)) {
	    lc = still_going = 1;
	    join_heads(i,j); /* tack them together */
	    break;		/* j has been freed */
	  }
	}
	if (lc)
	  break;
      }	/* end of hashfrom cruise */
      /* now cruise the hashto list for this hash index */
      for (i = roadhash[hashidx].tnr; i; i = i->tnr) {
	for (lc = 0,j = roadhash[hashidx].fnr; j; j = j->fnr) {/* tail -- head */
	  if (i != j && i->tail->to.lat == j->head->from.lat
	      && i->tail->to.lon == j->head->from.lon
	      && name_eq(i->tail,j->head,opt)) {
	    lc = still_going = 1;
	    join_tail_head(i,j); /* tack them together */
	    break;		/* j has been freed */
	  }
	}
	if (lc)
	  break;
	for (lc = 0,j = roadhash[hashidx].tnr; j; j = j->tnr) { /* tail -- tail */
	  if (i != j && i->tail->to.lat == j->tail->to.lat
	      && i->tail->to.lon == j->tail->to.lon
	      && name_eq(i->tail,j->tail,opt)) {
	    lc = still_going = 1;
	    join_tails(i,j); /* tack them together */
	    break;		/* j has been freed */
	  }
	}
	if (lc)
	  break;
      } /* end of hashto cruise */
    } /* end of hash index loop */
    if (still_going == 0) {
      if (fename_was) {
	still_going = 1;
	fename_was = 0;
	opt = none;		/* do some more passes matching just on CFCC */
	if (verbose) {
	  fprintf(stderr,"so far %d roads have been joined into %d.\n",
		  currd,currd-road_dropped);
	  fprintf(stderr,"Now joining roads without name matches....\n");
	}
      } else if (color_was) {
	still_going = 1;
	color_was = 0;
	opt = color;
	if (verbose) {
	  fprintf(stderr,"so far %d roads have been joined into %d.\n",
		  currd,currd-road_dropped);
	  fprintf(stderr,"Now joining roads with APRS color matches....\n");
	}
      }
    }
  } while(still_going);
  if (verbose)
    fprintf(stderr,"... %d roads were joined into %d\n",
	    currd,currd-road_dropped);
}

/*
 * join_head:
 *
 * 	road i pts = c     b     a
 *                   ^           ^
 *                   |           |
 *           seg1.from    ...    s1.to
 *            ^   
 *            |        ^
 *            |        |
 *           head.from tail.to
 *
 * 	road j pts = c     d     e     f     g    h
 *                   ^     ^     ^     ^     ^    ^
 *                   |     |     |     |     |    |
 *           seg1.from s1.to s2.from s2.way way+1 s2.to
 *            ^               ^
 *            |               |
 *             <------------->
 *            ^                  ^
 *            |                  |
 *      head.from                tail.to
 *
 *  1. flip_road(j):    h   g   f   e   d   c
 *                      ^                   ^
 *              head.from                   tail.to
 *
 *  2. join_road(i,j)
 *
 */

join_heads(i,j)
struct road *i,*j;		/* road i is kept. road j gets junked. */
{
  struct seg *oldhead;

  if (verbose > 2) {
    fprintf(stderr,"\njoining heads of roads %d and %d\n",i->roadid,j->roadid);
    fprintf(stderr," road i=%d :\n",i->roadid);
    fprint_road(stderr,i);
    fprintf(stderr," road j=%d before flip:\n",j->roadid);
    fprint_road(stderr,j);
  }
  flip_road(j);			/* reverse the direction of points */
  if (verbose > 2) {
    fprintf(stderr," road j=%d after flip:\n",j->roadid);
    fprint_road(stderr,j);
  }
  j->tail->ns = i->head;	/* glue i's 1st seg onto end of j's last */
  i->head->ps = j->tail;
  oldhead = i->head;
  i->head = j->head;		/* keep i as the new road ptr, w/new head */
  if (j->nr)
    j->nr->pr = j->pr;		/* remove j from list of roads */
  if (j->pr)
    j->pr->nr = j->nr;
  rehash_road_head(i,oldhead);
  drop_road_hash(j);
  free(j); ++road_dropped;
  if (verbose > 2) {
    fprintf(stderr," road i=%d after join:\n",i->roadid);
    fprint_road(stderr,i);
  }
}

join_tails(i,j)
struct road *i,*j;
{
  struct seg *oldtail;

  if (verbose > 2) {
    fprintf(stderr,"\njoining tails of roads %d and %d\n",i->roadid,j->roadid);
    fprintf(stderr," road i=%d :\n",i->roadid);
    fprint_road(stderr,i);
    fprintf(stderr," road j=%d before flip:\n",j->roadid);
    fprint_road(stderr,j);
  }
  flip_road(j);			/* reverse the direction of points */
  if (verbose > 2) {
    fprintf(stderr," road j=%d after flip:\n",j->roadid);
    fprint_road(stderr,j);
  }
  i->tail->ns = j->head;	/* glue j's 1st seg onto end of i's last */
  j->head->ps = i->tail;
  oldtail = i->tail;
  i->tail = j->tail;		/* keep i as the new road ptr, w/new tail */
  if (j->nr)
    j->nr->pr = j->pr;		/* remove j from list of roads */
  if (j->pr)
    j->pr->nr = j->nr;
  rehash_road_tail(i,oldtail);
  drop_road_hash(j);
  free(j); ++road_dropped;
  if (verbose > 2) {
    fprintf(stderr," road i=%d after join:\n",i->roadid);
    fprint_road(stderr,i);
  }
}

join_tail_head(i,j)
struct road *i,*j;
{
  struct seg *oldtail;

  if (verbose > 2) {
    fprintf(stderr,"\njoining tail of roads %d to head of road %d\n",i->roadid,j->roadid);
    fprintf(stderr," road i=%d:\n",i->roadid);
    fprint_road(stderr,i);
    fprintf(stderr," road j=%d:\n",j->roadid);
    fprint_road(stderr,j);
  }
  i->tail->ns = j->head;	/* glue j's 1st seg onto end of i's last */
  j->head->ps = i->tail;
  oldtail = i->tail;
  i->tail = j->tail;		/* keep i as the new road ptr, w/new tail */
  if (j->nr)
    j->nr->pr = j->pr;		/* remove j from list of roads */
  if (j->pr)
    j->pr->nr = j->nr;
  rehash_road_tail(i,oldtail);
  drop_road_hash(j);
  free(j); ++road_dropped;
  if (verbose > 2) {
    fprintf(stderr," road i=%d after join:\n",i->roadid);
    fprint_road(stderr,i);
  }
}

join_head_tail(i,j)
struct road *i,*j;
{
  struct seg *oldhead;

  if (verbose > 2) {
    fprintf(stderr,"\njoining head of road %d to tail of road %d\n",i->roadid,j->roadid);
    fprintf(stderr," road i=%d:\n",i->roadid);
    fprint_road(stderr,i);
    fprintf(stderr," road j=%d:\n",j->roadid);
    fprint_road(stderr,j);
  }
  j->tail->ns = i->head;
  i->head->ps = j->tail;	/* glue j's 1st seg onto end of i's last */
  oldhead = i->head;
  i->head = j->head;		/* keep i as the new road ptr, w/new head */
  if (j->nr)
    j->nr->pr = j->pr;		/* remove j from list of roads */
  if (j->pr)
    j->pr->nr = j->nr;
  rehash_road_head(i,oldhead);
  drop_road_hash(j);
  free(j); ++road_dropped;
  if (verbose > 2) {
    fprintf(stderr," road i=%d after join:\n",i->roadid);
    fprint_road(stderr,i);
  }
}

/*
 * flip_road: 
 *  1. reverse the order of the Pts[] in each seg.
 *  2. reverse the order of the segs pointed to be each link.
 *  3. swap the seg from and to pointers.
 *  4. swap the road head and tail pointers.
 */
flip_road(r)
struct road *r;
{
  struct seg *l,*nextl,*temp;

  drop_road_hash(r);		/* lose it before getting confused by flip */
  for (l = r->head; l ; l = nextl) { /* for each segment linked to */
    flip_seg(l);		/* flip segment's head and tail */
    nextl = l->ns;		/* remember this before clobbering it! */
    temp = l->ns;		/* swap next & previous pointers */
    l->ns = l->ps;
    l->ps = temp;
  }
  temp = r->head;		/* swap head and tail pointers */
  r->head = r->tail;
  r->tail = temp;
  add_road_hash(r);
}

flip_seg(s)
struct seg *s;
{
  struct latlon tll;
  int tpt;
  
  s->flags ^= SEG_REV;		/* flip reversal bit */
  tll = s->from;		/* swap seg from and to latlons */
  s->from = s->to;
  s->to = tll;
  tpt = s->first;		/* and reverse the meaning of 1st and last */
  s->first = s->last;
  s->last = tpt;
}

fprint_seg(f,l)
FILE *f;
struct seg *l;
{
  int j,i,n,direction;

  fprintf(f," seg %0x id %d from %d/%d to %d/%d\n",
	  l,l->tlid,l->from.lat,l->from.lon,l->to.lat,l->to.lon);
  n=abs(l->first-l->last);
  fprintf(f,"   %d way points (%d to %d):\n",n,l->first,l->last);
  direction=(l->flags&SEG_REV)?-1:1;
  for (j = l->first,i = 0; j >= 0 && i <= n; j += direction,++i) {
    fprintf(f,"    [%d] %d/%d\n",j,Pts[j].lat,Pts[j].lon);
  }
}

fprint_segs(f)
FILE *f;
{
  int i;
  
  fprintf(f,"FPRINT_SEGS: %d segs:\n",curseg);
  for (i = 0; i < curseg; i++)
    fprint_seg(f,&segs[i]);
}

fprint_road(f,r)
FILE *f;
struct road *r;
{
  struct seg *l;

  fprintf(f,"Road %d head at %d/%d, tail at %d/%d\n",
	  r->roadid, r->head->from.lat, r->head->from.lon,
	  r->tail->to.lat, r->tail->to.lon);
  fprintf(f," (preceding road is %d, next road is %d)\n",
	  (r->pr)?r->pr->roadid:-1,(r->nr)?r->nr->roadid:-1);
  for (l = r->head; l; l = l->ns) {
    fprint_seg(f,l);
  }
}

fprint_roads(f)
FILE *f;
{
  struct road *r;

  for (r = anchor.nr; r; r = r->nr) { /* loop over roads */
    if (r->flags&RD_SKIP)
      continue;
    fprint_road(f,r);
  }
}


/*
 * "lock" any road's seg's from or to that intersect with other visible roads.
 * Other segment intersection points are less important and can be fuzzed.
 * For each segment in a road, see if it shares a TO or FROM with a segment
 * not in the same road (e.g. the matching seg is not immediately adjacent).
 *
 *                          |
 *     road1 ---seg1---seg2-+-seg3---seg4
 *                          |
 *                         seg5
 *                          |
 *                         seg6
 *                         road2
 *
 * in the above example, the important intersection is where segs 2&3 meet 5.
 */

lock_roads()
{
  struct road *r;
  int locked = 0;

  if (verbose)
    fprintf(stderr,"Locking endpoints of visible intersections...");

  for (r = anchor.nr; r; r = r->nr) { /* loop over roads...*/
    struct seg *l;
    if (r->flags&RD_SKIP)
      continue;			/* ignore skipped roads */
    /* skip forward to first visible seg (if any) */
    for (l = r->head; l && l->flags&(SEG_IGN|SEG_FS); l = l->ns) 
      ; /* skip invis */
    if (!l)
      continue;			/* all segs in this road were skipped! */
    for (; l; l = l->ns) {	/* walk the segs... */
      struct seg *s;
      if (l->flags&(SEG_IGN|SEG_FS))	/* only want visible segments */
	continue;
      /* Compare my FROM with other seg's FROM... */
      for (s = seghash[LLHASH(l->from)].frns; s; s = s->frns) {
	/* don't match on ignored segs, my seg, my neighbors on same road! */
	if (s->flags&(SEG_IGN|SEG_FS) || s == l || s == l->ps || s == l->ns)
	  continue;
	if (l->from.lat == s->from.lat && l->from.lon == s->from.lon) {
	  s->flags |= SEG_LKFR;	/* this point is an intersection */
	  ++locked;
	  if (verbose > 3)
	    fprintf(stderr,"setting LKFR on seg %0x (matches %0x)\n",
		    s,l);
	}
      } /* ... compare my FROM with other segs' FROM */
      /* Compare my FROM with other seg's TO... */
      for (s = seghash[LLHASH(l->from)].tons; s; s = s->tons) {
	if (s->flags&(SEG_IGN|SEG_FS) || s == l || s == l->ps || s == l->ns)
	  continue;
	if (l->from.lat == s->to.lat && l->from.lon == s->to.lon) {
	  s->flags |= SEG_LKFR;	/* this point is an intersection */
	  ++locked;
	  if (verbose > 3)
	    fprintf(stderr,"setting LKFR on seg %0x (matches %0x)\n",
		    s,l);
	}
      } /* ... compare my FROM with other segs' TO */
      /* Compare my TO with other seg's FROM... */
      for (s = seghash[LLHASH(l->to)].frns; s; s = s->frns) {
	if (s->flags&(SEG_IGN|SEG_FS) || s == l || s == l->ps || s == l->ns)
	  continue;
	if (l->to.lat == s->from.lat && l->to.lon == s->from.lon) {
	  s->flags |= SEG_LKTO;	/* this point is an intersection */
	  ++locked;
	  if (verbose > 3)
	    fprintf(stderr,"setting LKTO on seg %0x (matches %0x)\n",
		    s,l);
	}
      } /* ... compare my TO with other segs' TO */
      /* Compare my TO with other seg's TO... */
      for (s = seghash[LLHASH(l->to)].tons; s; s = s->tons) {
	if (s->flags&(SEG_IGN|SEG_FS) || s == l || s == l->ps || s == l->ns)
	  continue;
	if (l->to.lat == s->to.lat && l->to.lon == s->to.lon) {
	  s->flags |= SEG_LKTO;	/* this point is an intersection */
	  ++locked;
	  if (verbose > 3)
	    fprintf(stderr,"setting LKTO on seg %0x (matches %0x)\n",
		    s,l);
	}
      } /* ... compare my from with other segs' to */
    } /* ... walk the segs */    
  } /* ...loop over roads */
  if (verbose)
    fprintf(stderr," %d intersections locked\n",locked);
}


/*
 * h:  head.from   tail.to       t: head.from  tail.to
 *      l->seg1     l->seg(n-1)        l->segn     l->segm
 * 
 *     head.from                               tail.to
 *      l->seg1               ...                  l->segm
 */

int
do_roads(m,count)
struct map *m;
int count;			/* 0 = print, 1 = count */
{
  struct road *r;
  struct seg *l;
  int i,j,n,direction;
  int x,y;	/* use these to eliminate redundant pts */
  int skipped;
  char *myname = (count)?"count":"print";

  ptsout = 0;
  /* what's left has been cleaned up somewhat */
  if (!count) {
    printf(" %d.%06d ,Lat of map origin\r\n",m->max.lat/1000000,m->max.lat%1000000);
    printf(" %d.%06d ,Lon of map origin\r\n",m->max.lon/1000000,m->max.lon%1000000);
    printf(" %d ,Pixels per degree\r\n",m->ppdy);
    printf(" %d.%06d ,Lat of map center\r\n",m->ctr.lat/1000000,m->ctr.lat%1000000);
    printf(" %d.%06d ,Lon of map center\r\n",m->ctr.lon/1000000,m->ctr.lon%1000000);
    printf(" %3.2f ,Map range from center\r\n",m->range);
    printf(" 0 ,Reserved\r\n");
    printf("Map generated by tig2aprs from %s US Census TIGER/Line.\r\n",
	   tigver);
    printf(BANNER,rev);
    printf("map has %d points f=%d wf=%.3f sf=%.3f \r\n",
	   m->pts,m->fuzz,m->water_filt,m->slopefuzz);
  }

  for (r = anchor.nr; r; r = r->nr) { /* loop over roads */
    if (r->flags&RD_SKIP)
      continue;
    if (verbose > 2) {
      fprintf(stderr,"%s_roads putting ",myname);
      fprint_road(stderr,r);
    }
    /* a road may need to be split into pieces if it meanders off the map. */
    for (l = r->head; l;) {
      /* skip from current head segment until we find first visible segment */
      for (skipped = 0; l && l->flags&SEG_IGN; l = l->ns) 
	++skipped;
      if (verbose > 2)
	fprintf(stderr,"Skipped %d segments, starting at seg link %0x\n",
		skipped,l);
      if (!l)
	break;			/* all segs in this road were skipped! */
      putnew(m,l,count);	/* use the first visible seg as the header */
      /* now continue here for this and all subsequent points */
      /* loop over all visible links in this road */
      for (; l; l = l->ns) {
	if (l->flags&SEG_IGN) {	/* only print visible segments */
	  /* a discontinuity requires ending this road & starting a new one */
	  break;
	}
	if (verbose > 2)	/* debug into the output file */
	  putxxx(m,l,count);	/* this breaks the map file */
	/* the Record Type 1 'from' point */
	putxy(m,l,&l->from,count,l->flags&SEG_LKFR);
	/* for all road segments, dump out all Record Type 2 way points next, 
	   then Record Type 1 'to' point */
      
	n=abs(l->first-l->last);
	direction=(l->flags&SEG_REV)?-1:1;
	for (i = 0, j = l->first; j >= 0 && i < n; j += direction,++i) {
	  putxy(m,l,&Pts[j],count,0);
	}
	putxy(m,l,&l->to,count,l->flags&SEG_LKTO);
      }
      putend(m,count);
    }
  }
  return (m->pts = ptsout);
}

int 
count_roads(m)
struct map *m;
{
  return do_roads(m,1);
}

int
print_roads(m)
struct map *m;
{
  return do_roads(m,0);
}

struct latlon *
find_area_ctr(m,al)
struct map *m;
struct arealink *al;
{
  struct arealink *l;
  struct latlon maxi = {-999999999,-999999999};
  struct latlon mini = {999999999,999999999};
  struct latlon r;
  int found = 0;
  double mindist = 999999999.0;
  struct latlon *minp;

  /* find max and min and then find the center. */
  for (l = al; l; l = l->nal) {
    if (isvis(m,&l->p->ctr)) {
      find_maxmin(&maxi,&mini,&l->p->ctr);
      ++found;
    }
  }
  if (!found)
    return NULL;
  /* here's the rectangular center point, but we really want to guarantee
     it is an interior point for odd-shaped areas, so find the closest
     elementary polygon interior point. */
  r.lat = mini.lat + ((maxi.lat - mini.lat)/2);
  r.lon = mini.lon + ((maxi.lon - mini.lon)/2);
  for (l = al; l; l = l->nal) {
    if (isvis(m,&l->p->ctr)) {
      double dist = hypot((double)(r.lat-l->p->ctr.lat),
			  (double)(r.lon-l->p->ctr.lon));
      if (dist < mindist) {
	mindist = dist;
	minp = &l->p->ctr;
      }
    }
  }
  return minp;
}

/* default visibility ranges for cities,towns,etc.
 * rangeshow is what range to start showing the label.  These labels
 * get included in maps with detail level <= 2*rangeshow.
 * rangecut is what detail level below which labels will not be included
 * in maps.
 */

#ifdef notdef
static double rangeshow[4] = {32,0,0,8};/* city,town(mcd),sub-mcd,place */
static double rangecut[4] = { 4,0,0,2};	/* towns are just too much clutter */
#endif

/* The online FIPS55 lists the darn codes but not what they mean! */

fipsrange(classcode,cut,show)
char *classcode;
double *cut,*show;
{
  if (!classcode) {
    *cut = *show = 0.0;
  } else {
    switch (classcode[0]) {
    case 'T':			/* Town */
      *cut = *show = 0.0;		/* too much clutter */
      break;
    case 'C':			/* City? */
      if (classcode[1] >= '5') {	/* medium-sized city */
	*show = 32.0;
	*cut = 4.0; 
      } else {			/* small city */
	*show = 8.0;
	*cut = 2.0; 
      }
      break;
    case 'U':			/* Uhhh, smaller than a city? */
      if (classcode[1] >= '5') {	/* medium-sized village */
	*show = 4.0;
	*cut = 2.0; 
      } else {			/* small village */
	*show = 4.0;
	*cut = 1.0; 
      }
      break;
    default:
      *cut = *show = 0.0;
      break;
    }
  }
}

double
distance(l1,l2)			/* distance in knots */
struct latlon *l1,*l2;
{
  return hypot((double)(l1->lat-l2->lat),(double)(l1->lon-l2->lon))/1000000.0*60.0;
}

#define APRSLABS 199

int put_label(struct map *,char *,char *,struct latlon *,double,struct feat *);

print_labels(m)
struct map *m;
{
  int i;
  struct kgl *k;
  double rcut,rshow;

  printf(" 0,-1\r\n0,labels\r\n");
  /* places */
  if (label_places) {
    for (i = 0; i < sizeof(areas)/sizeof(areas[0]); i++) {
      struct area *a = &areas[i];
      
      for (a = a->na; a; a = a->na) {
	struct latlon *ll = find_area_ctr(m,a->link);
	fipsrange(a->classcode,&rcut,&rshow);
	if (ll && m->detail/2.0 <= rshow && m->detail >= rcut) {
	  put_label(m,(rshow>=4.0)?"$/E":"",a->name,ll,rshow,0);
	  if (debug&DEBUG_LABELS) {
	    int pid = 0;
	    if (a->link && a->link->p)
	      pid = a->link->p->polyid;
	    fprintf(stderr,"label: area %0x fips %d class %s name %s poly %d\n",
		    a,a->fips,a->classcode,a->name,pid);
	  }
	}
      }
    }
  }
  /* point and area landmarks */
  if (label_landmarks) {
    for (i = 0; i < curland; i++) {
      struct land *l = &landmarks[i];
      
      cfccrange(&l->cfcc,&rcut,&rshow);
      if (l->flags&LAND_POINT) {	/* point landmark */
	if (isvis(m,&l->u.ctr) && m->detail/2.0 <= rshow && m->detail>= rcut) {
	  put_label(m,aprssymbol(&l->cfcc),l->name,&l->u.ctr,rshow,&l->cfcc);
	  if (debug&DEBUG_LABELS)
	    debug_land(l,"point",&l->u.ctr);
	}
      } else {			/* area landmark */
	struct latlon *ll = find_area_ctr(m,l->u.link);
	if (ll && m->detail/2.0 <= rshow && m->detail >= rcut) {
	  put_label(m,aprssymbol(&l->cfcc),l->name,ll,rshow,&l->cfcc);
	  if (debug&DEBUG_LABELS)
	    debug_land(l,"area",ll);
	}
      }
    }
  }
  /* cruise through the kglname list. */
  if (label_kgls) {
    for (k = kgls.nk; k; k = k->nk) {
      double rcut,rshow;
      struct latlon *ll = find_area_ctr(m,k->link);
      
      cfccrange(&k->cfcc,&rcut,&rshow);
      if (ll &&  m->detail/2.0 <= rshow && m->detail >= rcut) {
	put_label(m,aprssymbol(&k->cfcc),k->name,ll,rshow,&k->cfcc);
	if (debug&DEBUG_LABELS)
	  fprintf(stderr,"label: kgl %0x cfcc %c%02d name %s at %d/%d\n",
		  k,k->cfcc.class,k->cfcc.level,k->name,ll->lat,ll->lon);
      }
    }
  }
  flush_labels(m);
}

put_label(m,symbol,name,ll,range,cfcc)
struct map *m;
char *symbol,*name;
struct latlon *ll;
double range;
struct feat *cfcc;
{
  struct label *newl = (struct label *)calloc(1,sizeof(struct label));
  if (!newl) {
    fprintf(stderr,"failed to calloc label\n");
    return;
  }
  strncpy(newl->symbol,symbol,sizeof(newl->symbol));
  strncpy(newl->name,name,sizeof(newl->name));
  newl->len = strlen(newl->name);
  newl->ll = *ll;
  newl->range = range;
  if (cfcc)
    newl->cfcc = *cfcc;
  if (m->lastl) {
    m->lastl->next = newl;
    m->lastl = newl;
  } else {
    m->labels = m->lastl = newl;
  }
}

flush_labels(m)
struct map *m;
{
  struct label *l;
  int rows;

  clean_labels(m);
  for (rows = 0, l = m->labels; l; l = l->next) {
    if (rows >= APRSLABS) {
      fprintf(stderr,"map%03d: WARNING: too many labels (%d)\n",m->mapid,rows);
      return;
    }      
    if (!(l->flags&LABEL_IGN))
      rows += print_label(m,l->symbol,l->name,&l->ll,l->range);
  }
}

/* try to eliminate redundant labels */
clean_labels(m)
struct map *m;
{
  struct label *i,*j;
  double d;

  for (i = m->labels; i; i = i->next) {
    struct latlon *lli,*llj;

    if (i->flags&LABEL_IGN)
      continue;
    for (j = i->next; j ; j = j->next) { /* search for a dupe */
      if (j->flags&LABEL_IGN)
	continue;
      if ((d=distance(&i->ll,&j->ll)) < m->detail/2.0) {
	if (debug&DEBUG_LABELS) {
	  fprintf(stderr,"clean_labels: %0x too close (%.2f) to %0x...\n",i,d,j);
	  fprintf(stderr,"label: %0x range %f cfcc %c%02d name %s at %d/%d\n",
		    i,i->range,i->cfcc.class,i->cfcc.level,i->name,i->ll.lat,i->ll.lon);
	  fprintf(stderr,"label: %0x range %f cfcc %c%02d name %s at %d/%d\n",
		    j,j->range,j->cfcc.class,j->cfcc.level,j->name,j->ll.lat,j->ll.lon);
	}
	if (i->len > 0)	{	/* allow nameless symbols to overlap */
	  i->flags |= LABEL_IGN;
	  if (debug&DEBUG_LABELS)
	    fprintf(stderr,"label: %0x will be ignored.\n",i);
	}
      }
    }
  }      
}

debug_land(l,c,ll)
struct land *l;
char *c;
struct latlon *ll;
{
  fprintf(stderr,"label: %s land %0x st.cty/landid %02d%03d/%d cfcc %c%02d name %s at %d/%d\n",
	  c,l,l->state,l->county,l->landid,l->cfcc.class,l->cfcc.level,l->name,ll->lat,ll->lon);
}

int
print_label(m,symbol,name,ll,range)
struct map *m;
char *symbol,*name;
struct latlon *ll;
double range;
{
  char *word[6],sep[6],buf[APRSLMAX+1];
  int len[6];
  int i,n,l,down,lat = ll->lat;
  int rows = 0;

  /* break up into words and find their lengths */
  if (strlen(name) > APRSLMAX && strpbrk(name," -")) {
    down = 25*1000000/m->ppdy;	/* text is about 25 pixels high */
    for (n = 0, word[0] = name;
	 n < 5 && (word[n+1]=strpbrk(word[n]," -")); 
	 n++) {
      sep[n] = *word[n+1];
      len[n] = word[n+1]++ - word[n];
      if (len[n] > APRSLMAX) {	/* need to truncate or abbreviate */
	len[n] = APRSLMAX;
      }
    }
    len[n] = strlen(word[n]);
    sep[n] = '\0';
    n++;
  } else {
    word[0] = name;
    len[0] = strlen(name);
    if (len[0] > APRSLMAX)
      len[0] = APRSLMAX;
    sep[0] = '\0';
    n=1;
  }
  /* glue the words back together in chunks up to APRSLMAX */
  for (l = i = 0; i < n;) {
    if ((l + len[i]) <= APRSLMAX) {
      bcopy(word[i],&buf[l],len[i]);
      l += len[i];
      buf[l++] = sep[i];
      i++;
    } else {
      buf[--l] = '\0';
      if (*buf||*symbol) { 	/* can't have null symbol & name! */
	printf("%s%s,%d.%06d,%d.%06d,%.1f\r\n",
	       symbol,buf,
	       lat/1000000,
	       lat%1000000,
	       ll->lon/1000000,
	       ll->lon%1000000,
	       range);
	++rows;
	lat -= down;
	symbol = "";		/* clobber symbol after 1st time output */
	l = 0;
      }
    }
  }
  if (l) {			/* catch straggler */
    buf[l] = '\0';
    if (*buf||*symbol) {
      printf("%s%s,%d.%06d,%d.%06d,%.1f\r\n",
	     symbol,
	     buf,
	     lat/1000000,
	     lat%1000000,
	     ll->lon/1000000,
	     ll->lon%1000000,
	     range);
      ++rows;
    }
  }
  return rows;
}

/* non-null terminated stuff... */
u_long
strntoul(bp,base,len)
char *bp;
int base,len;
{
  char buf[50];

  bcopy(bp,buf,len);
  buf[len] = '\0';
  return strtoul(buf,NULL,base);
}

int
atoil(cp,len)
char *cp;
int len;
{
  char buf[50];

  bcopy(cp,buf,len);
  buf[len] = '\0';
  return atoi(buf);
}

/*
 * APRS x,y coordinate system:
 *  N.W. corner of map (top left) is origin.
 *  x coordinates grow from West to East (increasing longitude)
 *  y coordinates grow from North to South (decreasing latitude)
 */

/*
 * These still need to truncate at map edges.  The right way to
 * truncate is to calculate the intercept of the line from the
 * previous point to this point.  For now, if the point is off-map,
 * just toss it and see how well that works. 
 */
ll2xy(m,ll,x,y)
struct map *m;
struct latlon *ll;
int *x,*y;
{
  if (verbose > 3) {
    fprintf(stderr,"ll2xy: %d/%d -> ",ll->lat,ll->lon);
  }
  *x = (int)(.5+((double)(m->max.lon - ll->lon)*m->ppdy)/1000000.0);
  *y = (int)(.5+((double)(m->max.lat - ll->lat)*m->ppdy)/1000000.0);
  if (verbose > 3)
    fprintf(stderr,"%d/%d\n",*x,*y);
}

static int lastx[2],lasty[2],lastlocked;
static int out,started;
static struct seg *thelink;
static struct latlon lastll;

putnew(m,l,count)		/* start a new feature */
struct map *m;
struct seg *l;
int count;
{
  thelink = l;
  lastx[0] = lasty[0] = -1;
  lastx[1] = lasty[1] = -1;
  lastlocked = 0;
  started = out = 0;
  lastll.lat = lastll.lon = -1;
}

char *
feat_name(s)
struct seg *s;
{
  struct name *n = &s->name;
  static char buf[sizeof(struct name)+5];

  sprintf(buf,"(%c%02d)%s%s%s%s%s%s%s",
	  s->feat.class,s->feat.level,
	  n->fedirp,
	  (*n->fedirp)?" ":"", n->fename,
	  (*n->fename)?" ":"", n->fetype,
	  (*n->fetype)?" ":"", n->fedirs);
  return buf;
}

flushnew(count)
int count;
{
  struct seg *l = thelink;

  if (started)
    return;
  started = 1;
  ++ptsout;			/* the feature start counts against maxpts */
  if (!count) {
    printf("0,0\r\n"); 
    printf(" %d ,",aprscolor(&l->feat));
    if (all_tlid) {		/* list all tlid's for this road */
      for (; l; l = l->ns) {
	printf(" %s %d",feat_name(l),l->tlid);
	if (l->flags&(SEG_LKFR|SEG_LKTO))
	  printf("[%s%s]",(l->flags&SEG_LKFR)?"F":"",(l->flags&SEG_LKTO)?"T":"");
      }
    } else {
      printf("%s %d",feat_name(l),l->tlid);
    }
    printf("\r\n");
  }
}


putxxx(m,l,count)			/* start a new feature */
struct map *m;
struct seg *l;
int count;
{
  if (!count)
    printf("; %d ,%d %s %s %s %s\r\n",aprscolor(&l->feat),l->tlid,
	   l->name.fedirp,l->name.fename,l->name.fetype,
	   l->name.fedirs);
}

/* 
 * Point "noise" reduction techniques used:
 *
 * 1. `Identical' points with +-fuzz are skipped.
 * 2. Certain types of segments (like water) can be marked for faster fuzzing
 *    which means the fuzz factor gets scaled up quicker in order to toss
 *    points for these classes of segments earlier than others that are
 *    more important to keep at higher resolution (like roads).
 * 3. If `locked' is set, then the point is an intersection and can't be
 *    fuzzed away without messing up intersections.
 * 4. Intermediate points of a straight line segment are eliminated:
 *     
 *                    c
 *                    x,y
 *     
 *    
 *             b
 *             x1,y1
 *     a
 *    x0,y0
 * 
 *         (x1-x0)        (x-x0)   if (abs(s1-s2) <= slopefuzz) then drop x1,y1
 * s1 =    -------   s2 = ------   and replace with x,y.  Line a-b-c 
 *         (y1-y0)        (y-y0)   becomes line a-c.
 */

putxy(m,s,ll,count,locked)
struct map *m;
struct seg *s;
struct latlon *ll;
int count,locked;
{
  int curfuzz,x,y;
  double curslope;

  if (isvis(m,ll)) {		/* point is within the map */
    ll2xy(m,ll,&x,&y);
  } else if (lastll.lat > 0 && !isvis(m,&lastll) && crosses(m,ll,&lastll)) { 
    /* line crosses map, find intercepts.  What do do about line that has
       one point in and one point out? */
    fprintf(stderr,"putxy: segment %d crosses from %d/%d to %d/%d XXX\n",
	    s->tlid,lastll.lat,lastll.lon,ll->lat,ll->lon);
    return;
  } else {			/* point is invisible */
    lastll = *ll;
    return;
  }

  if (x == 0)			/* prevent goofy zero divide condition */
    x = 1;
  if (y == 0)
    y = 1;
  
  if (locked && !(s->flags&SEG_FS)) { /* must plot this one accurately */
    curfuzz = 1;
  } else if(s->flags&SEG_FF) {
    curfuzz = m->fuzz<<1;	/* faster fuzz */
    curslope = m->slopefuzz*2.0;
  } else {
    curfuzz = m->fuzz;		/* plain fuzz */
    curslope = m->slopefuzz;
  }
  if (lastlocked)		/* lastx[1],lasty[1] was a locked point */
    curslope = 0.0;		/* so we can't bypass it */
  lastlocked = locked;

  if (verbose > 3) {
    fprintf(stderr,">>>putxy(%d,%d): last[0]=%d,%d last[1]=%d,%d out=%d\n",
	    x,y,lastx[0],lasty[0],lastx[1],lasty[1],out);
    fprintf(stderr,"...fuzz=%d abs(lastx-x)=%d abs(lasty-y)=%d\n",
	    curfuzz,abs(lastx[1]-x),abs(lasty[1]-y));
  }

#ifdef TEST
  printf("%d,%d\r\n",x,y);
#else
  if (out == 0) {		/* this is the first point */
    lastx[1] = x;		/* just stash them away */
    lasty[1] = y;
    ++ptsout;
    ++out;
  } else {			/* out >= 1 */
    /* test to see if this point is different enuf from previous */
    if(abs(lastx[1]-x)>=curfuzz || abs(lasty[1]-y)>=curfuzz) {
      if (out >= 2 && curslope >= 0.0) { /* straight line smoothing */
	int dx1 = lastx[1] - lastx[0];
	int dy1 = lasty[1] - lasty[0];
	int dx = x - lastx[0];
	int dy = y - lastx[0];
	double s1 = (dy1==0)?10000000.0:(double)dx1/(double)dy1;
	double s2 = (dy==0)?10000000.0:(double)dx/(double)dy;

	if (fabs(s1-s2) <= curslope) {
	  if (verbose > 3)
	    fprintf(stderr,"equal slopes: %d,%d -- %d,%d -- %d,%d %f %f\n",
		    lastx[0],lasty[0],lastx[1],lasty[1],x,y,s1,s2);
	  lastx[1] = x;		/* just toss the line's midpoint */
	  lasty[1] = y;
	  lastll = *ll;
	  return;		/* and act like we were never here. */
	}
      }
      if (out >= 2) {	/* don't print 'til we've collected 2 pts */
	if (!started) {	/* now we have enuf pts to start... */
	  flushnew(count);
	}
	if (!count)
	  printf("%d,%d\r\n",lastx[0],lasty[0]);
	if (verbose > 3)
	  fprintf(stderr,"...printing %d,%d\n",lastx[0],lasty[0]);
      }
      lastx[0] = lastx[1];	/* shift out the pts that were just printed */
      lasty[0] = lasty[1];
      lastx[1] = x;		/* hold these two for later */
      lasty[1] = y;
      ++ptsout;
      ++out;
    }
  }
#endif
  if (verbose > 3)
    fprintf(stderr,"<<<now ptsout=%d out=%d\n",ptsout,out);
  lastll = *ll;
}

putend(m,count)			/* end the road */
struct map *m;
int count;
{
  if (out >= 2) {
    if (!started) {
      flushnew(count);
    }
    if (!count) {
      printf("%d,%d\r\n",lastx[0],lasty[0]);
      printf("%d,%d\r\n",lastx[1],lasty[1]);
    }
  }
  lastx[0] = lasty[0] = -1;
  lastx[1] = lasty[1] = -1;
  lastlocked = 0;
  started = out = 0;
  lastll.lat = lastll.lon = -1;
}

