/* aprsdigi: APRS-style in-band UI frame digipeater and node.
 *
 *  I think in-band digipeating of UI frames belongs in the kernel, but I'm 
 *  not sure it's there:-)  Anyway, APRS has some peculiar digipeating ideas,
 *  so let's test them here for now:
 *
 *  1. Several digipeater aliases such as RELAY, WIDE, GATE.
 *  2. WIDE-N flooding algorithm.
 *  3. SSID-based shorthand for digi path.
 *  
 *  See Bob Bruninga's (WB4APR) README/MIC-E.TXT for a description of
 *  methods 2 and 3.  #1 is conventional TNC digipeating with MYAlias, etc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * (See the file "COPYING" that is included with this source distribution.)
 * 
 * derived from ax25-utils/listen.c
 *
 * portions Copyright (c) 1996,1997 E. Alan Crosswell
 * Alan Crosswell, N2YGK
 * 144 Washburn Road
 * Briarcliff Manor, NY 10510, USA
 * n2ygk@weca.org
 *
 * TODO:
 *  Combine unproto() and parsecalls().
 *  Figure out why I can't just use standard full_sockaddr instead of
 *   struct ax_calls.
 *  Sanity check args more rigorously.
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <linux/if.h>
#include <linux/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <linux/ax25.h>
#include <linux/rose.h>
#include <signal.h>
#include <errno.h>
#include "axconfig.h"
#include "axutils.h"
#include "mic_e.h"

#define	ALEN		6
#define	AXLEN		7
/* SSID mask and various flags that are stuffed into the SSID byte */
#define	SSID		0x1E
#define	HDLCAEB		0x01
#define	REPEATED	0x80

#define	UI		0x03	/* unnumbered information (unproto) */
#define	PID_NO_L3	0xF0	/* no level-3 (text) */
#define	C		0x80
#define	SSSID_SPARE	0x40	/* must be set if not used */
#define	ESSID_SPARE	0x20

static int digi_SSID = 0;
static char *tag = NULL;	/* tag onto end of posit in SSID mode */
static int taglen = 0;
static char *logfile = NULL;
static int idinterval = (9*60)+30; /* default to 9:30 */
#define MAXALIASES 5
static ax25_address aliases[MAXALIASES];
#define MYCALL aliases[0]
static int n_aliases = 1;
static ax25_address floods[MAXALIASES];
#define WIDECALL floods[0]
static int n_floods = 0;
static ax25_address aprscall;
static int have_digipath = 0;
#define DIGISIZE 7		/* one more than usual! */
static ax25_address digipath[DIGISIZE]; /* for SSID-[1:7] w/o flooding */
static struct full_sockaddr_ax25 path[4]; /* for SSID-[8:15] N S E W */
static int keep = 20;		/* seconds to remember for dupe test */
static int digi_mycall = 0;	/* replace digi alias w/mycall */
static int kill_dupes = 0;	/* kill dupes even in conventional mode */
static int mice_xlate = 1;	/* translate mic-e to tracker format */
static struct {
  int rx;			/* packets received */
  int tx;			/* packets transmitted, sum of: */
  int digi;			/*  regular digipeats */
  int flood;			/*  flood-N digipeats */
  int ssid;			/*  ssid digipeats */
  int ids;			/*  id packets */
} stats;

static int RXs = -1,TXs = -1;	/* receive and transmit sockets */
static struct sockaddr RXsa,TXsa; /* and their sockaddrs */
static int RXsize = sizeof(RXsa),TXsize = sizeof(TXsa);

static int unproto(struct full_sockaddr_ax25 *, char *);
static int parsecalls(ax25_address*, int, char *);
static void print_it(FILE *,struct ax_calls *,unsigned char *,int len);
static void add_text(u_char **,int *,u_char *,int,char *,int);
static int dupe_packet(struct ax_calls *,u_char *,int);
static int xmitaprs(int,struct sockaddr *,struct ax_calls *,u_char *,int,
		u_char *tag,int);
static int xmit(int,struct sockaddr *,struct ax_calls *,u_char *,int,
		u_char *tag,int);
static void set_id(void);
static int rx_packet(int fd, u_char *buffer, int len);

/* signal handlers */
static void cleanup(int),identify(int),print_stats(int),reset_stats(int);

static void do_opts(int argc, char **argv);
static void do_ports(void);
static void set_sig(void);
static int rx_loop(void);
static char *rxport = NULL,*txport = NULL;

int
main(int argc, char **argv)
{
  int r;

  do_opts(argc,argv);
  do_ports();
  bzero(&stats,sizeof(stats));
  set_id();
  set_sig();
  r = rx_loop();
  exit(r);
}

static int
rx_loop()
{
  unsigned char buffer[AX25_MTU];

  for (;;) {
    int len;
    int r;
    fd_set rmask;
    int rbits = (RXs>TXs)?RXs+1:TXs+1;

    FD_ZERO(&rmask);
    FD_SET(RXs, &rmask);
    FD_SET(TXs, &rmask);

    if ((r = select(rbits,&rmask,NULL,NULL,NULL)) < 0) {
      if (errno == EINTR)
	continue;
      perror("select");
      return 1;
    }
    if (FD_ISSET(RXs,&rmask)) {
      if ((len = recvfrom(RXs,buffer,sizeof(buffer),0,&RXsa,&RXsize)) == -1) {
	if (errno == EINTR)
	  continue;
	perror("recv");
	return 1;
      }
      if (!rx_packet(RXs,buffer,len))
	continue;
    }
    if (RXs != TXs && FD_ISSET(TXs,&rmask)) {
      if ((len = recvfrom(TXs,buffer,sizeof(buffer),0,&TXsa,&TXsize)) == -1) {
	if (errno == EINTR)
	  continue;
	perror("recv");
	return 1;
      }
      if (!rx_packet(TXs,buffer,len))
	continue;
    }
  } /* end of for(;;) */
  return 0;			/* not reached */
}

static int
rx_packet(int fd, u_char *buffer, int len)
{
  struct ax_calls calls,out;
  int r,ssid,wide;
  int mic_e;
  u_char *cp = buffer;
  int i;

  ++stats.rx;
  /* determine if this is a packet I care about */
  r = parse_kiss(&cp,&len,&calls);
#ifdef DEBUG
  fprintf(stderr,"RX: ");
  print_it(stderr,&calls,cp,len);
#endif
  if (r == PK_INVALID)
    return 0;			/* invalid */
  if (r != PK_VALDIGI && !digi_SSID)
    return 0;			/* not doing SSID & no unrepeated digis */
  /* XXX -- If packet was last transmitted by me, then don't digipeat it! */
  if (kill_dupes && dupe_packet(&calls,cp,len)) {
#ifdef DEBUG
    fprintf(stderr,"packet is a dupe\n");
#endif
    return 0;
    }
#ifndef TEST
  /* If packet is addressed to one of my calls, then don't digipeat it! */
  for (i = 0; i < n_aliases; i++) {
    if (ax25cmp(&aliases[i],&calls.ax_to_call) == 0) {
      return 0;
    }
  }
#endif
  /* 
   * MIC-E SSID path selection:
   *  1. SSID must be non zero.
   *  2. Digipeater path must be empty.
   *  3. 1st byte of info field must be control-\ (SSID Cookie)
   */
  mic_e = (*cp == 0x60 || *cp == 0x27 || *cp == 0x1c || *cp == 0x1d);
  if (digi_SSID && calls.ax_n_digis == 0 
      && (ssid = (calls.ax_to_call.ax25_call[ALEN]&SSID)>>1)
      && len > 0 && mic_e) {
#ifdef DEBUG
    char dirs[5] = "NSEW";

    fprintf(stderr,"Got an SSID route for path %d.\n",ssid);
#endif
    out.ax_from_call = calls.ax_from_call;
    out.ax_to_call = calls.ax_to_call;
    out.ax_to_call.ax25_call[ALEN]&=~SSID; /* zero the SSID */
    out.ax_type = calls.ax_type;
    out.ax_pid = calls.ax_pid;
    if (ssid <= 7) {	/* omnidirectional */
      if (!have_digipath || ssid == 0) { /* in a flooding network? */
#ifdef DEBUG
	fprintf(stderr,"setting path to %s-%d\n",
		ax2asc(&WIDECALL),ssid);
#endif
	out.ax_n_digis = 1;
	out.ax_digi_call[0] = WIDECALL;
	out.ax_digi_call[0].ax25_call[ALEN] |= SSID&(ssid<<1);
      } else {		/* not in a flooding network */
	int startat,i;
	if (ssid < 4) {
	  startat = 0;	/* starting point in digipath */
	  out.ax_n_digis = ssid; /* number of digis from there. */
	} else {
	  startat = 3;
	  out.ax_n_digis = ssid-3;
	}
#ifdef DEBUG
	fprintf(stderr,"converting SSID WIDE-%d path to DIGI[%d:%d]\n",
		ssid,startat,startat+out.ax_n_digis-1);
#endif
	for (i = 0; i < out.ax_n_digis; i++,startat++) {
	  out.ax_digi_call[i] = digipath[startat];
	}
      }
    } else {		/* directional */
      int j;
#ifdef DEBUG
      fprintf(stderr,"setting path to %c UNPROTO%s\n",
	      dirs[ssid&3],(ssid&4)?" + WIDE":"");
#endif
      out.ax_n_digis = path[ssid&3].fsa_ax25.sax25_ndigis+1;
      out.ax_digi_call[0] = MYCALL; /* I am the 1st digi */
      out.ax_digi_call[0].ax25_call[ALEN] |= REPEATED; 
      for (j = 1; j < out.ax_n_digis; j++)
	out.ax_digi_call[j] = path[ssid&3].fsa_digipeater[j-1];
      if (ssid&4) {	/* directional + wide call */
	out.ax_digi_call[out.ax_n_digis++] = WIDECALL;
      }
    }
    ++stats.ssid;
    if (xmitaprs(TXs,&TXsa,&out,cp,len,tag,taglen) < 0)
      perror("xmitaprs");
    return 0;
  }
  
  /* FLOOD-N algorithm: one digi with non-zero ssid callsign */
  if (!have_digipath && n_floods && calls.ax_n_digis == 1 
      && (wide = (calls.ax_digi_call[0].ax25_call[ALEN]&SSID)>>1)) {
    for (i = 0; i < n_floods; i++) {
      if ((floods[i].ax25_call[ALEN]&SSID) == 0
	  && ax25cmp(&floods[i],&calls.ax_digi_call[0]) == 2) {
#ifdef DEBUG
	fprintf(stderr,"Got a flooding %s route.\n",
		ax2asc(&calls.ax_digi_call[0]));
#endif
	/* don't forget to save packet for a minute to prevent dupes! */
	if (dupe_packet(&calls,cp,len)) {
#ifdef DEBUG
	  fprintf(stderr,"packet is a dupe\n");
#endif
	  break;
	}
	calls.ax_digi_call[0].ax25_call[ALEN] &= ~SSID;
	calls.ax_digi_call[0].ax25_call[ALEN] |= ((wide-1) << 1)&SSID;
#ifdef DEBUG
	fprintf(stderr,"Rewriting it as %s:\n",
		ax2asc(&calls.ax_digi_call[0]));
#endif
	++stats.flood;
	if (xmitaprs(TXs,&TXsa,&calls,cp,len,0,0) < 0) 
	  perror("xmitaprs");
	break;
      }
    }
    return 0;
  }
  /* conventional: see if the next digipeater matches one of my calls */
  for (i = 0; i < n_aliases; i++) {
    if (ax25cmp(&aliases[i],&calls.ax_digi_call[calls.ax_next_digi]) == 0) {
      /* finally, a packet for me to digipeat */
#ifdef DEBUG
      fprintf(stderr,"Got a conventional digipeat.\n");
#endif
      if (digi_mycall)
	calls.ax_digi_call[calls.ax_next_digi] = MYCALL;
      calls.ax_digi_call[calls.ax_next_digi].ax25_call[ALEN] |= REPEATED;
      ++stats.digi;
      if (xmitaprs(TXs,&TXsa,&calls,cp,len,0,0) < 0) 
	perror("xmitaprs");
      break;
    }
  }
  return 1;
}	  

static void
add_text(op,oleft,text,len,tag,taglen)
unsigned char **op;
int *oleft;
u_char *text;
int len;
char *tag;
int taglen;
{
  if ((*oleft -= len) > 0) {
    bcopy(text,*op,len);	/* copy the text */
    *op += len;
  }
  if (taglen && tag && (*oleft -= taglen) > 0) {
    bcopy(tag,*op,taglen); /* and tack on the tag */
    *op += taglen;
  }
}

/* XXX - watch out for overflow when adding mycall and/or wide */
static int
unproto(calls,str)		/* parse a via path into a calls struct */
struct full_sockaddr_ax25 *calls;
char *str;
{
  char buf[200];

  bzero(calls, sizeof(*calls));
  sprintf(buf,"dummy via %s",str);
  return convert_call(buf,calls);
}

static int
parsecalls(calls,ncalls,str)	/* parse a via path into a calls struct */
ax25_address *calls;
int ncalls;			/* max number */
char *str;
{
  char *cp;
  int i;

  bzero(calls,ncalls*sizeof(*calls));
  cp = strtok(str," \t\n,");
  for (i = 0; cp && i < ncalls; i++) {
    if (convert_call_entry(cp,calls[i].ax25_call) < 0)
      return -1;
    cp = strtok(NULL," \t\n,");
  }
  return i;
}

static void
print_it(f,calls,data,len)
FILE *f;
struct ax_calls *calls;
unsigned char *data;
int len;
{
  int j;
  char asc_from[12],asc_to[12];

  if (f == NULL)
    return;
  strncpy(asc_to,ax2asc(&calls->ax_to_call),sizeof(asc_to));
  strncpy(asc_from,ax2asc(&calls->ax_from_call),sizeof(asc_from));
  fprintf(f,"%s>%s",asc_from,asc_to);
  for (j = 0; j < calls->ax_n_digis; j++) {
    fprintf(f,",%s%s",ax2asc(&calls->ax_digi_call[j]),
	    (j == calls->ax_next_digi-1)?"*":"");
  }
  fprintf(f,":%.*s\n",len,data);
}

/* packet transmitter */
static int
xmitaprs(s,sa,calls,cp,clen,tag,taglen)
int s;
struct sockaddr *sa;
struct ax_calls *calls;
u_char *cp;			/* original data payload */
int clen;			/*   length of it */
u_char *tag;			/* text to tag on to the end */
int taglen;			/*   and it's length */
{
  int r;
  unsigned char mic1[AX25_MTU], mic2[AX25_MTU];
  int l1,l2;
  time_t now;

  time(&now);
  if (mice_xlate && 
      fmt_mic_e(ax2asc(&calls->ax_to_call),cp,clen,mic1,&l1,mic2,&l2,now)) {
    calls->ax_to_call = aprscall; /* replace compressed lat w/"APRS" */
    if (l1)
      if ((r = xmit(s,sa,calls,mic1,l1,tag,taglen)) < 0) 
	return r;
    if (l2)
      if ((r = xmit(s,sa,calls,mic2,l2,tag,taglen)) < 0) 
	return r;
    return 0;
  } else {
    return xmit(s,sa,calls,cp,clen,tag,taglen);
  }
}

static int
xmit(s,sa,calls,cp,clen,tag,taglen)
int s;
struct sockaddr *sa;
struct ax_calls *calls;
u_char *cp;			/* original data payload */
int clen;			/*   length of it */
u_char *tag;			/* text to tag on to the end */
int taglen;			/*   and it's length */
{
  u_char obuf[AX25_MTU];
  u_char *op = obuf;
  int oleft = sizeof(obuf);
  int olen;
  int r;

  gen_kiss(&op,&oleft,calls);	/* generate the kiss header */
  add_text(&op,&oleft,cp,clen,tag,taglen); /* fill in the info field */
  olen = sizeof(obuf) - oleft;
  ++calls->ax_next_digi;
#ifdef DEBUG
  fprintf(stderr,"TX: ");
  print_it(stderr,calls,op-(clen+taglen),clen+taglen);
#endif
  if (logfile) {
    FILE *of;
    if (logfile[0] == '-' && logfile[1] == '\0')
      of = stdout;
    else
      of = fopen(logfile,"a");
    print_it(of,calls,op-(clen+taglen),clen+taglen);
    if (of != stdout)
      fclose(of);
  }
  if (idinterval >= 0)
    alarm(idinterval);		/* keep it legal & schedule an ID packet */
  ++stats.tx;
  r = sendto(s,obuf,olen,0,sa,sizeof(*sa)); /* ship it off */;
  return r;
}


/* packet dupe checking */
struct pkt {
  struct pkt *next;
  time_t t;			/* when recevied */
  ax25_address to;		/* destination */
  ax25_address fr;		/* source */
  int l;			/* length of text */
  u_char d[AX25_MTU];		/* the text */
};

static struct pkt *top = NULL;	/* stacked in reverse chronological order */

static int
dupe_packet(calls,cp,len)
struct ax_calls *calls;
u_char *cp;
int len;
{
  struct pkt *p,*new,*prev;
  time_t now = time(0);
  time_t stale = now - keep;

  /* first toss stale packets */
  for (prev = new = NULL, p = top; p; p = (prev=p)->next) { 
    if (p->t < stale) {		/* all from here on are stale */
      struct pkt *n;

      if (prev)			/* this is new end of list */
	prev->next = NULL;
      if (!new) {
	new = p;		/* reuse this buffer in a minute */
	p = p->next;
      }
      for (n=NULL; p; p=n) { /* toss the rest */
	n = p->next;
	free(p);
      }
      break;
    } else {			/* not stale, so compare it I guess */
      if (p->l == len && bcmp(p->d,cp,len) == 0
	  && bcmp(&calls->ax_to_call,&p->to,sizeof(p->to)) == 0
	  && bcmp(&calls->ax_from_call,&p->fr,sizeof(p->fr)) == 0) {
	/* a match!  Update the timestamp and move to front */
	if (new)
	  free(new);		/* didn't need it */
	p->t = now;		/* updated timestamp */
	if (p == top)		/* already on top */
	  return 1;
	if (prev)
	  prev->next = p->next;
	p->next = top;
	top=p;			/* move to top of stack */
	return 1;
      }
    }
  }
  /* fall through means no match so we need to save it for later */
  if (!new)
    new = (struct pkt *)calloc(1,sizeof(struct pkt));
  if (!new) {
    fprintf(stderr,"failed to calloc!\n");
    return 0;			
  }
  new->t = now;
  new->l = (len>AX25_MTU)?AX25_MTU:len;
  bcopy(cp,new->d,new->l);
  new->to = calls->ax_to_call;
  new->fr = calls->ax_from_call;
  new->next = top;
  top = new;
  return 0;
}

static void
cleanup(sig)
int sig;			/* dont't really care what it was */
{
  if (alarm(0) > 0)		/* an alarm was pending, so... */
    identify(sig);		/*  ID one last time */
  print_stats(sig);		/* dump final stats */
  exit(0);			/* and exit */
}

static void
print_stats(sig)
int sig;
{
  FILE *of = stderr;
  
  if (logfile) {
    if (logfile[0] == '-' && logfile[1] == '\0')
      of = stdout;
    else
      of = fopen(logfile,"a");
  }
  if (of == NULL)
    return;
  fprintf(of,"# rx %d tx %d digi %d flood %d ssid %d ids %d\n",
	  stats.rx,stats.tx,stats.digi,stats.flood,stats.ssid,stats.ids);
  if (of != stdout)
    fclose(of);
  if (sig >= 0)
    signal(sig,print_stats);
}

static void
reset_stats(sig)
int sig;
{
  print_stats(-1);
  bzero(&stats,sizeof(stats));
  signal(sig,reset_stats);
}

static u_char idbuf[AX25_MTU];
static int idlen = 0;

/* setup a bare minimum legal ID (don't QRM the world with a digi path!) */
static void
set_id()
{
  struct ax_calls id;
  u_char *op = idbuf;
  int oleft = sizeof(idbuf);
  char idinfo[AX25_MTU];
  int i;

  bzero(&id,sizeof(id));
  convert_call_entry("ID",id.ax_to_call.ax25_call);
  id.ax_to_call.ax25_call[ALEN] |= C; /* it's a command */
  id.ax_from_call = MYCALL;
  id.ax_type = UI;
  id.ax_pid = PID_NO_L3;
  id.ax_n_digis = 0;

  gen_kiss(&op,&oleft,&id);	/* generate the kiss header */
  *idinfo = '\0';
  for (i = 0; i < n_aliases; i++) {
    strcat(idinfo,ax2asc(&aliases[i]));
    strcat(idinfo,"/R ");
  }
  if (!have_digipath)		/* overrides flooding */
    for (i = 0; i < n_floods; i++) {
      strcat(idinfo,ax2asc(&floods[i]));
      strcat(idinfo,"-n/R ");
    }
  add_text(&op,&oleft,idinfo,strlen(idinfo),0,0);
  idlen = sizeof(idbuf) - oleft;
}

static void
identify(sig)			/* wake up and ID! */
int sig;
{
  alarm(0);			/* we don't wake up again without new reason */
  if (sendto(TXs,idbuf,idlen,0,&TXsa,TXsize) < 0) /* ship it off */
    perror("sendto");
  ++stats.ids;
  signal(sig,identify);
}

static void
do_opts(int argc, char **argv)
{
  char s;

  while ((s = getopt(argc, argv, "a:f:n:s:e:w:t:k:l:i:d:r:x:mDM")) != -1) {
    int p = -1;			/* used for NSEW hack, below */
    switch (s) {
    case 'a':			/* aliases to listen to */
      if (n_aliases >= MAXALIASES) {
	fprintf(stderr,"too many aliases!\n");
	exit(1);
      }
      if (convert_call_entry(optarg,aliases[n_aliases++].ax25_call) < 0) {
	fprintf(stderr,"Don't understand callsign %s\n",optarg);
	exit(1);
      }
      break;
    case 'f':			/* flooding algorithm alias */
      if (n_floods >= MAXALIASES) {
	fprintf(stderr,"too many flooding aliases!\n");
	exit(1);
      }
      if (convert_call_entry(optarg,floods[n_floods++].ax25_call) < 0) {
	fprintf(stderr,"Don't understand callsign %s\n",optarg);
	exit(1);
      }
      break;
    case 'w':			/* directional unproto path */
      ++p;
    case 'e':
      ++p;
    case 's':
      ++p;
    case 'n':
      ++p;
      ++digi_SSID;
      if (unproto(&path[p],optarg) < 0) {
	fprintf(stderr,"Don't understand %c path %s\n",toupper(s),optarg);
	exit(1);
      }
      break;
    case 'd':			/* DIGI-style(non-SSID) unproto path */
      ++have_digipath;
      if (parsecalls(digipath,DIGISIZE,optarg)!=DIGISIZE) {
	fprintf(stderr,"Don't understand %c path %s\n",toupper(s),optarg);
	exit(1);
      }
      break;
    case 't':			/* tag to tack on to end of all packets */
      tag = optarg;
      taglen = strlen(optarg);
      break;
    case 'k':
      if ((keep = atoi(optarg)) <= 0)
	keep = 20;		/* default keep is 20 */
      break;
    case 'l':
      logfile = optarg;		/* log digipeated packets */
      break;
    case 'i':
      idinterval = atoi(optarg);
      break;
    case 'r':
      rxport = optarg;
      break;
    case 'x':
      txport = optarg;
      break;
    case 'm':
      digi_mycall++;
      break;
    case 'D':
      kill_dupes++;
      break;
    case 'M':
      mice_xlate=0;
      break;
    case '?':
      fprintf(stderr,"Usage: aprsdigi [-mDM] [-a call] [-n|s|e|w path] [-f call] [-t tag] [-k secs] [-l logfile] [-i interval] [-r rxport] [-x txport]\n");
      exit(1);
    }
  }
  if (rxport == NULL)		/* let them specify just one for both */
    rxport = txport;
  if (txport == NULL)
    txport = rxport;

  if (rxport == NULL) {
    fprintf(stderr,"aprsdigi: you must supply one or both of -r or -x\n");
    exit(1);
  }
}
static void
do_ports()
{
  struct ifreq ifr;
  char *rxdev = NULL, *txdev = NULL;
#ifdef TEST
  int proto = ETH_P_ALL;
#else
  int proto = ETH_P_AX25;
#endif

  convert_call_entry("APRS",aprscall.ax25_call);
  if (ax25_config_load_ports() == 0)
    fprintf(stderr, "aprsdigi: no AX.25 port data configured\n");
  
  if ((rxdev = ax25_config_get_dev(rxport)) == NULL) {
    fprintf(stderr, "aprsmon: invalid rx port name - %s\n", rxport);
    exit(1);
  }

  if ((RXs = socket(AF_INET, SOCK_PACKET, htons(proto))) == -1) {
    perror("socket");
    exit(1);
  }
  strcpy(RXsa.sa_data, rxdev);
  RXsa.sa_family = AF_AX25;	/* AF_INET?? */
  if (bind(RXs, &RXsa, sizeof(struct sockaddr)) < 0) {
    perror("bind");
    exit(1);
  }
  strcpy(ifr.ifr_name, rxdev);	/* get this port's callsign */
  if (ioctl(RXs, SIOCGIFHWADDR, &ifr) < 0) {
    perror("GIFADDR");
    exit(1);
  }
  if (ifr.ifr_hwaddr.sa_family != AF_AX25) {
    fprintf(stderr,"%s: not AX25\n",rxport);
    exit(1);
  }
  if (strcmp(txport,rxport) == 0) { /* tx/rx are the same device */
    txdev = rxdev;
    TXs = RXs;
    TXsa = RXsa;
  } else {
    if ((txdev = ax25_config_get_dev(txport)) == NULL) {
      fprintf(stderr, "aprsmon: invalid tx port name - %s\n", txport);
      exit(1);
    }

    if ((TXs = socket(AF_INET, SOCK_PACKET, htons(proto))) == -1) {
      perror("socket");
      exit(1);
    }
    bzero(&TXsa,sizeof(TXsa));
    TXsa.sa_family = AF_AX25;	/* AF_INET?? */
    strcpy(TXsa.sa_data,txdev);
    if (bind(TXs, &TXsa, sizeof(struct sockaddr)) < 0) {
      perror("bind");
      exit(1);
    }
    strcpy(ifr.ifr_name, txdev);	/* get this port's callsign */
    if (ioctl(TXs, SIOCGIFHWADDR, &ifr) < 0) {
      perror("GIFADDR");
      exit(1);
    }
    if (ifr.ifr_hwaddr.sa_family != AF_AX25) {
      fprintf(stderr,"%s: not AX25\n",txport);
      exit(1);
    }
  }
  /* set mycall to the tx port's */
  bcopy(ifr.ifr_hwaddr.sa_data,MYCALL.ax25_call,sizeof(MYCALL.ax25_call));

  MYCALL.ax25_call[ALEN] |= ESSID_SPARE|SSSID_SPARE;
}

static void
set_sig()
{
  signal(SIGHUP,cleanup);
  signal(SIGINT,cleanup);
  signal(SIGQUIT,cleanup);
  signal(SIGTERM,cleanup);
  signal(SIGPWR,cleanup);
  signal(SIGPIPE,cleanup);
  signal(SIGUSR1,print_stats);
  signal(SIGUSR2,reset_stats);
  signal(SIGALRM,identify);
}
