/* SMTP Server state machine - see RFC 821
 *  enhanced 4/88 Dave Trulli nn2z
 */
#include <stdio.h>
#include <time.h>
#ifdef UNIX
#include <sys/types.h>
#endif
#if	defined(__STDC__) || defined(__TURBOC__)
#include <stdarg.h>
#endif
#include <ctype.h>
#include <setjmp.h>
#include "global.h"
#include "config.h"
#include "mbuf.h"
#include "cmdparse.h"
#include "socket.h"
#ifdef LZW
#include "lzw.h"
#endif
#include "iface.h"
#include "proc.h"
#include "smtp.h"
#include "commands.h"
#include "dirutil.h"
#include "mailbox.h"
#include "bm.h"
#ifdef NNTP
#include "nntp.h"
#endif
#ifdef STATUS		/* value not used - keep lint happy */
#undef STATUS
#endif
#include "domain.h"
#include "files.h"

static char *Days[7] = {  "Sun","Mon","Tue","Wed","Thu","Fri","Sat" };
char *Months[12] = { "Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec" };

static struct list * near expandalias __ARGS((struct list **head,char *user));
static int near getmsgtxt __ARGS((struct smtpsv *mp));
static struct smtpsv * near mail_create __ARGS((void));
static void near mail_clean __ARGS((struct smtpsv *mp));
static int near mailit __ARGS((FILE *data,char *from,struct list *tolist));
static int near router_queue __ARGS((FILE *data,char *from,struct list *to));
static void near smtplog __ARGS((char *fmt,...));
static void smtpserv __ARGS((int s,void *unused,void *p));
static int near mailuser __ARGS((FILE *data,char *from,char *to));

/* Reply messages */
static char Help[] = "214-Commands:\n214-HELO NOOP MAIL QUIT RCPT HELP DATA RSET EXPN\n214 End\n";
static char SmtpBanner[] = "220 %s SMTP ready\n";
static char Closing[] = "221 Closing\n";
static char Ok[] = "250 Ok\n";
static char Reset[] = "250 Reset state\n";
static char Sent[] = "250 Sent\n";
static char Ourname[] = "250 Hello, pleased to meet you\n";
#ifdef LZW
static char LZWOk[] = "252 LZW Ok\n";
#endif
static char Enter[] = "354 Enter mail, end with .\n";
static char Lowmem[] = "421 System overloaded\n";
static char Ioerr[] = "452 Temp file write error\n";
static char Badcmd[] = "500 Command unrecognized\n";
static char Syntax[] = "501 Syntax error\n";
static char Needrcpt[] = "503 Need RCPT (recipient)\n";
static char Unknown[] = "550 <%s> address unknown\n";
static char Noalias[] = "550 No alias for <%s>\n";

static int Ssmtp = -1; /* prototype socket for service */

/* Start up SMTP receiver service */
int
smtp1(int argc,char *argv[],void *p)
{
	struct sockaddr_in lsocket;
	int s;

	if(Ssmtp != -1){
		return 0;
	}
	psignal(Curproc,0);	/* Don't keep the parser waiting */
	chname(Curproc,"SMTP listener");

	lsocket.sin_family = AF_INET;
	lsocket.sin_addr.s_addr = INADDR_ANY;
	lsocket.sin_port = (argc < 2) ? IPPORT_SMTP : atoi(argv[1]);

	Ssmtp = socket(AF_INET,SOCK_STREAM,0);
	bind(Ssmtp,(char *)&lsocket,sizeof(lsocket));
	listen(Ssmtp,1);
	for(;;){
		if((s = accept(Ssmtp,NULLCHAR,(int *)NULL)) == -1)
			break;	/* Service is shutting down */

		if(availmem() < Memthresh){
			usputs(s,Lowmem);
			shutdown(s,1);
		} else {
			/* Spawn a server */
			newproc("SMTP server",2816,smtpserv,s,NULL,NULL,0);
		}
	}
	return 0;
}

/* Shutdown SMTP service (existing connections are allowed to finish) */
int
smtp0(int argc,char *argv[],void *p)
{
	close_s(Ssmtp);
	Ssmtp = -1;
	return 0;
}

static void
smtpserv(int s,void *unused,void *p)
{
	struct smtpsv *mp;
	char **cmdp,*arg,*cp,*cmd,*newaddr,buf[LINELEN];
	struct list *ap,*list;
	int cnt;
	char address_type;
#ifdef LZW
	extern int Smtplzw;
	int lzwbits, lzwmode;
#endif

	/* Command table */
	char *commands[] = {
		"helo",
	#define	HELO_CMD	0
		"noop",
	#define	NOOP_CMD	1
		"mail from:",
	#define	MAIL_CMD	2
		"quit",
	#define	QUIT_CMD	3
		"rcpt to:",
	#define	RCPT_CMD	4
		"help",
	#define	HELP_CMD	5
		"data",
	#define	DATA_CMD	6
		"rset",
	#define	RSET_CMD	7
		"expn",
	#define EXPN_CMD	8
		"xlzw",
	#define XLZW_CMD	9
		NULLCHAR
	};

	sockmode(s,SOCK_ASCII);
	sockowner(s,Curproc);		/* We own it now */
	log(s,"SMTP open");

	if((mp = mail_create()) == NULLSMTPSV){
		tputs(Nospace);
		goto quit;
	}
	mp->s = s;

	usprintf(mp->s,SmtpBanner,Hostname);
loop:
	if ((cnt = recvline(s,buf,LINELEN)) == -1) {
		/* He closed on us */
		goto quit;
	}
	if(cnt < 4){
		/* Can't be a legal command */
		usputs(mp->s,Badcmd);
		goto loop;
	}
	rip(buf);
	cmd = buf;

	/* Translate entire buffer to lower case */
	for(cp = cmd;*cp != '\0';cp++)
		*cp = tolower(*cp);

	/* Find command in table; if not present, return syntax error */
	for(cmdp = commands;*cmdp != NULLCHAR;cmdp++)
		if(strncmp(*cmdp,cmd,strlen(*cmdp)) == 0)
			break;
	if(*cmdp == NULLCHAR){
		usputs(mp->s,Badcmd);
		goto loop;
	}
	arg = &cmd[strlen(*cmdp)];
	/* Skip spaces after command */
	while(*arg == ' ')
		arg++;
	/* Execute specific command */
	switch((int)(cmdp - commands)) {
	case XLZW_CMD:
#ifdef LZW
		if(Smtplzw) {
			char *cp1;
			lzwmode = lzwbits = 0;

			cp1 = strchr(arg,' ');
			*cp1 = '\0';
			lzwbits = atoi(arg);
			cp1++;
			lzwmode = atoi(cp1);
			if(lzwbits > lzwmode) {
				usputs(mp->s,LZWOk);
				lzwinit(mp->s,lzwbits,lzwmode);
			} else {
				usputs(mp->s,Badcmd);
			}
		} else {
			usputs(mp->s,Badcmd);
		}
#else
		usputs(mp->s,Badcmd);
#endif
		break;
	case HELO_CMD:
		xfree(mp->system);
		mp->system = strxdup(arg);
		usputs(mp->s,Ourname);
		break;
	case NOOP_CMD:
		usputs(mp->s,Ok);
		break;
	case MAIL_CMD:
		if((cp = getname(arg)) == NULLCHAR){
			usputs(mp->s,Syntax);
			break;
		}
		xfree(mp->from);
		mp->from = strxdup(cp);
		usputs(mp->s,Ok);
		break;
	case QUIT_CMD:
		usputs(mp->s,Closing);
		goto quit;
	case RCPT_CMD:	/* Specify recipient */
		if((cp = getname(arg)) == NULLCHAR){
			usputs(mp->s,Syntax);
			break;
		}
		/* rewrite address if possible */
		if((newaddr = rewrite_address(cp)) != NULLCHAR) {
			strcpy(buf,newaddr);
			cp = buf;
			xfree(newaddr);
		}
		/* check if address is ok */
		if ((address_type = validate_address(cp)) == BADADDR) {
			usprintf(mp->s,Unknown,cp);
			break;
		}
		/* if a local address check for an alias */
		if (address_type == LOCAL) {
			expandalias(&mp->to, cp);
		} else {
			/* a remote address is added to the list */
			addlist(&mp->to, cp, address_type);
		}
		usputs(mp->s,Ok);
		break;
	case HELP_CMD:
		usputs(mp->s,Help);
		break;
	case DATA_CMD:
		if(mp->to == NULLLIST)
			usputs(mp->s,Needrcpt);
		else if ((mp->data = temp_file(0,1)) == NULLFILE)
			usputs(mp->s,Ioerr);
		else
			getmsgtxt(mp);
		break;
	case RSET_CMD:
		del_list(mp->to);
		mp->to = NULLLIST;
		usputs(mp->s,Reset);
		break;
	case EXPN_CMD:
		if (*arg == '\0') {
			usputs(mp->s,Syntax);
			break;
		}
		list = NULLLIST;
		/* rewrite address if possible */
		if((newaddr = rewrite_address(arg)) != NULLCHAR)
			if(strcmp(newaddr,arg) == 0) {
				xfree(newaddr);
				newaddr = NULLCHAR;
			} else {
				strcpy(buf,newaddr);
				arg = buf;
			}
		list = NULLLIST;
		newaddr = NULLCHAR;
		expandalias(&list,arg);
		if (strcmp(list->val,arg) == 0 && list->next == NULLLIST)
			if(newaddr == NULLCHAR) {
				usprintf(mp->s,Noalias,arg);
				del_list(list);
				break;
			}
		ap = list;
		while (ap->next != NULLLIST) {
			usprintf(mp->s,"250-%s\n",ap->val);
			ap = ap->next;
		}
		usprintf(mp->s,"250 %s\n",ap->val);
		del_list(list);
		xfree(newaddr);
		break;
	}
	goto loop;

quit:
	log(s,"SMTP close");
	close_s(s);
	if(mp != NULLSMTPSV) {
		mail_clean(mp);
		smtptick(NULL);			/* start SMTP daemon immediately */
	}
}

/* read the message text */
static int near
getmsgtxt(struct smtpsv *mp)
{
	char buf[LINELEN];
	char *p = buf;

	/* Add timestamp; ptime adds newline */
	fputs("Received: ",mp->data);
	if(mp->system != NULLCHAR)
		fprintf(mp->data,"from %s ",mp->system);
	fprintf(mp->data,"by %s with SMTP\n\tid AA%ld; %s",
			Hostname, get_msgid(), ptime(&currtime));
	if(ferror(mp->data)){
		usputs(mp->s,Ioerr);
		return 1;
	} else {
		usputs(mp->s,Enter);
	}
	while(1) {
		if(recvline(mp->s,p,sizeof(buf)) == -1){
			return 1;
		}
		rip(p);
		/* check for end of message ie a . or escaped .. */
		if (*p == '.') {
			if (*++p == '\0') {
				/* Also sends appropriate response */
				if (mailit(mp->data,mp->from,mp->to) != 0)
					usputs(mp->s,Ioerr);
				else
					usputs(mp->s,Sent);
				fclose(mp->data);
				mp->data = NULLFILE;
				del_list(mp->to);
				mp->to = NULLLIST;
				return 0;
			} else if (!(*p == '.' && *(p+1) == '\0'))
				p--;
		}
		/* for UNIX mail compatiblity */
		if (strncmp(p,"From ",5) == 0)
			putc('>',mp->data);
		/* Append to data file */
		if(fprintf(mp->data,"%s\n",p) < 0) {
			usputs(mp->s,Ioerr);
			return 1;
		}
	}
}

/* Create control block, initialize */
static struct smtpsv * near
mail_create()
{
	struct smtpsv *mp = (struct smtpsv *)mxallocw(sizeof(struct smtpsv));
	mp->from = strxdup("");	/* Default to null From address */
	return mp;
}

/* Free resources, delete control block */
static void near
mail_clean(struct smtpsv *mp)
{
	if (mp == NULLSMTPSV)
		return;
	xfree(mp->system);
	xfree(mp->from);
	if(mp->data != NULLFILE)
		fclose(mp->data);
	del_list(mp->to);
	xfree((char *)mp);
}


/* Given a string of the form <user@host>, extract the part inside the
 * brackets and return a pointer to it.
 */
char *
getname(char *cp)
{
	char *cp1;

	if ((cp = strchr(cp,'<')) == NULLCHAR)
		return NULLCHAR;
	cp++;	/* cp -> first char of name */
	if ((cp1 = strchr(cp,'>')) == NULLCHAR)
		return NULLCHAR;
	*cp1 = '\0';
	return cp;
}

/* General mailit function. It takes a list of addresses which have already
** been verified and expanded for aliases. Base on the current mode the message
** is place in an mbox, the outbound smtp queue or the rqueue interface
*/
static int near
mailit(data,from,tolist)
FILE *data;
char *from;
struct list *tolist;
{
	struct list *ap, *dlist = NULLLIST;
	register FILE *fp;
	char mailbox[LINELEN], *cp, *host, *qhost;
	int	c, fail = 0;
	extern int16 Smtpquiet;

	if ((Smtpmode & QUEUE) != 0)
		return(router_queue(data,from,tolist));

	do {
		qhost = NULLCHAR;
		for(ap = tolist;ap != NULLLIST;ap = ap->next) {
			if (ap->type == DOMAIN) {
				if ((host = strrchr(ap->val,'@')) != NULLCHAR)
					host++;
				else
					host = Hostname;
				if(qhost == NULLCHAR)
					qhost = host;
				if(stricmp(qhost,host) == 0) {
					ap->type = BADADDR;
					addlist(&dlist,ap->val,0);
				}
			}
		}
		if(qhost != NULLCHAR) {
			rewind(data);
			queuejob(data,qhost,dlist,from);
			del_list(dlist);
			dlist = NULLLIST;
		}
	} while(qhost != NULLCHAR);

   /*-------------------------------------------------------------------*
   * dk5dc, process nntp entries !xxxx....                              *
   *--------------------------------------------------------------------*/
#ifdef NNTP
  for(ap = tolist;ap != NULLLIST;ap = ap->next) {
      if (ap->type != NNTP_GATE)
         continue;

      /*----------------------------------------------------------------*
      * start processing                                                *
      *-----------------------------------------------------------------*/
	  nnGpost(data,from,ap);
      ap->type = BADADDR;
   }

 #endif

	for(ap = tolist;ap != NULLLIST;ap = ap->next) {
		if(ap->type != LOCAL) {
			ap->type = DOMAIN;
			continue;
		}
		rewind(data);
		/* strip off host name of LOCAL addresses */
		if ((cp = strchr(ap->val,'@')) != NULLCHAR)
			*cp = '\0';

		/* truncate long user names */
		if (strlen(ap->val) > MBOXLEN)
			ap->val[MBOXLEN] = '\0';

		/* if mail file is busy save it in our smtp queue
		 * and let the smtp daemon try later.
		 */
		if (mlock(Mailspool,ap->val)) {
			addlist(&dlist,ap->val,0);
			fail = queuejob(data,Hostname,dlist,from);
			del_list(dlist);
			dlist = NULLLIST;
		} else {
			char buf[LINELEN];
			int tocnt = 0;
			sprintf(mailbox,"%s/%s.txt",Mailspool,ap->val);
#ifndef	AMIGA
			if((fp = fopen(mailbox,APPEND_TEXT)) != NULLFILE) {
#else
			if((fp = fopen(mailbox,"r+")) != NULLFILE) {
				fseek(fp, 0L, 2);
#endif
				fprintf(fp,"From %s %s",from,ptime(&currtime));
				host = NULLCHAR;
				while(fgets(buf,LINELEN,data) != NULLCHAR){
					if(buf[0] == '\n'){
						if(tocnt == 0)
							fprintf(fp,"%s%s\n",Hdrs[APPARTO],ap->val);
						fputc('\n',fp);
						break;
					}
					fputs(buf,fp);
					rip(buf);
					switch(htype(buf)){
					case TO:
					case CC:
						++tocnt;
						break;
					case RRECEIPT:
						if((cp = getaddress(buf,0)) != NULLCHAR){
							xfree(host);
							host = strxdup(cp);
						}
						break;
					}
				}
				while((c = fread(buf,1,LINELEN,data)) > 0)
					if(fwrite(buf,1,c,fp) != c)
						break;
				if(ferror(fp))
					fail = 1;
				else
					fputc('\n',fp);
				/* Leave a blank line between msgs */
				fclose(fp);
/*
				addmprompt(ap->val);
*/
				if(Smtpquiet < 2)
					/* timestamp by dc3sn */
					tprintf("New mail for %s from <%s> at %s%s",
						ap->val,from,ctime(&currtime),
						(Smtpquiet < 1)  ? "\007" : "");
				if(Smtpquiet == 3)
					log(-1,"SMTP new mail <%s>",from);
				if(host != NULLCHAR){
					rewind(data); /* Send return receipt */
					mdaemon(data,host,NULLLIST,0);
					xfree(host);
				}
			} else {
				fail = 1;
			}
			rmlock(Mailspool,ap->val);
			if (fail)
				break;
			smtplog("deliver: To: %s From: %s",ap->val,from);
		}
	}
	return fail;
}

/* Return Date/Time in Arpanet format in passed string */
char *
ptime(t)
int32 *t;
{
	/* Print out the time and date field as
	 *		"DAY day MONTH year hh:mm:ss ZONE"
	 */
	struct tm *ltm;
	static char str[40];
	char tz[5];

	/* Read the system time */
	ltm = localtime(t);

		sprintf(tz," %s",daylight ? tzname[1] : tzname[0]);

	/* rfc 822 format */
	sprintf(str,"%s, %.2d %s %02d %02d:%02d:%02d%.4s\n",
		Days[ltm->tm_wday],
		ltm->tm_mday,
		Months[ltm->tm_mon],
		ltm->tm_year,
		ltm->tm_hour,
		ltm->tm_min,
		ltm->tm_sec,
		tz);
	return(str);
}

long
get_msgid()
{
	char sfilename[LINELEN], s[20];
	long sequence = 0;
	FILE *sfile;

	sprintf(sfilename,"%s/sequence.seq",Mailqdir);
	/* if sequence file exists, get the value, otherwise set it */
	if((sfile = fopen(sfilename,READ_TEXT)) != NULLFILE) {
		fgets(s,sizeof(s),sfile);
		sequence = atol(s);
		/* Keep it in range of and 8 digit number to use for dos name prefix. */
		if (sequence < 0L || sequence > 99999999L )
			sequence = 0;
		fclose(sfile);
	}
	/* increment sequence number, and write to sequence file */
	if((sfile = open_file(sfilename,WRITE_TEXT,0,1)) != NULLFILE) {
		fprintf(sfile,"%ld",++sequence);
		fclose(sfile);
	}
	return (sequence == 0) ? 1 : sequence;
}

#ifdef	MSDOS
/* Illegal characters in a DOS filename */
static char baddoschars[] = "\"[]:|<>+=;,";
#endif

/* test if mail address is valid */
int
validate_address(s)
char *s;
{
	char *cp;
	int32 addr;

   /*if first character in address is a bang (!) address is NNTP_GATE,dk5dc*/
   if(*s == '!') {
 #ifdef NNTP
		return NNTP_GATE;
 #else 
		return BADADDR;
 #endif
	}
	/* if address has @ in it the check dest address */
	if ((cp = strrchr(s,'@')) != NULLCHAR) {
		cp++;
		/* 1st check if its our hostname
		* if not then check the hosts file and see
		* if we can resolve ther address to a know site
		* or one of our aliases
		*/
		if (strcmp(cp,Hostname) != 0) {
			if ((addr = mailroute(cp)) == 0 && (Smtpmode & QUEUE) == 0)
				return BADADDR;
			if (ismyaddr(addr) == NULLIF)
				return DOMAIN;
		}

		/* on a local address remove the host name part */
		*--cp = '\0';
	}

	/* if using an external router leave address alone */
	if ((Smtpmode & QUEUE) != 0)
		return LOCAL;

	/* check for the user%host hack */
	if ((cp = strrchr(s,'%')) != NULLCHAR) {
		*cp = '@';
		cp++;
		/* reroute based on host name following the % seperator */
		if (mailroute(cp) == 0)
			return BADADDR;
		else
			return DOMAIN;
	}

#ifdef MSDOS	/* dos file name checks */
	/* Check for characters illegal in MS-DOS file names */
	for(cp = baddoschars;*cp != '\0';cp++){
		if(strchr(s,*cp) != NULLCHAR)
			return BADADDR;
	}
#endif
	return LOCAL;
}

/* place a mail job in the outbound queue */
int
queuejob(dfile,host,to,from)
FILE *dfile;
char *host;
struct list *to;
char *from;
{
	FILE *fp;
	struct list *ap;
	char tmpstring[50], prefix[9], buf[LINELEN];
	int cnt;

	sprintf(prefix,"%ld",get_msgid());
	mlock(Mailqdir,prefix);
	sprintf(tmpstring,"%s/%s.txt",Mailqdir,prefix);
	if((fp = open_file(tmpstring,WRITE_TEXT,0,1)) == NULLFILE) {
		rmlock(Mailqdir,prefix);
		return 1;
	}
	while((cnt = fread(buf, 1, LINELEN, dfile)) > 0)
		if(fwrite(buf, 1, cnt, fp) != cnt)
			break;
	if(ferror(fp)){
		fclose(fp);
		rmlock(Mailqdir,prefix);
		return 1;
	}
	fclose(fp);
	sprintf(tmpstring,"%s/%s.wrk",Mailqdir,prefix);
	if((fp = open_file(tmpstring,WRITE_TEXT,0,1)) == NULLFILE) {
		rmlock(Mailqdir,prefix);
		return 1;
	}
	fprintf(fp,"%s\n%s\n",host,from);
	for(ap = to; ap != NULLLIST; ap = ap->next) {
		fprintf(fp,"%s\n",ap->val);
		smtplog("queue job %s To: %s From: %s",prefix,ap->val,from);
	}
	fclose(fp);
	rmlock(Mailqdir,prefix);
	return 0;
}

/* Deliver mail to the appropriate mail boxes */
static int near
router_queue(data,from,to)
FILE *data;
char *from;
struct list *to;
{
	int c;
	struct list *ap;
	FILE *fp;
	char tmpstring[50], prefix[9];

	sprintf(prefix,"%ld",get_msgid());
	mlock(Routeqdir,prefix);
	sprintf(tmpstring,"%s/%s.txt",Routeqdir,prefix);
	if((fp = open_file(tmpstring,WRITE_TEXT,0,1)) == NULLFILE) {
		rmlock(Routeqdir,prefix);
		return 1;
	}
	rewind(data);
	while((c = getc(data)) != EOF)
		if(putc(c,fp) == EOF)
			break;
	if(ferror(fp)){
		fclose(fp);
		rmlock(Routeqdir,prefix);
		return 1;
	}
	fclose(fp);
	sprintf(tmpstring,"%s/%s.wrk",Routeqdir,prefix);
	if((fp = open_file(tmpstring,WRITE_TEXT,0,1)) == NULLFILE) {
		rmlock(Routeqdir,prefix);
		return 1;
	}
	fprintf(fp,"From: %s\n",from);
	for(ap = to;ap != NULLLIST;ap = ap->next) {
		fprintf(fp,"To: %s\n",ap->val);
	}
	fclose(fp);
	rmlock(Routeqdir,prefix);
	smtplog("rqueue job %s From: %s",prefix,from);
	return 0;
}

/* add an element to the front of the list pointed to by head
** return NULLLIST if out of memory.
*/
struct list *
addlist(head,val,type)
struct list **head;
char *val;
int type;
{
	struct list *tp = (struct list *)mxallocw(sizeof(struct list));

	tp->next = NULLLIST;

	/* allocate storage for the char string */
	tp->val = strxdup(val);
	tp->type = type;

	/* add entry to front of existing list */
	if (*head == NULLLIST)
		*head = tp;
	else {
		tp->next = *head;
		*head = tp;
	}
	return tp;

}

#define SKIPWORD(X) while(*X && *X!=' ' && *X!='\t' && *X!='\n' && *X!= ',') X++;
#define SKIPSPACE(X) while(*X ==' ' || *X =='\t' || *X =='\n' || *X == ',') X++;

/* check for and alias and expand alias into a address list */
static struct list * near
expandalias(head, user)
struct list **head;
char *user;
{
	FILE *fp;
	int len, inalias = 0;
	struct list *tp = NULLLIST;
	char *s, *p, buf[LINELEN];

	/* no alias file found */
	if ((fp = fopen(Alias,READ_TEXT)) == NULLFILE) {
		/* Try to resolve M* domain name records */
		if((s = resolve_mailb(user)) != NULLCHAR) {
			/* remove the trailing dot */
			len = strlen(s) - 1;
			if(s[len] == '.')
				s[len] = '\0';
			/* replace first dot with @ if there is no @ */
			if(strchr(s,'@') == NULLCHAR && (p = strchr(s,'.')) != NULLCHAR)
				*p = '@';
			tp = addlist(head,s,(strchr(s,'@') != NULLCHAR) ? DOMAIN : LOCAL);
			++inalias;
		}
		return (inalias ? tp : addlist(head,user,LOCAL));
	}

	while (fgets(buf,LINELEN,fp) != NULLCHAR) {
		p = buf;
		if ( *p == '#' || *p == '\0')
			continue;
		rip(p);

		/* if not in an matching entry skip continuation lines */
		if (!inalias && isspace(*p))
			continue;

		/* when processing an active alias check for a continuation */
		if (inalias) {
			if (!isspace(*p))
				break;	/* done */
		} else {
			s = p;
			SKIPWORD(p);
			*p++ = '\0';	/* end the alias name */
			if (strcmp(s,user) != 0)
				continue;	/* no match go on */
			inalias = 1;
		}

		/* process the recipients on the alias line */
		SKIPSPACE(p);
		while(*p != '\0' && *p != '#') {
			s = p;
			SKIPWORD(p);
			if (*p != '\0')
				*p++ = '\0';

			/* find hostname */
 #ifdef NNTP
			if(*s == '!')
				tp = addlist(head,s,NNTP_GATE);
			else
 #endif
			  if (strchr(s,'@') != NULLCHAR)
				tp = addlist(head,s,DOMAIN);
			else
				tp = addlist(head,s,LOCAL);
			SKIPSPACE(p);
		}
	}
	fclose(fp);

	if (inalias) {	/* found and processed and alias. */
		return tp;
	}
	/* no alias found treat as a local address */
	return addlist(head, user, LOCAL);
}

#if	defined(ANSIPROTO)
static void near
smtplog(char *fmt, ...)
{
	va_list ap;
	char *cp;
	long t;
	FILE *fp;

	if ((fp = open_file(Maillog,APPEND_TEXT,0,1)) == NULLFILE)
		return;
	time(&t);
	cp = ctime(&t);
	rip(cp);
	fprintf(fp,"%s ",cp);
	va_start(ap,fmt);
	vfprintf(fp,fmt,ap);
	va_end(ap);
	fputc('\n',fp);
	fclose(fp);
}

#else

static void near
smtplog(fmt,arg1,arg2,arg3,arg4)
char *fmt;
int arg1,arg2,arg3,arg4;
{
	char *cp;
	long t;
	FILE *fp;

	if ((fp = open_file(Maillog,APPEND_TEXT,0,1)) == NULLFILE)
		return;
	time(&t);
	cp = ctime(&t);
	rip(cp);
	fprintf(fp,"%s ",cp);
	fprintf(fp,fmt,arg1,arg2,arg3,arg4);
	fputc('\n',fp);
	fclose(fp);
}
#endif

/* send mail to a single user. Can be called from the ax24 mailbox or
** from the return mail function in the smtp client
*/
static int near
mailuser(data,from,to)
FILE *data;
char *from;
char *to;
{
		int address_type, ret;
		struct list *tolist = NULLLIST;

		/* check if address is ok */
		if ((address_type = validate_address(to)) == BADADDR) {
			return 1;
		}
		/* if a local address check for an alias */
		if (address_type == LOCAL)
			expandalias(&tolist, to);
		else
			/* a remote address is added to the list */
			addlist(&tolist, to, address_type);

		ret = mailit(data,from,tolist);
		del_list(tolist);
		return ret;

}

/* Mailer daemon return mail mechanism */
int
mdaemon(data,to,lp,bounce)
FILE *data;		/* pointer to rewound data file */
char *to;		/* Overridden by Errors-To: line if bounce is true */
struct list *lp;	/* error log for failed mail */
int bounce;		/* True for failed mail, otherwise return receipt */
{
	FILE *tfile;
	char buf[LINELEN], *cp, *newto = NULLCHAR;
	int cnt;

	if(to == NULLCHAR || (to != NULLCHAR && *to == '\0') || bounce){
		while(fgets(buf,sizeof(buf),data) != NULLCHAR) {
			if(*buf == '\n')
				break;
			/* Look for Errors-To: */
			if(htype(buf) == ERRORSTO &&
			   (cp = getaddress(buf,0)) != NULLCHAR){
				xfree(newto);
				newto = strxdup(cp);
				break;
			}
		}
		if(newto == NULLCHAR && ((to != NULLCHAR && *to == '\0') ||
		   to == NULLCHAR))
			return -1;
		rewind(data);
	}
	if((tfile = temp_file(0,1)) == NULLFILE)
		return -1;
	fprintf(tfile,"%s%s",Hdrs[DATE],ptime(&currtime));
	fprintf(tfile,"%s<%ld@%s>\n",Hdrs[MSGID],get_msgid(),Hostname);
	fprintf(tfile,"%sMAILER-DAEMON@%s (Mail Delivery Subsystem)\n",
		Hdrs[FROM],Hostname);
	fprintf(tfile,"%s%s\n",Hdrs[TO],newto != NULLCHAR ? newto : to);
	fprintf(tfile,"%s%s\n\n",Hdrs[SUBJECT],
		bounce ? "Failed mail" : "Return receipt");
	if(bounce) {
		fputs("  ==== transcript follows ====\n\n",tfile);
		for (; lp != NULLLIST; lp = lp->next)
			fprintf(tfile,"%s\n",lp->val);
		fputs("\n  ==== Unsent message follows ====\n",tfile);
	} else {
		fputs("  ==== Message header follows ====\n",tfile);
	}
	while(fgets(buf,sizeof(buf),data) != NULLCHAR){
		if(*buf == '\n')
			break;
		fputs(buf,tfile);
	}
	if(bounce){
		fputc('\n',tfile);
		while((cnt = fread(buf,1,sizeof(buf),data)) > 0)
			fwrite(buf,1,cnt,tfile);
	}
	fseek(tfile,0L,0);
	/* A null From<> so no looping replys to MAIL-DAEMONS */
	mailuser(tfile,"",newto != NULLCHAR ? newto : to);
	fclose(tfile);
	xfree(newto);
	return 0;
}
