/*
 *-----------------------------------------------------------------------
 *
 * TFLINK - program to make TFPCX TSR running a Baycom modem on a
 * COM port appear as a serial (HOST mode) TNC on a second COM port.
 *
 * tfpcx.c - routines to handle TFPCX low-level i/o
 *
 *-----------------------------------------------------------------------
 */

/* standard inclusion files */
#include "tflink.h"

/* local definitions */
/* define TFPCX constants and defaults */
#define TFPCX_IDENT		"N5NX"	/* identification string */
#define TFPCX_IDENT_OFFSET	3	/* find ident 3 bytes after INT */
#define TFPCX_INT_REQUEST	0x01	/* character ready? function */
#define TFPCX_INT_READ		0x02	/* character read function */
#define TFPCX_INT_WRITE		0x03	/* character write function */
#define TFPCX_INT_VERSION	0xFE	/* get version function */
#define TFPCX_ERROR		0xFFFF	/* if bad int subfunction used */


/*-----------------------------------------------------------------------*/
/*
 * verify that tfpcx is installed
 */
bool tfpcx_check()
{
	char far*	tfpcx_ident;	/* TFPCX ident string address */
	int		i;		/* loop counter/index */

	/* set up a pointer to the TFPCX ident string */
	tfpcx_ident = (char far*)getvect( tfpcx_interrupt )
			+ TFPCX_IDENT_OFFSET;

	/* check for the presence of the ident string; for some reason,
	   strncmp seems unhappy with char far *tfpcx_ident, so do
	   the comparison the hard way */
	for ( i = 0; i < strlen( TFPCX_IDENT ); i++ ) {
		/* if character mismatch then return false */
		if ( *(tfpcx_ident + i) != TFPCX_IDENT[ i ] )
			return ( FALSE );
	}
	return ( TRUE );
}


/*-----------------------------------------------------------------------*/
/*
 * tfpcx ready routine, to detect if a character is waiting
 */
bool tfpcx_request()
{
	union	REGS inr,outr;	/* registers */

	/* ensure TFPCX is still available */
	if (! tfpcx_check() )
		return ( FALSE );

	/* see if TFPCX has a character ready to read */
	inr.h.ah = TFPCX_INT_REQUEST;
	int86( tfpcx_interrupt, &inr, &outr );

	/* test return value of AL */
	if ( outr.h.al )
		return ( TRUE );
	else
		return ( FALSE );
}


/*-----------------------------------------------------------------------*/
/*
 * tfpcx read character routine for low-level i/o
 */
u_char tfpcx_readch()
{
	union	REGS inr,outr;	/* registers */

	/* ensure TFPCX is still available */
	if (! tfpcx_check() )
		return ( (u_char)0 );

	/* read the tfpcx character awaiting read */
	inr.h.ah = TFPCX_INT_READ;
	int86( tfpcx_interrupt, &inr, &outr );

	/* return character read */
	return ( (u_char)outr.h.al );
}


/*-----------------------------------------------------------------------*/
/*
 * tfpcx write character routine for low-level i/o.
 */
void tfpcx_writech( u_char c )
{
	union	REGS inr,outr;	/* registers */

	/* ensure TFPCX is still available */
	if (! tfpcx_check() )
		return;

	/* set tfpcx function code/data and call interrupt */
	inr.h.ah = TFPCX_INT_WRITE;
	inr.h.al = c;
	int86( tfpcx_interrupt, &inr, &outr );
}


/*-----------------------------------------------------------------------*/
/*
 * read the tfpcx version numbers, and return as one integer
 */
u_int tfpcx_version()
{
	union	REGS inr,outr;	/* registers */

	/* ensure TFPCX is still available */
	if (! tfpcx_check() )
		return ( (u_int)0 );

	/* set tfpcx function code and call interrupt */
	inr.h.ah = TFPCX_INT_VERSION;
	int86( tfpcx_interrupt, &inr, &outr );

	/* return the version value */
	return ( (u_int)((outr.h.ah << 8) | outr.h.al) );
}
