/*
 *-----------------------------------------------------------------------
 *
 * TFLINK - program to make TFPCX TSR running a Baycom modem on a
 * COM port appear as a serial (HOST mode) TNC on a second COM port.
 *
 * getopts.c - handle command line options
 *
 *-----------------------------------------------------------------------
 */

/* standard include file */
#include "tflink.h"

/* local definitions */
/* usage and help messages */
#define USAGE	"\
Usage: TFLINK -P COMn [ -N ] [ -B baud ] [ -I interrupt ]\n\
       TFLINK -P COMn [ -N ] -U\n\n\
       TFLINK -H for help\n"
#define HELPMSG	"\
\n\
Usage: TFLINK -P COMn [ -N ] [ -B baud ] [ -I interrupt ]\n\
       TFLINK -P COMn [ -N ] -U\n\
\n\
  -P COM<n>       Link TFPCX to port COM<n>.  COM1 to COM4 allowed.\n\
                  No default value.\n\
\n\
  -N              No startup or unload messages\n\
\n\
  -B <baud>       Set serial baud rate: 110, 300, 600, 1200, 2400,\n\
                  4800, 9600, or 19200 are valid for <baud>.  Default\n\
                  value is 9600 baud\n\
\n\
  -I <interrupt>  Indicate interrupt used by TFPCX.  Default is 0xFD\n\
\n\
  -U              Unload currently running TFLINK\n\
\n"

/* baud rates table */
#define	NUM_BAUD_RATES	8
char	baud_table[NUM_BAUD_RATES][6] = {
	"110", "300", "600", "1200", "2400", "4800", "9600", "19200"
};

/* defaults for the various selectable items in this module */
#define TFPCX_DEFAULT_INT	0xFD	/* default interrupt vector */
#define MIN_SIO_PORT		1	/* COM1 lowest port */
#define MAX_SIO_PORT		4	/* COM4 highest port */
#define SIO_DEFAULT_BAUD	9600	/* settle on 9600 baud initially */


/*-----------------------------------------------------------------------*/
/*
 * Get command line options and store away the information
 * held in them.  Return FALSE if command line error
 */
bool getopts( int argc, char **argv )
{
	int	argp;		/* argument array pointer */
	int	cindex;		/* char index for string traversal */
	char	*optarg;	/* pointer to option argument */
	int	baud_index;	/* baud rate search index */
	bool	retcode;	/* return code for the routine */
	bool	pflag, bflag, iflag;	/* local flags */

	/* initialise defaults on all globals we control */
	pflag = bflag = iflag = FALSE;
	nflag = rflag = uflag = FALSE;
	tfpcx_interrupt	= TFPCX_DEFAULT_INT;
	sio_baud	= SIO_DEFAULT_BAUD;

	/* start assuming everything is OK */
	retcode = TRUE;

	/* loop over all arguments except the command name in [0] */
	for ( argp = 1; argp < argc; argp++ ) {
		if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'H'){
			printf( HELPMSG );
			exit( 0 );
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'N'){
			nflag = TRUE;
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'R'){
			rflag = TRUE;
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'U'){
			uflag = TRUE;
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'B'){
			bflag = TRUE;
			optarg = ( strlen( argv[argp] ) == 2 ) ?
					argv[++argp] : argv[argp] + 2;
			if ( argp == argc ) { retcode = FALSE; break; }
			for ( baud_index = 0; baud_index < NUM_BAUD_RATES;
						baud_index++ ) {
				if ( strcmp(optarg, baud_table[baud_index]) == 0 ) {
					sscanf( optarg, "%d", &sio_baud );
					break;
				}
			}
			if ( baud_index >= NUM_BAUD_RATES ) {
				fprintf( stderr,
					"Unrecognised baud rate %s\n", optarg );
				retcode = FALSE; break;
			}
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'I'){
			iflag = TRUE;
			optarg = ( strlen( argv[argp] ) == 2 ) ?
					argv[++argp] : argv[argp] + 2;
			if ( argp == argc ) { retcode = FALSE; break; }
			for ( cindex=0; cindex<strlen( optarg ); cindex++ ) {
				optarg[cindex] = toupper( optarg[cindex] );
			}
			if ( *optarg == '0' && *(optarg+1) == 'X' )
				optarg += 2;
			if ( sscanf( optarg, "%02x", &tfpcx_interrupt ) != 1 ||
					strlen( optarg ) != 2 ) {
				fprintf( stderr,
					"Bad interrupt value %s\n", optarg );
				retcode = FALSE; break;
			}
		}
		else if (argv[argp][0] == '-' && toupper(argv[argp][1]) == 'P'){
			pflag = TRUE;
			optarg = ( strlen( argv[argp] ) == 2 ) ?
					argv[++argp] : argv[argp] + 2;
			if ( argp == argc ) { retcode = FALSE; break; }
			for ( cindex=0; cindex<strlen( optarg ); cindex++ ) {
				optarg[cindex] = toupper( optarg[cindex] );
			}
			if ( sscanf( optarg, "COM%1d", &sio_port ) != 1 ||
					strlen( optarg ) != strlen( "COMx" ) ||
					sio_port < MIN_SIO_PORT ||
					sio_port > MAX_SIO_PORT ) {
				fprintf( stderr,
					"Unrecognised port %s\n", optarg );
				retcode = FALSE; break;
			}
		}
		else {
			/* some unrecognised option */
			retcode = FALSE; break;
		}
	}

	/* check for odd flag combinations and mandatory -P */
	if ( ! pflag )
		retcode = FALSE;
	if ( uflag && ( bflag || iflag ))
		retcode = FALSE;

	/* print usage if needed and return */
	if (! retcode) {
		fprintf( stderr, USAGE );
	}
	return ( retcode );
}
